#include <AMReX_FArrayBox.H>
#include <AMReX_IArrayBox.H>
#include <AMReX_Geometry.H>
#include <LagrangianPolynomicInterpolater.H>
#include <AMReX_Interp_C.H>
#include <AMReX_MFInterp_C.H>

#include <climits>

#define POLINT_MACRO_LINEAR(y1, y2, R, p) ( (R*y1-p*y1+p*y2)/R )
#define POLINT_MACRO_CUBIC(y1, y2, y3, y4, R, p) ( (1.0/6)*(6*R*R*R*y2-2*R*R*p*y1-3*R*R*p*y2+6*R*R*p*y3-R*R*p*y4+3*R*p*p*y1-6*R*p*p*y2+3*R*p*p*y3-p*p*p*y1+3*p*p*p*y2-3*p*p*p*y3+p*p*p*y4)/(R*R*R) )
#define POLINT_MACRO_QUINTIC(y1, y2, y3, y4, y5, y6, R, p) ( (1.0/120)*(120*R*R*R*R*R*y3+6*R*R*R*R*p*y1-60*R*R*R*R*p*y2-40*R*R*R*R*p*y3+120*R*R*R*R*p*y4-30*R*R*R*R*p*y5+4*R*R*R*R*p*y6-5*R*R*R*p*p*y1+80*R*R*R*p*p*y2-150*R*R*R*p*p*y3+80*R*R*R*p*p*y4-5*R*R*R*p*p*y5-5*R*R*p*p*p*y1-5*R*R*p*p*p*y2+50*R*R*p*p*p*y3-70*R*R*p*p*p*y4+35*R*R*p*p*p*y5-5*R*R*p*p*p*y6+5*R*p*p*p*p*y1-20*R*p*p*p*p*y2+30*R*p*p*p*p*y3-20*R*p*p*p*p*y4+5*R*p*p*p*p*y5-p*p*p*p*p*y1+5*p*p*p*p*p*y2-10*p*p*p*p*p*y3+10*p*p*p*p*p*y4-5*p*p*p*p*p*y5+p*p*p*p*p*y6)/(R*R*R*R*R) )

#define POLINT_MACRO_LINEAR_R2(y1, y2) ( 0.5 * y1 + 0.5 * y2 )
#define POLINT_MACRO_CUBIC_R2(y1, y2, y3, y4) (0.5625*(y2 + y3)-0.0625*(y1 + y4))
#define POLINT_MACRO_QUINTIC_R2(y1, y2, y3, y4, y5, y6) (0.5859375 * (y3 + y4) - 0.09765625 * (y2 + y5) + 0.01171875 * (y1+y6))

AMREX_GPU_HOST_DEVICE
AMREX_FORCE_INLINE
double POLINT_MACRO_QUINTIC_LINEAR_R2(double y1, double y2, double y3, double y4, double y5, double y6) {
   double yorig, ymin, ymax;
   yorig = POLINT_MACRO_QUINTIC_R2(y1, y2, y3, y4, y5, y6);
   ymin = std::min(y3, y4);
   ymax = std::max(y3, y4);
   if ( yorig > ymin && yorig < ymax) {
      return yorig;
   } else {
      return POLINT_MACRO_LINEAR_R2(y3, y4);
   }
}

AMREX_GPU_HOST_DEVICE
AMREX_FORCE_INLINE
double POLINT_MACRO_QUINTIC_LINEAR(double y1, double y2, double y3, double y4, double y5, double y6, int R, int p) {
   double yorig, ymin, ymax;
   yorig = POLINT_MACRO_QUINTIC(y1, y2, y3, y4, y5, y6, R, p);
   ymin = std::min(y3, y4);
   ymax = std::max(y3, y4);
   if ( yorig > ymin && yorig < ymax) {
      return yorig;
   } else {
      return POLINT_MACRO_LINEAR(y3, y4, R, p);
   }
}

using namespace amrex;

LagrangianPolynomic::LagrangianPolynomic (const bool refine_boundary, const std::string operator_name)
{
    int order;
    if (operator_name == "LINEAR_REFINE") {
        order = 1;
    }
    if (operator_name == "CUBIC_REFINE") {
        order = 3;
    }
    if (operator_name == "QUINTIC_REFINE" || operator_name == "QUINTIC_LINEAR_REFINE") {
        order = 5;
    }

    BL_ASSERT(order == 1 || order == 3 || order == 5);
    d_stencil = order + 1;
    d_half_stencil = (order + 1)/2;
    d_refine_boundary = refine_boundary;
    d_operator_name = operator_name;
}

LagrangianPolynomic::~LagrangianPolynomic () {}

Box
LagrangianPolynomic::CoarseBox (const Box& fine,
                         int        ratio)
{
    Box b = amrex::coarsen(fine,ratio);

    for (int i = 0; i < AMREX_SPACEDIM; i++)
    {
        if (b.length(i) < 2)
        {
            //
            // Don't want degenerate boxes.
            //
            b.growHi(i,1);
        }
    }
    if (d_half_stencil == 2) {
        b.grow(1);
    }
    if (d_half_stencil == 3) {
        b.grow(2);
    }

    return b;
}

Box
LagrangianPolynomic::CoarseBox (const Box&     fine,
                         const IntVect& ratio)
{
    Box b = amrex::coarsen(fine,ratio);

    for (int i = 0; i < AMREX_SPACEDIM; i++)
    {
        if (b.length(i) < 2)
        {
            //
            // Don't want degenerate boxes.
            //
            b.growHi(i,1);
        }
    }

    if (d_half_stencil == 2) {
        b.grow(1);
    }
    if (d_half_stencil == 3) {
        b.grow(2);
    }

    return b;
}

void
LagrangianPolynomic::interp (const FArrayBox&  crse,
                      int               crse_comp,
                      FArrayBox&        fine,
                      int               fine_comp,
                      int               ncomp,
                      const Box&        fine_region,
                      const IntVect&    ratio,
                      const Geometry&   crse_geom ,
                      const Geometry&   fine_geom ,
                      Vector<BCRec> const& bcr,
                      int               actual_comp,
                      int               actual_state,
                      RunOn             runon)
{
    BL_PROFILE("LagrangianPolynomic::interp()");

    if (d_operator_name == "LINEAR_REFINE") {
        interp_linear (crse, crse_comp, fine, fine_comp, ncomp, fine_region, ratio, crse_geom, fine_geom, bcr, actual_comp, actual_state, runon);
    }
    if (d_operator_name == "CUBIC_REFINE") {
        interp_cubic (crse, crse_comp, fine, fine_comp, ncomp, fine_region, ratio, crse_geom, fine_geom, bcr, actual_comp, actual_state, runon);
    }
    if (d_operator_name == "QUINTIC_REFINE") {
        interp_quintic (crse, crse_comp, fine, fine_comp, ncomp, fine_region, ratio, crse_geom, fine_geom, bcr, actual_comp, actual_state, runon);
    }
    if (d_operator_name == "QUINTIC_LINEAR_REFINE") {
        interp_quintic_linear (crse, crse_comp, fine, fine_comp, ncomp, fine_region, ratio, crse_geom, fine_geom, bcr, actual_comp, actual_state, runon);
    }
}

void
LagrangianPolynomic::interp_linear (const FArrayBox&  crseFAB,
                      int               ccomp,
                      FArrayBox&        fineFAB,
                      int               fcomp,
                      int               ncomp,
                      const Box&        fine_region,
                      const IntVect&    ratio,
                      const Geometry&   crse_geom ,
                      const Geometry&   fine_geom ,
                      Vector<BCRec> const& bcr,
                      int               actual_comp,
                      int               actual_state,
                      RunOn             runon)
{
    Array4<Real const> const& crse = crseFAB.const_array();
    Array4<Real> const& fine = fineFAB.array();
    AMREX_HOST_DEVICE_PARALLEL_FOR_4D_FLAG(runon,fine_region,ncomp,i,j,k,n,
    {
#if AMREX_SPACEDIM == 2
        double ymtmp[2];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        if (px != 0 && py != 0) {
            // Node on a X-Y face
            if (rx > 2) {
               ymtmp[0] = POLINT_MACRO_LINEAR(crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), rx, px);
               ymtmp[1] = POLINT_MACRO_LINEAR(crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), rx, px);
            }
            else {
               ymtmp[0] = POLINT_MACRO_LINEAR_R2(crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp));
               ymtmp[1] = POLINT_MACRO_LINEAR_R2(crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp));
            }                        
            if (ry > 2) {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR(ymtmp[0], ymtmp[1], ry, py);
            }
            else {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR_R2(ymtmp[0], ymtmp[1]);
            }
        } else if (px != 0) {
            // Node on X line
            if (rx > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR(crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), rx, px);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR_R2(crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp));
            }
        } else if (py != 0) {
            // Node on Y line
            if (ry > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR(crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), ry, py);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_LINEAR_R2(crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,0,n+fcomp) = crse(ic,jc,0,n+ccomp);
        }
#endif
#if AMREX_SPACEDIM == 3
        double ymtmp[2];
        double yntmp[2];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int kc = amrex::coarsen(k,ratio[2]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        int pz = k - kc*ratio[2];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        Real rz = Real(ratio[2]);
        if (px != 0 && py != 0 && pz != 0) {
            for (int l=0;l<2;l++) {
                for (int m=0;m<2;m++) {
                    if (rx > 2) {
                        yntmp[m] = POLINT_MACRO_LINEAR(crse(ic  ,jc+m,kc+l,n+ccomp), crse(ic+1,jc+m,kc+l,n+ccomp), rx, px);
                    }
                    else {
                        yntmp[m] = POLINT_MACRO_LINEAR_R2(crse(ic  ,jc+m,kc+l,n+ccomp), crse(ic+1,jc+m,kc+l,n+ccomp));
                    }
                }
                if (ry > 2) {
                    ymtmp[l] = POLINT_MACRO_LINEAR(yntmp[0],yntmp[1], ry, py);
                }
                else{
                    ymtmp[l] = POLINT_MACRO_LINEAR_R2(yntmp[0],yntmp[1]);
                }
            }
            if (rz > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(ymtmp[0],ymtmp[1], rz, pz);   
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(ymtmp[0],ymtmp[1]);
            }
        } else if (px != 0 && py != 0) {
            for (int l=0;l<2;l++) {
                if (rx > 2) {
                    yntmp[l] = POLINT_MACRO_LINEAR(crse(ic,jc+l,kc,n+ccomp), crse(ic+1,jc+l,kc,n+ccomp), rx, px);
                }
                else {
                    yntmp[l] = POLINT_MACRO_LINEAR_R2(crse(ic,jc+l,kc,n+ccomp), crse(ic+1,jc+l,kc,n+ccomp));
                }
            }
            if (ry > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(yntmp[0],yntmp[1], ry, py);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(yntmp[0],yntmp[1]);
            }
        } else if (px != 0 && pz != 0) {
            for (int l=0;l<2;l++) {
                if (rx > 2) {
                    yntmp[l] = POLINT_MACRO_LINEAR(crse(ic,jc,kc+l,n+ccomp), crse(ic+1,jc,kc+l,n+ccomp), rx, px);
                }
                else {
                    yntmp[l] = POLINT_MACRO_LINEAR_R2(crse(ic,jc,kc+l,n+ccomp), crse(ic+1,jc,kc+l,n+ccomp));
                }
            }
            if (rz > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(yntmp[0],yntmp[1], rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(yntmp[0],yntmp[1]);
            }
        } else if (py != 0 && pz != 0) {
            for (int l=0;l<2;l++) {
                if (ry > 2) {
                    yntmp[l] = POLINT_MACRO_LINEAR(crse(ic,jc,kc+l,n+ccomp), crse(ic,jc+1,kc+l,n+ccomp), ry, py);
                }
                else {
                    yntmp[l] = POLINT_MACRO_LINEAR_R2(crse(ic,jc,kc+l,n+ccomp), crse(ic,jc+1,kc+l,n+ccomp));
                }
            }
            if (rz > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(yntmp[0],yntmp[1], rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(yntmp[0],yntmp[1]);
            }
        } else if (px != 0) {
            if (rx > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), rx, px);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp));
            }
        } else if (py != 0) {
            if (ry > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), ry, py);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp));
            }
        } else if (pz != 0) {
            if (rz > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR(crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_LINEAR_R2(crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,k,n+fcomp) = crse(ic,jc,kc,n+ccomp);
        }
#endif
    });
}

void
LagrangianPolynomic::interp_cubic (const FArrayBox&  crseFAB,
                      int               ccomp,
                      FArrayBox&        fineFAB,
                      int               fcomp,
                      int               ncomp,
                      const Box&        fine_region,
                      const IntVect&    ratio,
                      const Geometry&   crse_geom ,
                      const Geometry&   fine_geom ,
                      Vector<BCRec> const& bcr,
                      int               actual_comp,
                      int               actual_state,
                      RunOn             runon)
{
    Array4<Real const> const& crse = crseFAB.const_array();
    Array4<Real> const& fine = fineFAB.array();
    AMREX_HOST_DEVICE_PARALLEL_FOR_4D_FLAG(runon,fine_region,ncomp,i,j,k,n,
    {
#if AMREX_SPACEDIM == 2
        double ymtmp[4];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        
        if (px != 0 && py != 0) {
            // Node on a X-Y face
            if (rx > 2) {
               ymtmp[0] = POLINT_MACRO_CUBIC(crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp), rx, px);
               ymtmp[1] = POLINT_MACRO_CUBIC(crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp), rx, px);
               ymtmp[2] = POLINT_MACRO_CUBIC(crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp), rx, px);
               ymtmp[3] = POLINT_MACRO_CUBIC(crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp), rx, px);
            }
            else {
               ymtmp[0] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp));
               ymtmp[1] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp));
               ymtmp[2] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp));
               ymtmp[3] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp));
            }                        
            if (ry > 2) {

               fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ry, py);
               
            }
            else {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC_R2(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3]);
            }
        } else if (px != 0) {
            // Node on X line
            if (rx > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC(crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp), rx, px);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp));
            }
        } else if (py != 0) {
            // Node on Y line
            if (ry > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC(crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp), ry, py);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_CUBIC_R2(crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,0,n+fcomp) = crse(ic,jc,0,n+ccomp);
        }
#endif
#if AMREX_SPACEDIM == 3
        double ymtmp[4];
        double yntmp[4];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int kc = amrex::coarsen(k,ratio[2]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        int pz = k - kc*ratio[2];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        Real rz = Real(ratio[2]);
        if (px != 0 && py != 0 && pz != 0) {
           for (int m=0;m<4;m++) {
              for (int l=0;l<4;l++) {
                 if (rx > 2) {
                    yntmp[l] = POLINT_MACRO_CUBIC(crse(ic-1,jc-1+l,kc-1+m,n+ccomp), crse(ic,jc-1+l,kc-1+m,n+ccomp), crse(ic+1,jc-1+l,kc-1+m,n+ccomp), crse(ic+2,jc-1+l,kc-1+m,n+ccomp), rx, px);
                 }
                 else {
                    yntmp[l] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc-1+l,kc-1+m,n+ccomp), crse(ic,jc-1+l,kc-1+m,n+ccomp), crse(ic+1,jc-1+l,kc-1+m,n+ccomp), crse(ic+2,jc-1+l,kc-1+m,n+ccomp));
                 }
              }
              if (ry > 2) {
                 ymtmp[m] = POLINT_MACRO_CUBIC(yntmp[0],yntmp[1],yntmp[2],yntmp[3], ry, py);
              }
              else {
                 ymtmp[m] = POLINT_MACRO_CUBIC_R2(yntmp[0],yntmp[1],yntmp[2],yntmp[3]);
              }
           }
           if (rz > 2) {
              fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(ymtmp[0],ymtmp[1],ymtmp[2],ymtmp[3], rz, pz);
           }
           else {   
              fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(ymtmp[0],ymtmp[1],ymtmp[2],ymtmp[3]);
           }
        } else if (px != 0 && py != 0) {
            for (int l=0;l<4;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_CUBIC(crse(ic-1,jc-1+l,kc,n+ccomp), crse(ic,jc-1+l,kc,n+ccomp), crse(ic+1,jc-1+l,kc,n+ccomp), crse(ic+2,jc-1+l,kc,n+ccomp), rx, px);
               }
               else {
                  yntmp[l] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc-1+l,kc,n+ccomp), crse(ic,jc-1+l,kc,n+ccomp), crse(ic+1,jc-1+l,kc,n+ccomp), crse(ic+2,jc-1+l,kc,n+ccomp));
               }
            }
            if (ry > 2) {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(yntmp[0],yntmp[1],yntmp[2],yntmp[3], ry, py);
            }
            else {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(yntmp[0],yntmp[1],yntmp[2],yntmp[3]);
            }
        } else if (px != 0 && pz != 0) {
            for (int l=0;l<4;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_CUBIC(crse(ic-1,jc,kc-1+l,n+ccomp), crse(ic,jc,kc-1+l,n+ccomp), crse(ic+1,jc,kc-1+l,n+ccomp), crse(ic+2,jc,kc-1+l,n+ccomp), rx, px);
               }
               else{
                  yntmp[l] = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc,kc-1+l,n+ccomp), crse(ic,jc,kc-1+l,n+ccomp), crse(ic+1,jc,kc-1+l,n+ccomp), crse(ic+2,jc,kc-1+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(yntmp[0],yntmp[1],yntmp[2],yntmp[3], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(yntmp[0],yntmp[1],yntmp[2],yntmp[3]);
            }
        } else if (py != 0 && pz != 0) {
            for (int l=0;l<4;l++) {
               if (ry > 2) {
                  yntmp[l] = POLINT_MACRO_CUBIC(crse(ic,jc-1,kc-1+l,n+ccomp), crse(ic,jc,kc-1+l,n+ccomp), crse(ic,jc+1,kc-1+l,n+ccomp), crse(ic,jc+2,kc-1+l,n+ccomp), ry, py);
               }
               else {
                  yntmp[l] = POLINT_MACRO_CUBIC_R2(crse(ic,jc-1,kc-1+l,n+ccomp), crse(ic,jc,kc-1+l,n+ccomp), crse(ic,jc+1,kc-1+l,n+ccomp), crse(ic,jc+2,kc-1+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(yntmp[0],yntmp[1],yntmp[2],yntmp[3], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(yntmp[0],yntmp[1],yntmp[2],yntmp[3]);  
            }
        } else if (px != 0) {
            if (rx > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp), rx, px);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp));
            }
        } else if (py != 0) {
            if (ry > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp), ry, py);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp));
            }
        } else if (pz != 0) {
            if (rz > 2) {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC(crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp), rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) = POLINT_MACRO_CUBIC_R2(crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,k,n+fcomp) = crse(ic,jc,kc,n+ccomp);
        }
#endif
    });
}

void
LagrangianPolynomic::interp_quintic (const FArrayBox&  crseFAB,
                      int               ccomp,
                      FArrayBox&        fineFAB,
                      int               fcomp,
                      int               ncomp,
                      const Box&        fine_region,
                      const IntVect&    ratio,
                      const Geometry&   crse_geom ,
                      const Geometry&   fine_geom ,
                      Vector<BCRec> const& bcr,
                      int               actual_comp,
                      int               actual_state,
                      RunOn             runon)
{
    Array4<Real const> const& crse = crseFAB.const_array();
    Array4<Real> const& fine = fineFAB.array();
    AMREX_HOST_DEVICE_PARALLEL_FOR_4D_FLAG(runon,fine_region,ncomp,i,j,k,n,
    {
#if AMREX_SPACEDIM == 2
        double ymtmp[6];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        if (px != 0 && py != 0) {
            // Node on a X-Y face
            if (rx > 2) {
               ymtmp[0] = POLINT_MACRO_QUINTIC(crse(ic-2,jc-2,0,n+ccomp), crse(ic-1,jc-2,0,n+ccomp), crse(ic  ,jc-2,0,n+ccomp), crse(ic+1,jc-2,0,n+ccomp), crse(ic+2,jc-2,0,n+ccomp), crse(ic+3,jc-2,0,n+ccomp), rx, px);
               ymtmp[1] = POLINT_MACRO_QUINTIC(crse(ic-2,jc-1,0,n+ccomp), crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp), crse(ic+3,jc-1,0,n+ccomp), rx, px);
               ymtmp[2] = POLINT_MACRO_QUINTIC(crse(ic-2,jc  ,0,n+ccomp), crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp), crse(ic+3,jc  ,0,n+ccomp), rx, px);
               ymtmp[3] = POLINT_MACRO_QUINTIC(crse(ic-2,jc+1,0,n+ccomp), crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp), crse(ic+3,jc+1,0,n+ccomp), rx, px);
               ymtmp[4] = POLINT_MACRO_QUINTIC(crse(ic-2,jc+2,0,n+ccomp), crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp), crse(ic+3,jc+2,0,n+ccomp), rx, px);
               ymtmp[5] = POLINT_MACRO_QUINTIC(crse(ic-2,jc+3,0,n+ccomp), crse(ic-1,jc+3,0,n+ccomp), crse(ic  ,jc+3,0,n+ccomp), crse(ic+1,jc+3,0,n+ccomp), crse(ic+2,jc+3,0,n+ccomp), crse(ic+3,jc+3,0,n+ccomp), rx, px);
            }
            else {
               ymtmp[0] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc-2,0,n+ccomp), crse(ic-1,jc-2,0,n+ccomp), crse(ic  ,jc-2,0,n+ccomp), crse(ic+1,jc-2,0,n+ccomp), crse(ic+2,jc-2,0,n+ccomp), crse(ic+3,jc-2,0,n+ccomp));
               ymtmp[1] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc-1,0,n+ccomp), crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp), crse(ic+3,jc-1,0,n+ccomp));
               ymtmp[2] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc  ,0,n+ccomp), crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp), crse(ic+3,jc  ,0,n+ccomp));
               ymtmp[3] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc+1,0,n+ccomp), crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp), crse(ic+3,jc+1,0,n+ccomp));
               ymtmp[4] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc+2,0,n+ccomp), crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp), crse(ic+3,jc+2,0,n+ccomp));
               ymtmp[5] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc+3,0,n+ccomp), crse(ic-1,jc+3,0,n+ccomp), crse(ic  ,jc+3,0,n+ccomp), crse(ic+1,jc+3,0,n+ccomp), crse(ic+2,jc+3,0,n+ccomp), crse(ic+3,jc+3,0,n+ccomp));
            }                        
            if (ry > 2) {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5], ry, py);
            }
            else {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_R2(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5]);
            }
        } else if (px != 0) {
            // Node on X line
            if (rx > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC(crse(ic-2,jc,0,n+ccomp), crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp), crse(ic+3,jc,0,n+ccomp), rx, px);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc,0,n+ccomp), crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp), crse(ic+3,jc,0,n+ccomp));
            }
        } else if (py != 0) {
            // Node on Y line
            if (ry > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC(crse(ic,jc-2,0,n+ccomp), crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp), crse(ic,jc+3,0,n+ccomp), ry, py);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_R2(crse(ic,jc-2,0,n+ccomp), crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp), crse(ic,jc+3,0,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,0,n+fcomp) = crse(ic,jc,0,n+ccomp);
        }
#endif
#if AMREX_SPACEDIM == 3
        double ymtmp[6];
        double yntmp[6];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int kc = amrex::coarsen(k,ratio[2]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        int pz = k - kc*ratio[2];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        Real rz = Real(ratio[2]);
        if (px != 0 && py != 0 && pz != 0) {
            for (int m=0;m<6;m++) {
                for (int l=0;l<6;l++) {
                    if (rx > 2) {
                        yntmp[l] = POLINT_MACRO_QUINTIC(crse(ic-2,jc-2+l,kc-2+m,n+ccomp), crse(ic-1,jc-2+l,kc-2+m,n+ccomp), crse(ic,jc-2+l,kc-2+m,n+ccomp), crse(ic+1,jc-2+l,kc-2+m,n+ccomp), crse(ic+2,jc-2+l,kc-2+m,n+ccomp), crse(ic+3,jc-2+l,kc-2+m,n+ccomp), rx, px);
                    }
                    else {
                        yntmp[l] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc-2+l,kc-2+m,n+ccomp), crse(ic-1,jc-2+l,kc-2+m,n+ccomp), crse(ic,jc-2+l,kc-2+m,n+ccomp), crse(ic+1,jc-2+l,kc-2+m,n+ccomp), crse(ic+2,jc-2+l,kc-2+m,n+ccomp), crse(ic+3,jc-2+l,kc-2+m,n+ccomp));
                    }
                }
                if (ry > 2) {
                    ymtmp[m] =  POLINT_MACRO_QUINTIC(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], ry, py);
                }
                else {
                    ymtmp[m] =  POLINT_MACRO_QUINTIC_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
                }
            }
            if (rz > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5], rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5]);
            }
        } else if (px != 0 && py != 0) {
            for (int l=0;l<6;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC(crse(ic-2,jc-2+l,kc,n+ccomp), crse(ic-1,jc-2+l,kc,n+ccomp), crse(ic,jc-2+l,kc,n+ccomp), crse(ic+1,jc-2+l,kc,n+ccomp), crse(ic+2,jc-2+l,kc,n+ccomp), crse(ic+3,jc-2+l,kc,n+ccomp), rx, px);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc-2+l,kc,n+ccomp), crse(ic-1,jc-2+l,kc,n+ccomp), crse(ic,jc-2+l,kc,n+ccomp), crse(ic+1,jc-2+l,kc,n+ccomp), crse(ic+2,jc-2+l,kc,n+ccomp), crse(ic+3,jc-2+l,kc,n+ccomp));
               }
            }
            if (ry > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], ry, py);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (px != 0 && pz != 0) {
            for (int l=0;l<6;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC(crse(ic-2,jc,kc-2+l,n+ccomp), crse(ic-1,jc,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic+1,jc,kc-2+l,n+ccomp), crse(ic+2,jc,kc-2+l,n+ccomp), crse(ic+3,jc,kc-2+l,n+ccomp), rx, px);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc,kc-2+l,n+ccomp), crse(ic-1,jc,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic+1,jc,kc-2+l,n+ccomp), crse(ic+2,jc,kc-2+l,n+ccomp), crse(ic+3,jc,kc-2+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (py != 0 && pz != 0) {
            for (int l=0;l<6;l++) {
               if (ry > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC(crse(ic,jc-2,kc-2+l,n+ccomp), crse(ic,jc-1,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic,jc+1,kc-2+l,n+ccomp), crse(ic,jc+2,kc-2+l,n+ccomp), crse(ic,jc+3,kc-2+l,n+ccomp), ry, py);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_R2(crse(ic,jc-2,kc-2+l,n+ccomp), crse(ic,jc-1,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic,jc+1,kc-2+l,n+ccomp), crse(ic,jc+2,kc-2+l,n+ccomp), crse(ic,jc+3,kc-2+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (px != 0) {
            if (rx > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(crse(ic-2,jc,kc,n+ccomp), crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp), crse(ic+3,jc,kc,n+ccomp), rx, px);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(crse(ic-2,jc,kc,n+ccomp), crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp), crse(ic+3,jc,kc,n+ccomp));                               
            }
        } else if (py != 0) {
            if (ry > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(crse(ic,jc-2,kc,n+ccomp), crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp), crse(ic,jc+3,kc,n+ccomp), ry, py);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(crse(ic,jc-2,kc,n+ccomp), crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp), crse(ic,jc+3,kc,n+ccomp));                               
            }
        } else if (pz != 0) {
            if (rz > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC(crse(ic,jc,kc-2,n+ccomp), crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp), crse(ic,jc,kc+3,n+ccomp), rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_R2(crse(ic,jc,kc-2,n+ccomp), crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp), crse(ic,jc,kc+3,n+ccomp));                               
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,k,n+fcomp) = crse(ic,jc,kc,n+ccomp);
        }
#endif
    });
}

void
LagrangianPolynomic::interp_quintic_linear (const FArrayBox&  crseFAB,
                      int               ccomp,
                      FArrayBox&        fineFAB,
                      int               fcomp,
                      int               ncomp,
                      const Box&        fine_region,
                      const IntVect&    ratio,
                      const Geometry&   crse_geom ,
                      const Geometry&   fine_geom ,
                      Vector<BCRec> const& bcr,
                      int               actual_comp,
                      int               actual_state,
                      RunOn             runon)
{
    Array4<Real const> const& crse = crseFAB.const_array();
    Array4<Real> const& fine = fineFAB.array();

    AMREX_HOST_DEVICE_PARALLEL_FOR_4D_FLAG(runon,fine_region,ncomp,i,j,k,n,
    {
#if AMREX_SPACEDIM == 2
        double ymtmp[6];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        if (px != 0 && py != 0) {
            // Node on a X-Y face
            if (rx > 2) {
               ymtmp[0] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc-2,0,n+ccomp), crse(ic-1,jc-2,0,n+ccomp), crse(ic  ,jc-2,0,n+ccomp), crse(ic+1,jc-2,0,n+ccomp), crse(ic+2,jc-2,0,n+ccomp), crse(ic+3,jc-2,0,n+ccomp), rx, px);
               ymtmp[1] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc-1,0,n+ccomp), crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp), crse(ic+3,jc-1,0,n+ccomp), rx, px);
               ymtmp[2] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc  ,0,n+ccomp), crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp), crse(ic+3,jc  ,0,n+ccomp), rx, px);
               ymtmp[3] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc+1,0,n+ccomp), crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp), crse(ic+3,jc+1,0,n+ccomp), rx, px);
               ymtmp[4] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc+2,0,n+ccomp), crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp), crse(ic+3,jc+2,0,n+ccomp), rx, px);
               ymtmp[5] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc+3,0,n+ccomp), crse(ic-1,jc+3,0,n+ccomp), crse(ic  ,jc+3,0,n+ccomp), crse(ic+1,jc+3,0,n+ccomp), crse(ic+2,jc+3,0,n+ccomp), crse(ic+3,jc+3,0,n+ccomp), rx, px);
            }
            else {
               ymtmp[0] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc-2,0,n+ccomp), crse(ic-1,jc-2,0,n+ccomp), crse(ic  ,jc-2,0,n+ccomp), crse(ic+1,jc-2,0,n+ccomp), crse(ic+2,jc-2,0,n+ccomp), crse(ic+3,jc-2,0,n+ccomp));
               ymtmp[1] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc-1,0,n+ccomp), crse(ic-1,jc-1,0,n+ccomp), crse(ic  ,jc-1,0,n+ccomp), crse(ic+1,jc-1,0,n+ccomp), crse(ic+2,jc-1,0,n+ccomp), crse(ic+3,jc-1,0,n+ccomp));
               ymtmp[2] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc  ,0,n+ccomp), crse(ic-1,jc  ,0,n+ccomp), crse(ic  ,jc  ,0,n+ccomp), crse(ic+1,jc  ,0,n+ccomp), crse(ic+2,jc  ,0,n+ccomp), crse(ic+3,jc  ,0,n+ccomp));
               ymtmp[3] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc+1,0,n+ccomp), crse(ic-1,jc+1,0,n+ccomp), crse(ic  ,jc+1,0,n+ccomp), crse(ic+1,jc+1,0,n+ccomp), crse(ic+2,jc+1,0,n+ccomp), crse(ic+3,jc+1,0,n+ccomp));
               ymtmp[4] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc+2,0,n+ccomp), crse(ic-1,jc+2,0,n+ccomp), crse(ic  ,jc+2,0,n+ccomp), crse(ic+1,jc+2,0,n+ccomp), crse(ic+2,jc+2,0,n+ccomp), crse(ic+3,jc+2,0,n+ccomp));
               ymtmp[5] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc+3,0,n+ccomp), crse(ic-1,jc+3,0,n+ccomp), crse(ic  ,jc+3,0,n+ccomp), crse(ic+1,jc+3,0,n+ccomp), crse(ic+2,jc+3,0,n+ccomp), crse(ic+3,jc+3,0,n+ccomp));
            }                        
            if (ry > 2) {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5], ry, py);
            }
            else {
               fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR_R2(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5]);
            }
        } else if (px != 0) {
            // Node on X line
            if (rx > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc,0,n+ccomp), crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp), crse(ic+3,jc,0,n+ccomp), rx, px);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc,0,n+ccomp), crse(ic-1,jc,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic+1,jc,0,n+ccomp), crse(ic+2,jc,0,n+ccomp), crse(ic+3,jc,0,n+ccomp));
            }
        } else if (py != 0) {
            // Node on Y line
            if (ry > 2) {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR(crse(ic,jc-2,0,n+ccomp), crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp), crse(ic,jc+3,0,n+ccomp), ry, py);
            }
            else {
                fine(i,j,0,n+fcomp) = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic,jc-2,0,n+ccomp), crse(ic,jc-1,0,n+ccomp), crse(ic,jc,0,n+ccomp), crse(ic,jc+1,0,n+ccomp), crse(ic,jc+2,0,n+ccomp), crse(ic,jc+3,0,n+ccomp));
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,0,n+fcomp) = crse(ic,jc,0,n+ccomp);
        }
#endif
#if AMREX_SPACEDIM == 3
        double ymtmp[6];
        double yntmp[6];
        int ic = amrex::coarsen(i,ratio[0]);
        int jc = amrex::coarsen(j,ratio[1]);
        int kc = amrex::coarsen(k,ratio[2]);
        int px = i - ic*ratio[0];
        int py = j - jc*ratio[1];
        int pz = k - kc*ratio[2];
        Real rx = Real(ratio[0]);
        Real ry = Real(ratio[1]);
        Real rz = Real(ratio[2]);
        if (px != 0 && py != 0 && pz != 0) {
            for (int m=0;m<6;m++) {
                for (int l=0;l<6;l++) {
                    if (rx > 2) {
                        yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc-2+l,kc-2+m,n+ccomp), crse(ic-1,jc-2+l,kc-2+m,n+ccomp), crse(ic,jc-2+l,kc-2+m,n+ccomp), crse(ic+1,jc-2+l,kc-2+m,n+ccomp), crse(ic+2,jc-2+l,kc-2+m,n+ccomp), crse(ic+3,jc-2+l,kc-2+m,n+ccomp), rx, px);
                    }
                    else {
                        yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc-2+l,kc-2+m,n+ccomp), crse(ic-1,jc-2+l,kc-2+m,n+ccomp), crse(ic,jc-2+l,kc-2+m,n+ccomp), crse(ic+1,jc-2+l,kc-2+m,n+ccomp), crse(ic+2,jc-2+l,kc-2+m,n+ccomp), crse(ic+3,jc-2+l,kc-2+m,n+ccomp));
                    }
                }
                if (ry > 2) {
                    ymtmp[m] =  POLINT_MACRO_QUINTIC_LINEAR(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], ry, py);
                }
                else {
                    ymtmp[m] =  POLINT_MACRO_QUINTIC_LINEAR_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
                }
            }
            if (rz > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5], rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(ymtmp[0], ymtmp[1], ymtmp[2], ymtmp[3], ymtmp[4], ymtmp[5]);
            }
        } else if (px != 0 && py != 0) {
            for (int l=0;l<6;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc-2+l,kc,n+ccomp), crse(ic-1,jc-2+l,kc,n+ccomp), crse(ic,jc-2+l,kc,n+ccomp), crse(ic+1,jc-2+l,kc,n+ccomp), crse(ic+2,jc-2+l,kc,n+ccomp), crse(ic+3,jc-2+l,kc,n+ccomp), rx, px);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc-2+l,kc,n+ccomp), crse(ic-1,jc-2+l,kc,n+ccomp), crse(ic,jc-2+l,kc,n+ccomp), crse(ic+1,jc-2+l,kc,n+ccomp), crse(ic+2,jc-2+l,kc,n+ccomp), crse(ic+3,jc-2+l,kc,n+ccomp));
               }
            }
            if (ry > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], ry, py);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (px != 0 && pz != 0) {
            for (int l=0;l<6;l++) {
               if (rx > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc,kc-2+l,n+ccomp), crse(ic-1,jc,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic+1,jc,kc-2+l,n+ccomp), crse(ic+2,jc,kc-2+l,n+ccomp), crse(ic+3,jc,kc-2+l,n+ccomp), rx, px);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc,kc-2+l,n+ccomp), crse(ic-1,jc,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic+1,jc,kc-2+l,n+ccomp), crse(ic+2,jc,kc-2+l,n+ccomp), crse(ic+3,jc,kc-2+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (py != 0 && pz != 0) {
            for (int l=0;l<6;l++) {
               if (ry > 2) {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR(crse(ic,jc-2,kc-2+l,n+ccomp), crse(ic,jc-1,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic,jc+1,kc-2+l,n+ccomp), crse(ic,jc+2,kc-2+l,n+ccomp), crse(ic,jc+3,kc-2+l,n+ccomp), ry, py);
               }
               else {
                  yntmp[l] = POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic,jc-2,kc-2+l,n+ccomp), crse(ic,jc-1,kc-2+l,n+ccomp), crse(ic,jc,kc-2+l,n+ccomp), crse(ic,jc+1,kc-2+l,n+ccomp), crse(ic,jc+2,kc-2+l,n+ccomp), crse(ic,jc+3,kc-2+l,n+ccomp));
               }
            }
            if (rz > 2) {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5], rz, pz);
            }
            else {
               fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(yntmp[0], yntmp[1], yntmp[2], yntmp[3], yntmp[4], yntmp[5]);
            }
        } else if (px != 0) {
            if (rx > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(crse(ic-2,jc,kc,n+ccomp), crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp), crse(ic+3,jc,kc,n+ccomp), rx, px);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic-2,jc,kc,n+ccomp), crse(ic-1,jc,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic+1,jc,kc,n+ccomp), crse(ic+2,jc,kc,n+ccomp), crse(ic+3,jc,kc,n+ccomp));                               
            }
        } else if (py != 0) {
            if (ry > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(crse(ic,jc-2,kc,n+ccomp), crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp), crse(ic,jc+3,kc,n+ccomp), ry, py);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic,jc-2,kc,n+ccomp), crse(ic,jc-1,kc,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc+1,kc,n+ccomp), crse(ic,jc+2,kc,n+ccomp), crse(ic,jc+3,kc,n+ccomp));                               
            }
        } else if (pz != 0) {
            if (rz > 2) {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR(crse(ic,jc,kc-2,n+ccomp), crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp), crse(ic,jc,kc+3,n+ccomp), rz, pz);
            }
            else {
                fine(i,j,k,n+fcomp) =  POLINT_MACRO_QUINTIC_LINEAR_R2(crse(ic,jc,kc-2,n+ccomp), crse(ic,jc,kc-1,n+ccomp), crse(ic,jc,kc,n+ccomp), crse(ic,jc,kc+1,n+ccomp), crse(ic,jc,kc+2,n+ccomp), crse(ic,jc,kc+3,n+ccomp));                               
            }
        } else {
            // Node coincident with coarse node
            fine(i,j,k,n+fcomp) = crse(ic,jc,kc,n+ccomp);
        }
#endif
    });
}
