#include "Functions.H"
#include "Commons.h"
#include <string>
#include <hdf5.h>

#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))
#define SIGN(X) (((X) > 0) - ((X) < 0))

using namespace external;

/*
 * Reads a table in hdf5 file and stores all relevant data.
 * Only inputFile is an input parameter. The others are filled in this routine.
 *    inputFile:         Name of the table
 *    coord0, ...:       variable storing values of table dimensions
 *    coord0Size, ...:   size of table dimensions
 *    coord0Dx, ...:     spacing of table dimensions
 *    data:              variable data from the table
 *    nVars:             number of variables in the table
 *    exists:            if file exists
 */
void readTable(std::string inputFile, Gpu::ManagedVector<Real>& coord0, int &coord0Size, double &coord0Dx, Gpu::ManagedVector<Real>& data, int& nVars, bool& exists) {
	hid_t file;
	if ((file = H5Fopen(inputFile.c_str(), H5F_ACC_RDONLY, H5P_DEFAULT)) == H5I_INVALID_HID) {
		exists = false;
		std::cout << "**********************************************" << std::endl;
		std::cout << "WARNING. File "<<inputFile<<" cannot be found." << std::endl;
		std::cout << "**********************************************" << std::endl;
	} else {
		exists = true;
		std::cout<<"*******************************"<<std::endl;
		std::cout<<"Reading table file: "<<inputFile<<std::endl;
		std::cout<<"*******************************"<<std::endl;

		// Use these two defines to easily read in a lot of variables in the same way
		// The first reads in one variable of a given type completely
		#define READ_EOS_HDF5(NAME,VAR,TYPE,MEM)                                      \
		do {                                                                        \
			hid_t dataset;                                                            \
			dataset = H5Dopen(file, NAME.c_str(), H5P_DEFAULT);                                \
			H5Dread(dataset, TYPE, MEM, H5S_ALL, H5P_DEFAULT, VAR);       \
			H5Dclose(dataset);                                            \
		} while (0)
		// The second reads a given variable into a hyperslab of the alltables_temp array
		#define READ_EOSTABLE_HDF5(NAME,OFF)                                     \
		do {                                                                   \
			hsize_t offset[2]     = {static_cast<hsize_t>(OFF),0};                                     \
			H5Sselect_hyperslab(mem3, H5S_SELECT_SET, offset, NULL, var3, NULL); \
			READ_EOS_HDF5(NAME,data_temp,H5T_NATIVE_DOUBLE,mem3);           \
		} while (0)

		BL_PROFILE_VAR("readTable(): HDF5 reading", readTable_hdf5);
		// Read size of tables
		hid_t dataset;
		dataset = H5Dopen(file, "nvars", H5P_DEFAULT);
		H5Dread(dataset, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, &nVars);
		H5Dclose(dataset);

		int rank;
		hid_t dataspace;
		std::string coordName;
		hsize_t* dims;
		// Read table coordinates
		int* sizeArr = new int[1];
		dataset = H5Dopen(file, "coord0", H5P_DEFAULT);
		dataspace = H5Dget_space(dataset);
		rank = H5Sget_simple_extent_ndims(dataspace);
		dims = new hsize_t[rank];
		H5Sget_simple_extent_dims(dataspace, dims, dims);
		double *coord0_tmp = new double[dims[0]];
		coord0Size = dims[0];
		coord0.resize(coord0Size);
		coordName = "coord0";
		sizeArr[0] = coord0Size;
		READ_EOS_HDF5(coordName, coord0.data(), H5T_NATIVE_DOUBLE, H5S_ALL);
		coord0Dx = (coord0[coord0Size - 1] - coord0[0])/(coord0Size-1);

		// Allocate memory for table
		double* data_temp;
		if (!(data_temp = (double*)malloc(coord0Size  * nVars * sizeof(double)))) {
			amrex::Abort("Cannot allocate memory for input table.");
		}

		// Prepare HDF5 to read hyperslabs into data
		hsize_t table_dims[2] = {static_cast<hsize_t>(nVars), static_cast<hsize_t>((hsize_t)coord0Size )};
		hsize_t var3[2]       = { 1, (hsize_t)coord0Size };
		hid_t mem3 =  H5Screate_simple(2, table_dims, NULL);
		// Read data
		for (int i = 0; i < nVars; i++) {
			std::string varName = "var" + std::to_string(i);
			READ_EOSTABLE_HDF5(varName,  i);
		}
		H5Fclose(file);

		BL_PROFILE_VAR_STOP(readTable_hdf5);
		BL_PROFILE_VAR("readTable(): transposing", readTable_transposing);
		// change ordering of alltables array so that
		// the table kind is the fastest changing index
		int data_size = nVars;
		for(int count = 0 ; count < 1 ; count++){
			data_size = data_size * sizeArr[count];
		}
		data.resize(data_size);
		for(int iv = 0;iv<nVars;iv++) {
			for(int i0 = 0; i0<coord0Size; i0++) {
				int indold = (i0 + coord0Size * iv);
				int indnew = iv + nVars * i0;
				data[indnew] = data_temp[indold];
			}
		}
		BL_PROFILE_VAR_STOP(readTable_transposing);
		// free memory of temporary array
		free(data_temp);
		free(sizeArr);
	}
}

