#ifndef Functions_H_
#define Functions_H_

#include "AmrCoreProblem.H"
#include "Commons.h"

using namespace simflowny_vars;

/*
 * Calculates coefficients for quintic lagrangian interpolation.
 *    coefs;      coefficients to obtain
 *    coord:      point in which the interpolation must be calculated
 *    position:   surrounding points for interpolation
 */
AMREX_GPU_HOST_DEVICE
AMREX_FORCE_INLINE
void calculateCoefficientsMapFile(double* coefs, double coord, Real* position) {

    for (int i = 0; i < 6; i++) {
        coefs[i] = 0;
    }
    //Search for a perfect fit

    for (int i = 0; i < 6; i++) {
        if (position[i] == coord) {
            coefs[i] = 1;
            return;
        }
    }

    double x1 = position[0];
    double x2 = position[1];
    double x3 = position[2];
    double x4 = position[3];
    double x5 = position[4];
    double x6 = position[5];

    coefs[0] = (coord-x2)*(coord-x3)*(coord-x4)*(coord-x5)*(coord-x6) / ((x1-x2)*(x1-x3)*(x1-x4)*(x1-x5)*(x1-x6));
    coefs[1] = (coord-x1)*(coord-x3)*(coord-x4)*(coord-x5)*(coord-x6) / ((x2-x1)*(x2-x3)*(x2-x4)*(x2-x5)*(x2-x6));
    coefs[2] = (coord-x1)*(coord-x2)*(coord-x4)*(coord-x5)*(coord-x6) / ((x3-x1)*(x3-x2)*(x3-x4)*(x3-x5)*(x3-x6));
    coefs[3] = (coord-x1)*(coord-x2)*(coord-x3)*(coord-x5)*(coord-x6) / ((x4-x1)*(x4-x2)*(x4-x3)*(x4-x5)*(x4-x6));
    coefs[4] = (coord-x1)*(coord-x2)*(coord-x3)*(coord-x4)*(coord-x6) / ((x5-x1)*(x5-x2)*(x5-x3)*(x5-x4)*(x5-x6));
    coefs[5] = (coord-x1)*(coord-x2)*(coord-x3)*(coord-x4)*(coord-x5) / ((x6-x1)*(x6-x2)*(x6-x3)*(x6-x4)*(x6-x5));
}

/*
 * Reads fields from a table interpolating in given coordinates.
 * Uses quintic lagrangian interpolation.
 *    nCoords:            number of coordinates of table
 *    nOutFields:         number of fields to read
 *    coord0, ...:        coordinate values to search in the table
 *    data0, ...:         output variables to store interpolated fields
 *    data0Index, ...:    index of fields inside the table
 *    coord0Data, ...:    variables storing all values of table coordinates
 *    coord0Dx, ...:      spacing of table coordinates
 *    coord0Size, ...:    size of table coordinates
 *    varData, ...:       table data
 *    nVars:              number of variables in the table
 *    exists:             if file exists
 */
AMREX_GPU_HOST_DEVICE
AMREX_FORCE_INLINE
void readFromFileQuintic(int nCoords, int nOutFields, double coord0, double &data0, int data0Index, Real* coord0Data, Real coord0Dx, int coord0Size, Real* varData, int nVars, bool exists) {
	if (!exists) {
		printf("*****************************************************************************************************n");
		printf("ERROR: file does not exist. Check for inexistent file information at the beginning of the simulation.n");
		printf("*****************************************************************************************************n");
		amrex::Abort("");
	}
	int i0  = int((coord0 - coord0Data[0])/coord0Dx);

	if (i0 >= coord0Size || i0 < 0) {
		printf("Mapping File: the input file does not have a value for coordinate %f n", coord0);
		amrex::Abort("");
	}

	int i0b = std::min(std::max(i0-2,0),coord0Size-6);
	double coefs0[6];
	calculateCoefficientsMapFile(coefs0, coord0, &coord0Data[i0b]);
	data0 = 0;
	for (int i = 0; i < 6; i++) {
		data0 = data0 + coefs0[i] *  varData[data0Index + nVars * (i0b + i)];
	}
}


#define POLINT_MACRO_LINEAR_1(y1, y2, i_ext, s_ext, dx, dt_lev) (-i_ext*y1 + i_ext*y2 + s_ext*y2)
#define POLINT_MACRO_LINEAR_2(y1, y2, i_ext, s_ext, dx, dt_lev) (-i_ext*y1 + i_ext*y2 - s_ext*y1)
#define POLINT_MACRO_QUADRATIC_1(y1,y2,y3, i_ext, s_ext, dx, dt_lev) (0.5*(i_ext*i_ext*y1-2*i_ext*i_ext*y2+i_ext*i_ext*y3+i_ext*s_ext*y1-4*i_ext*s_ext*y2+3*i_ext*s_ext*y3+2*s_ext*s_ext*y3)/(s_ext*s_ext))
#define POLINT_MACRO_QUADRATIC_2(y1,y2,y3, i_ext, s_ext, dx, dt_lev) (0.5*(i_ext*i_ext*y1-2*i_ext*i_ext*y2+i_ext*i_ext*y3+3*i_ext*s_ext*y1-4*i_ext*s_ext*y2+i_ext*s_ext*y3+2*s_ext*s_ext*y1)/(s_ext*s_ext))
#define POLINT_MACRO_CUBIC_1(y1,y2,y3, y4, i_ext, s_ext, dx, dt_lev) (-(1.0/6.0)*(i_ext*i_ext*i_ext*y1-3*i_ext*i_ext*i_ext*y2+3*i_ext*i_ext*i_ext*y3-i_ext*i_ext*i_ext*y4+3*i_ext*i_ext*s_ext*y1-12*i_ext*i_ext*s_ext*y2+15*i_ext*i_ext*s_ext*y3-6*i_ext*i_ext*s_ext*y4+2*i_ext*s_ext*s_ext*y1-9*i_ext*s_ext*s_ext*y2+18*i_ext*s_ext*s_ext*y3-11*i_ext*s_ext*s_ext*y4-6*s_ext*s_ext*s_ext*y4)/(s_ext*s_ext*s_ext))
#define POLINT_MACRO_CUBIC_2(y1,y2,y3, y4, i_ext, s_ext, dx, dt_lev) ((1.0/6.0)*(i_ext*i_ext*i_ext*y1-3*i_ext*i_ext*i_ext*y2+3*i_ext*i_ext*i_ext*y3-i_ext*i_ext*i_ext*y4+6*i_ext*i_ext*s_ext*y1-15*i_ext*i_ext*s_ext*y2+12*i_ext*i_ext*s_ext*y3-3*i_ext*i_ext*s_ext*y4+11*i_ext*s_ext*s_ext*y1-18*i_ext*s_ext*s_ext*y2+9*i_ext*s_ext*s_ext*y3-2*i_ext*s_ext*s_ext*y4+6*s_ext*s_ext*s_ext*y1)/(s_ext*s_ext*s_ext))

#define Unit_MCD(i, j, k) ((((int)i) % 10 == 0 && ((int)j) % 10 == 0 && ((int)k) % 10 == 0) ? 10 : (((int)i) % 9 == 0 && ((int)j) % 9 == 0 && ((int)k) % 9 == 0) ? 9 : (((int)i) % 8 == 0 && ((int)j) % 8 == 0 && ((int)k) % 8 == 0) ? 8 : (((int)i) % 7 == 0 && ((int)j) % 7 == 0 && ((int)k) % 7 == 0) ? 7 : (((int)i) % 6 == 0 && ((int)j) % 6 == 0 && ((int)k) % 6 == 0) ? 6 : (((int)i) % 5 == 0 && ((int)j) % 5 == 0 && ((int)k) % 5 == 0) ? 5 : (((int)i) % 4 == 0 && ((int)j) % 4 == 0 && ((int)k) % 4 == 0) ? 4 : (((int)i) % 3 == 0 && ((int)j) % 3 == 0 && ((int)k) % 3 == 0) ? 3 : (((int)i) % 2 == 0 && ((int)j) % 2 == 0 && ((int)k) % 2 == 0) ? 2 : 1)

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4_i(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari + 2, parj, park, paru_idx)) + 8.0 * paru(pari + 1, parj, park, paru_idx)) + ((-8.0 * paru(pari - 1, parj, park, paru_idx)) + paru(pari - 2, parj, park, paru_idx))) / (12.0 * dx[0]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4_j(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj + 2, park, paru_idx)) + 8.0 * paru(pari, parj + 1, park, paru_idx)) + ((-8.0 * paru(pari, parj - 1, park, paru_idx)) + paru(pari, parj - 2, park, paru_idx))) / (12.0 * dx[1]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4_k(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj, park + 2, paru_idx)) + 8.0 * paru(pari, parj, park + 1, paru_idx)) + ((-8.0 * paru(pari, parj, park - 1, paru_idx)) + paru(pari, parj, park - 2, paru_idx))) / (12.0 * dx[2]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double lieforward_i(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((-3.0 * paru(pari - 1, parj, park, paru_idx)) + (-10.0 * paru(pari, parj, park, paru_idx)) + 18.0 * paru(pari + 1, parj, park, paru_idx) + (-6.0 * paru(pari + 2, parj, park, paru_idx)) + paru(pari + 3, parj, park, paru_idx)) / (dx[0] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double lieforward_j(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((-3.0 * paru(pari, parj - 1, park, paru_idx)) + (-10.0 * paru(pari, parj, park, paru_idx)) + 18.0 * paru(pari, parj + 1, park, paru_idx) + (-6.0 * paru(pari, parj + 2, park, paru_idx)) + paru(pari, parj + 3, park, paru_idx)) / (dx[1] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double lieforward_k(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((-3.0 * paru(pari, parj, park - 1, paru_idx)) + (-10.0 * paru(pari, parj, park, paru_idx)) + 18.0 * paru(pari, parj, park + 1, paru_idx) + (-6.0 * paru(pari, parj, park + 2, paru_idx)) + paru(pari, parj, park + 3, paru_idx)) / (dx[2] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double liebackward_i(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari - 3, parj, park, paru_idx)) + 6.0 * paru(pari - 2, parj, park, paru_idx)) - 18.0 * paru(pari - 1, parj, park, paru_idx) + (10.0 * paru(pari, parj, park, paru_idx) + 3.0 * paru(pari + 1, parj, park, paru_idx))) / (dx[0] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double liebackward_j(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj - 3, park, paru_idx)) + 6.0 * paru(pari, parj - 2, park, paru_idx)) - 18.0 * paru(pari, parj - 1, park, paru_idx) + (10.0 * paru(pari, parj, park, paru_idx) + 3.0 * paru(pari, parj + 1, park, paru_idx))) / (dx[1] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double liebackward_k(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj, park - 3, paru_idx)) + 6.0 * paru(pari, parj, park - 2, paru_idx)) - 18.0 * paru(pari, parj, park - 1, paru_idx) + (10.0 * paru(pari, parj, park, paru_idx) + 3.0 * paru(pari, parj, park + 1, paru_idx))) / (dx[2] * 12.0);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D2CDO4_i(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari + 2, parj, park, paru_idx)) + 16.0 * paru(pari + 1, parj, park, paru_idx)) - 30.0 * paru(pari, parj, park, paru_idx) + 16.0 * paru(pari - 1, parj, park, paru_idx) - paru(pari - 2, parj, park, paru_idx)) / (12.0 * (dx[0] * dx[0]));

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D2CDO4_j(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj + 2, park, paru_idx)) + 16.0 * paru(pari, parj + 1, park, paru_idx)) - 30.0 * paru(pari, parj, park, paru_idx) + 16.0 * paru(pari, parj - 1, park, paru_idx) - paru(pari, parj - 2, park, paru_idx)) / (12.0 * (dx[1] * dx[1]));

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D2CDO4_k(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (((-paru(pari, parj, park + 2, paru_idx)) + 16.0 * paru(pari, parj, park + 1, paru_idx)) - 30.0 * paru(pari, parj, park, paru_idx) + 16.0 * paru(pari, parj, park - 1, paru_idx) - paru(pari, parj, park - 2, paru_idx)) / (12.0 * (dx[2] * dx[2]));

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_ii(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari - 2, parj, park, paru_idx) + 8.0 * paru(pari + 1, parj, park, paru_idx) + 64.0 * paru(pari - 1, parj, park, paru_idx) + 8.0 * paru(pari + 2, parj, park, paru_idx) + 8.0 * paru(pari - 2, parj, park, paru_idx) + 64.0 * paru(pari + 1, parj, park, paru_idx) + 8.0 * paru(pari - 1, parj, park, paru_idx) + paru(pari + 2, parj, park, paru_idx)) - (8.0 * paru(pari - 1, parj, park, paru_idx) + paru(pari + 2, parj, park, paru_idx) + 8.0 * paru(pari - 2, parj, park, paru_idx) + 64.0 * paru(pari + 1, parj, park, paru_idx) + 64.0 * paru(pari - 1, parj, park, paru_idx) + 8.0 * paru(pari + 2, parj, park, paru_idx) + paru(pari - 2, parj, park, paru_idx) + 8.0 * paru(pari + 1, parj, park, paru_idx))) / (144.0 * dx[0] * dx[0]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_ij(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari - 2, parj - 2, park, paru_idx) + 8.0 * paru(pari + 1, parj - 2, park, paru_idx) + 64.0 * paru(pari - 1, parj - 1, park, paru_idx) + 8.0 * paru(pari + 2, parj - 1, park, paru_idx) + 8.0 * paru(pari - 2, parj + 1, park, paru_idx) + 64.0 * paru(pari + 1, parj + 1, park, paru_idx) + 8.0 * paru(pari - 1, parj + 2, park, paru_idx) + paru(pari + 2, parj + 2, park, paru_idx)) - (8.0 * paru(pari - 1, parj - 2, park, paru_idx) + paru(pari + 2, parj - 2, park, paru_idx) + 8.0 * paru(pari - 2, parj - 1, park, paru_idx) + 64.0 * paru(pari + 1, parj - 1, park, paru_idx) + 64.0 * paru(pari - 1, parj + 1, park, paru_idx) + 8.0 * paru(pari + 2, parj + 1, park, paru_idx) + paru(pari - 2, parj + 2, park, paru_idx) + 8.0 * paru(pari + 1, parj + 2, park, paru_idx))) / (144.0 * dx[0] * dx[1]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_ik(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari - 2, parj, park - 2, paru_idx) + 8.0 * paru(pari + 1, parj, park - 2, paru_idx) + 64.0 * paru(pari - 1, parj, park - 1, paru_idx) + 8.0 * paru(pari + 2, parj, park - 1, paru_idx) + 8.0 * paru(pari - 2, parj, park + 1, paru_idx) + 64.0 * paru(pari + 1, parj, park + 1, paru_idx) + 8.0 * paru(pari - 1, parj, park + 2, paru_idx) + paru(pari + 2, parj, park + 2, paru_idx)) - (8.0 * paru(pari - 1, parj, park - 2, paru_idx) + paru(pari + 2, parj, park - 2, paru_idx) + 8.0 * paru(pari - 2, parj, park - 1, paru_idx) + 64.0 * paru(pari + 1, parj, park - 1, paru_idx) + 64.0 * paru(pari - 1, parj, park + 1, paru_idx) + 8.0 * paru(pari + 2, parj, park + 1, paru_idx) + paru(pari - 2, parj, park + 2, paru_idx) + 8.0 * paru(pari + 1, parj, park + 2, paru_idx))) / (144.0 * dx[0] * dx[2]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_ji(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari - 2, parj - 2, park, paru_idx) + 8.0 * paru(pari - 2, parj + 1, park, paru_idx) + 64.0 * paru(pari - 1, parj - 1, park, paru_idx) + 8.0 * paru(pari - 1, parj + 2, park, paru_idx) + 8.0 * paru(pari + 1, parj - 2, park, paru_idx) + 64.0 * paru(pari + 1, parj + 1, park, paru_idx) + 8.0 * paru(pari + 2, parj - 1, park, paru_idx) + paru(pari + 2, parj + 2, park, paru_idx)) - (8.0 * paru(pari - 2, parj - 1, park, paru_idx) + paru(pari - 2, parj + 2, park, paru_idx) + 8.0 * paru(pari - 1, parj - 2, park, paru_idx) + 64.0 * paru(pari - 1, parj + 1, park, paru_idx) + 64.0 * paru(pari + 1, parj - 1, park, paru_idx) + 8.0 * paru(pari + 1, parj + 2, park, paru_idx) + paru(pari + 2, parj - 2, park, paru_idx) + 8.0 * paru(pari + 2, parj + 1, park, paru_idx))) / (144.0 * dx[1] * dx[0]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_jj(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari, parj - 2, park, paru_idx) + 8.0 * paru(pari, parj + 1, park, paru_idx) + 64.0 * paru(pari, parj - 1, park, paru_idx) + 8.0 * paru(pari, parj + 2, park, paru_idx) + 8.0 * paru(pari, parj - 2, park, paru_idx) + 64.0 * paru(pari, parj + 1, park, paru_idx) + 8.0 * paru(pari, parj - 1, park, paru_idx) + paru(pari, parj + 2, park, paru_idx)) - (8.0 * paru(pari, parj - 1, park, paru_idx) + paru(pari, parj + 2, park, paru_idx) + 8.0 * paru(pari, parj - 2, park, paru_idx) + 64.0 * paru(pari, parj + 1, park, paru_idx) + 64.0 * paru(pari, parj - 1, park, paru_idx) + 8.0 * paru(pari, parj + 2, park, paru_idx) + paru(pari, parj - 2, park, paru_idx) + 8.0 * paru(pari, parj + 1, park, paru_idx))) / (144.0 * dx[1] * dx[1]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_jk(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari, parj - 2, park - 2, paru_idx) + 8.0 * paru(pari, parj + 1, park - 2, paru_idx) + 64.0 * paru(pari, parj - 1, park - 1, paru_idx) + 8.0 * paru(pari, parj + 2, park - 1, paru_idx) + 8.0 * paru(pari, parj - 2, park + 1, paru_idx) + 64.0 * paru(pari, parj + 1, park + 1, paru_idx) + 8.0 * paru(pari, parj - 1, park + 2, paru_idx) + paru(pari, parj + 2, park + 2, paru_idx)) - (8.0 * paru(pari, parj - 1, park - 2, paru_idx) + paru(pari, parj + 2, park - 2, paru_idx) + 8.0 * paru(pari, parj - 2, park - 1, paru_idx) + 64.0 * paru(pari, parj + 1, park - 1, paru_idx) + 64.0 * paru(pari, parj - 1, park + 1, paru_idx) + 8.0 * paru(pari, parj + 2, park + 1, paru_idx) + paru(pari, parj - 2, park + 2, paru_idx) + 8.0 * paru(pari, parj + 1, park + 2, paru_idx))) / (144.0 * dx[1] * dx[2]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_ki(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari - 2, parj, park - 2, paru_idx) + 8.0 * paru(pari - 2, parj, park + 1, paru_idx) + 64.0 * paru(pari - 1, parj, park - 1, paru_idx) + 8.0 * paru(pari - 1, parj, park + 2, paru_idx) + 8.0 * paru(pari + 1, parj, park - 2, paru_idx) + 64.0 * paru(pari + 1, parj, park + 1, paru_idx) + 8.0 * paru(pari + 2, parj, park - 1, paru_idx) + paru(pari + 2, parj, park + 2, paru_idx)) - (8.0 * paru(pari - 2, parj, park - 1, paru_idx) + paru(pari - 2, parj, park + 2, paru_idx) + 8.0 * paru(pari - 1, parj, park - 2, paru_idx) + 64.0 * paru(pari - 1, parj, park + 1, paru_idx) + 64.0 * paru(pari + 1, parj, park - 1, paru_idx) + 8.0 * paru(pari + 1, parj, park + 2, paru_idx) + paru(pari + 2, parj, park - 2, paru_idx) + 8.0 * paru(pari + 2, parj, park + 1, paru_idx))) / (144.0 * dx[2] * dx[0]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_kj(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari, parj - 2, park - 2, paru_idx) + 8.0 * paru(pari, parj - 2, park + 1, paru_idx) + 64.0 * paru(pari, parj - 1, park - 1, paru_idx) + 8.0 * paru(pari, parj - 1, park + 2, paru_idx) + 8.0 * paru(pari, parj + 1, park - 2, paru_idx) + 64.0 * paru(pari, parj + 1, park + 1, paru_idx) + 8.0 * paru(pari, parj + 2, park - 1, paru_idx) + paru(pari, parj + 2, park + 2, paru_idx)) - (8.0 * paru(pari, parj - 2, park - 1, paru_idx) + paru(pari, parj - 2, park + 2, paru_idx) + 8.0 * paru(pari, parj - 1, park - 2, paru_idx) + 64.0 * paru(pari, parj - 1, park + 1, paru_idx) + 64.0 * paru(pari, parj + 1, park - 1, paru_idx) + 8.0 * paru(pari, parj + 1, park + 2, paru_idx) + paru(pari, parj + 2, park - 2, paru_idx) + 8.0 * paru(pari, parj + 2, park + 1, paru_idx))) / (144.0 * dx[2] * dx[1]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double D1CDO4crossed_kk(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return ((paru(pari, parj, park - 2, paru_idx) + 8.0 * paru(pari, parj, park + 1, paru_idx) + 64.0 * paru(pari, parj, park - 1, paru_idx) + 8.0 * paru(pari, parj, park + 2, paru_idx) + 8.0 * paru(pari, parj, park - 2, paru_idx) + 64.0 * paru(pari, parj, park + 1, paru_idx) + 8.0 * paru(pari, parj, park - 1, paru_idx) + paru(pari, parj, park + 2, paru_idx)) - (8.0 * paru(pari, parj, park - 1, paru_idx) + paru(pari, parj, park + 2, paru_idx) + 8.0 * paru(pari, parj, park - 2, paru_idx) + 64.0 * paru(pari, parj, park + 1, paru_idx) + 64.0 * paru(pari, parj, park - 1, paru_idx) + 8.0 * paru(pari, parj, park + 2, paru_idx) + paru(pari, parj, park - 2, paru_idx) + 8.0 * paru(pari, parj, park + 1, paru_idx))) / (144.0 * dx[2] * dx[2]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double meshDissipation_i(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (paru(pari + 3, parj, park, paru_idx) + (-6.0 * paru(pari + 2, parj, park, paru_idx)) + 15.0 * paru(pari + 1, parj, park, paru_idx) + (-20.0 * paru(pari, parj, park, paru_idx)) + 15.0 * paru(pari - 1, parj, park, paru_idx) + (-6.0 * paru(pari - 2, parj, park, paru_idx)) + paru(pari - 3, parj, park, paru_idx)) / (64.0 * dx[0]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double meshDissipation_j(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (paru(pari, parj + 3, park, paru_idx) + (-6.0 * paru(pari, parj + 2, park, paru_idx)) + 15.0 * paru(pari, parj + 1, park, paru_idx) + (-20.0 * paru(pari, parj, park, paru_idx)) + 15.0 * paru(pari, parj - 1, park, paru_idx) + (-6.0 * paru(pari, parj - 2, park, paru_idx)) + paru(pari, parj - 3, park, paru_idx)) / (64.0 * dx[1]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double meshDissipation_k(int paru_idx, amrex::Array4<amrex::Real> const& paru, int pari, int parj, int park, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (paru(pari, parj, park + 3, paru_idx) + (-6.0 * paru(pari, parj, park + 2, paru_idx)) + 15.0 * paru(pari, parj, park + 1, paru_idx) + (-20.0 * paru(pari, parj, park, paru_idx)) + 15.0 * paru(pari, parj, park - 1, paru_idx) + (-6.0 * paru(pari, parj, park - 2, paru_idx)) + paru(pari, parj, park - 3, paru_idx)) / (64.0 * dx[2]);

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double RK4P1_(Real parRHS, Real parQn, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return parQn + (dt_lev * parRHS) / 2.0;

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double RK4P2_(Real parRHS, Real parQn, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return parQn + (dt_lev * parRHS) / 2.0;

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double RK4P3_(Real parRHS, Real parQn, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return parQn + dt_lev * parRHS;

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
double RK4P4_(Real parRHS, Real parQn, Real parQK1, Real parQK2, Real parQK3, const GpuArray<Real,BL_SPACEDIM> dx, const double dt_lev) {

	return (-parQn / 3.0) + parQK1 / 3.0 + 2.0 * parQK2 / 3.0 + parQK3 / 3.0 + (dt_lev * parRHS) / 6.0;

};

AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
void extrapolate_field(int pard_i_f_idx, amrex::Array4<amrex::Real> const& pard_i_f, int pari, int pard_j_f_idx, amrex::Array4<amrex::Real> const& pard_j_f, int parj, int pard_k_f_idx, amrex::Array4<amrex::Real> const& pard_k_f, int park, int parto_f_idx, amrex::Array4<amrex::Real> const& parto_f, int parFOV_idx, amrex::Array4<amrex::Real> const& parFOV, const GpuArray<Real,BL_SPACEDIM> dx) {
	Real sign_i_ext, sign_j_ext, sign_k_ext, disp_i_j, disp_i_k, disp_j_i, disp_j_k, disp_k_i, disp_k_j, y1, y2, y3, to_i_f, to_j_f, to_k_f, mod_d, term_i, term_j, term_k;

	sign_i_ext = SIGN(pard_i_f(pari, parj, park, pard_i_f_idx));
	sign_j_ext = SIGN(pard_j_f(pari, parj, park, pard_j_f_idx));
	sign_k_ext = SIGN(pard_k_f(pari, parj, park, pard_k_f_idx));
	disp_i_j = 0.0;
	disp_i_k = 0.0;
	disp_j_i = 0.0;
	disp_j_k = 0.0;
	disp_k_i = 0.0;
	disp_k_j = 0.0;
	if (!equalsEq(pard_i_f(pari, parj, park, pard_i_f_idx), 0.0) && equalsEq(parFOV(pari - pard_i_f(pari, parj, park, pard_i_f_idx), parj, park, parFOV_idx), 0.0)) {
		disp_j_i = pard_j_f(pari, parj, park, pard_j_f_idx);
		disp_k_i = pard_k_f(pari, parj, park, pard_k_f_idx);
	}
	if (!equalsEq(pard_j_f(pari, parj, park, pard_j_f_idx), 0.0) && equalsEq(parFOV(pari, parj - pard_j_f(pari, parj, park, pard_j_f_idx), park, parFOV_idx), 0.0)) {
		disp_i_j = pard_i_f(pari, parj, park, pard_i_f_idx);
		disp_k_j = pard_k_f(pari, parj, park, pard_k_f_idx);
	}
	if (!equalsEq(pard_k_f(pari, parj, park, pard_k_f_idx), 0.0) && equalsEq(parFOV(pari, parj, park - pard_k_f(pari, parj, park, pard_k_f_idx), parFOV_idx), 0.0)) {
		disp_i_k = pard_i_f(pari, parj, park, pard_i_f_idx);
		disp_j_k = pard_j_f(pari, parj, park, pard_j_f_idx);
	}
	if (pard_i_f(pari, parj, park, pard_i_f_idx) > 0.0) {
		y1 = parto_f(pari - (pard_i_f(pari, parj, park, pard_i_f_idx) + 2 * sign_i_ext), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		y2 = parto_f(pari - (pard_i_f(pari, parj, park, pard_i_f_idx) + sign_i_ext), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		y3 = parto_f(pari - pard_i_f(pari, parj, park, pard_i_f_idx), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		to_i_f = POLINT_MACRO_QUADRATIC_1(y1, y2, y3, pard_i_f(pari, parj, park, pard_i_f_idx), sign_i_ext, dx, dt_lev);
	}
	if (pard_i_f(pari, parj, park, pard_i_f_idx) < 0.0) {
		y3 = parto_f(pari - (pard_i_f(pari, parj, park, pard_i_f_idx) + 2 * sign_i_ext), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		y2 = parto_f(pari - (pard_i_f(pari, parj, park, pard_i_f_idx) + sign_i_ext), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		y1 = parto_f(pari - pard_i_f(pari, parj, park, pard_i_f_idx), parj - disp_j_i, park - disp_k_i, parto_f_idx);
		to_i_f = POLINT_MACRO_QUADRATIC_2(y1, y2, y3, pard_i_f(pari, parj, park, pard_i_f_idx), sign_i_ext, dx, dt_lev);
	}
	if (pard_j_f(pari, parj, park, pard_j_f_idx) > 0.0) {
		y1 = parto_f(pari - disp_i_j, parj - (pard_j_f(pari, parj, park, pard_j_f_idx) + 2 * sign_j_ext), park - disp_k_j, parto_f_idx);
		y2 = parto_f(pari - disp_i_j, parj - (pard_j_f(pari, parj, park, pard_j_f_idx) + sign_j_ext), park - disp_k_j, parto_f_idx);
		y3 = parto_f(pari - disp_i_j, parj - pard_j_f(pari, parj, park, pard_j_f_idx), park - disp_k_j, parto_f_idx);
		to_j_f = POLINT_MACRO_QUADRATIC_1(y1, y2, y3, pard_j_f(pari, parj, park, pard_j_f_idx), sign_j_ext, dx, dt_lev);
	}
	if (pard_j_f(pari, parj, park, pard_j_f_idx) < 0.0) {
		y3 = parto_f(pari - disp_i_j, parj - (pard_j_f(pari, parj, park, pard_j_f_idx) + 2 * sign_j_ext), park - disp_k_j, parto_f_idx);
		y2 = parto_f(pari - disp_i_j, parj - (pard_j_f(pari, parj, park, pard_j_f_idx) + sign_j_ext), park - disp_k_j, parto_f_idx);
		y1 = parto_f(pari - disp_i_j, parj - pard_j_f(pari, parj, park, pard_j_f_idx), park - disp_k_j, parto_f_idx);
		to_j_f = POLINT_MACRO_QUADRATIC_2(y1, y2, y3, pard_j_f(pari, parj, park, pard_j_f_idx), sign_j_ext, dx, dt_lev);
	}
	if (pard_k_f(pari, parj, park, pard_k_f_idx) > 0.0) {
		y1 = parto_f(pari - disp_i_k, parj - disp_j_k, park - (pard_k_f(pari, parj, park, pard_k_f_idx) + 2 * sign_k_ext), parto_f_idx);
		y2 = parto_f(pari - disp_i_k, parj - disp_j_k, park - (pard_k_f(pari, parj, park, pard_k_f_idx) + sign_k_ext), parto_f_idx);
		y3 = parto_f(pari - disp_i_k, parj - disp_j_k, park - pard_k_f(pari, parj, park, pard_k_f_idx), parto_f_idx);
		to_k_f = POLINT_MACRO_QUADRATIC_1(y1, y2, y3, pard_k_f(pari, parj, park, pard_k_f_idx), sign_k_ext, dx, dt_lev);
	}
	if (pard_k_f(pari, parj, park, pard_k_f_idx) < 0.0) {
		y3 = parto_f(pari - disp_i_k, parj - disp_j_k, park - (pard_k_f(pari, parj, park, pard_k_f_idx) + 2 * sign_k_ext), parto_f_idx);
		y2 = parto_f(pari - disp_i_k, parj - disp_j_k, park - (pard_k_f(pari, parj, park, pard_k_f_idx) + sign_k_ext), parto_f_idx);
		y1 = parto_f(pari - disp_i_k, parj - disp_j_k, park - pard_k_f(pari, parj, park, pard_k_f_idx), parto_f_idx);
		to_k_f = POLINT_MACRO_QUADRATIC_2(y1, y2, y3, pard_k_f(pari, parj, park, pard_k_f_idx), sign_k_ext, dx, dt_lev);
	}
	mod_d = sqrt(pard_i_f(pari, parj, park, pard_i_f_idx) * pard_i_f(pari, parj, park, pard_i_f_idx) + pard_j_f(pari, parj, park, pard_j_f_idx) * pard_j_f(pari, parj, park, pard_j_f_idx) + pard_k_f(pari, parj, park, pard_k_f_idx) * pard_k_f(pari, parj, park, pard_k_f_idx));
	term_i = 0.0;
	if (!equalsEq(pard_i_f(pari, parj, park, pard_i_f_idx), 0.0)) {
		term_i = (to_i_f - parto_f(pari - pard_i_f(pari, parj, park, pard_i_f_idx), parj - disp_j_i, park - disp_k_i, parto_f_idx)) / dx[0];
	}
	term_j = 0.0;
	if (!equalsEq(pard_j_f(pari, parj, park, pard_j_f_idx), 0.0)) {
		term_j = (to_j_f - parto_f(pari - disp_i_j, parj - pard_j_f(pari, parj, park, pard_j_f_idx), park - disp_k_j, parto_f_idx)) / dx[1];
	}
	term_k = 0.0;
	if (!equalsEq(pard_k_f(pari, parj, park, pard_k_f_idx), 0.0)) {
		term_k = (to_k_f - parto_f(pari - disp_i_k, parj - disp_j_k, park - pard_k_f(pari, parj, park, pard_k_f_idx), parto_f_idx)) / dx[2];
	}
	parto_f(pari, parj, park, parto_f_idx) = parto_f(pari - pard_i_f(pari, parj, park, pard_i_f_idx), parj - pard_j_f(pari, parj, park, pard_j_f_idx), park - pard_k_f(pari, parj, park, pard_k_f_idx), parto_f_idx) + (term_i + term_j + term_k) * sqrt((pard_i_f(pari, parj, park, pard_i_f_idx) * dx[0]) * (pard_i_f(pari, parj, park, pard_i_f_idx) * dx[0]) + (pard_j_f(pari, parj, park, pard_j_f_idx) * dx[1]) * (pard_j_f(pari, parj, park, pard_j_f_idx) * dx[1]) + (pard_k_f(pari, parj, park, pard_k_f_idx) * dx[2]) * (pard_k_f(pari, parj, park, pard_k_f_idx) * dx[2])) / mod_d;

};

void readTable(std::string inputFile, Gpu::ManagedVector<Real>& coord0, int &coord0Size, double &coord0Dx, Gpu::ManagedVector<Real>& data, int& nVars, bool& exists);


#endif
