
#include <AMReX_ParallelDescriptor.H>
#include <AMReX_ParmParse.H>
#include <AMReX_MultiFabUtil.H>
#include <AMReX_PlotFileUtil.H>
#include <AMReX_VisMF.H>
#include <AMReX_PhysBCFunct.H>

#ifdef AMREX_MEM_PROFILING
#include <AMReX_MemProfiler.H>
#endif

#include <AmrCoreProblem.H>
#include <Tagging.H>
#include <bc_fill.H>
#include <InitialCondition.H>
#include <LagrangianPolynomicInterpolater.H>
#include <AMReX_FillPatchUtil_mod.H>
#include "HDF5DataWriter.h"
#include "IntegrateDataWriter.h"
#include "PointDataWriter.h"
#include "Functions.H"
#include "SliceDataWriter.h"
#include "SphereDataWriter.h"



#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))
#define SIGN(X) (((X) > 0) - ((X) < 0))
#define isEven(a) ((a) % 2 == 0 ? true : false)
#define greaterEq(a,b) ((fabs((a) - (b))/1.0E-15 > 10 ? false : (floor(fabs((a) - (b))/1.0E-15) < 1)) || (b)<(a))
#define lessEq(a,b) ((fabs((a) - (b))/1.0E-15 > 10 ? false: (floor(fabs((a) - (b))/1.0E-15) < 1)) || (a)<(b))
#define equalsEq(a,b) ((fabs((a) - (b))/1.0E-15 > 10 ? false: (floor(fabs((a) - (b))/1.0E-15) < 1)))
#define reducePrecision(x, p) (floor(((x) * pow(10, (p)) + 0.5)) / pow(10, (p)))

using namespace amrex;
using namespace simflowny_vars;


std::map<std::string, simflowny_vars::AllVariables> AmrCoreProblem::mapVariableValues;
std::map<std::string, simflowny_vars::fieldIndices> AmrCoreProblem::mapVarFieldIdx;
std::map<std::string, simflowny_vars::unSyncAuxFieldIndices> AmrCoreProblem::mapVarUnSyncAuxFieldIdx;
std::map<std::string, simflowny_vars::analysisIndices> AmrCoreProblem::mapVarAnalysisIdx;


	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_dralpha1d_2;
	AMREX_GPU_MANAGED int vars_size_dralpha1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_dralpha1d_2;
	AMREX_GPU_MANAGED int coord0_size_dralpha1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_dralpha1d_2;
	AMREX_GPU_MANAGED bool exists_dralpha1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_dralpha1d_1;
	AMREX_GPU_MANAGED int vars_size_dralpha1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_dralpha1d_1;
	AMREX_GPU_MANAGED int coord0_size_dralpha1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_dralpha1d_1;
	AMREX_GPU_MANAGED bool exists_dralpha1d_1;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_drpsi1d_1;
	AMREX_GPU_MANAGED int vars_size_drpsi1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_drpsi1d_1;
	AMREX_GPU_MANAGED int coord0_size_drpsi1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_drpsi1d_1;
	AMREX_GPU_MANAGED bool exists_drpsi1d_1;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_phi1d_2;
	AMREX_GPU_MANAGED int vars_size_phi1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_phi1d_2;
	AMREX_GPU_MANAGED int coord0_size_phi1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_phi1d_2;
	AMREX_GPU_MANAGED bool exists_phi1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_drpsi1d_2;
	AMREX_GPU_MANAGED int vars_size_drpsi1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_drpsi1d_2;
	AMREX_GPU_MANAGED int coord0_size_drpsi1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_drpsi1d_2;
	AMREX_GPU_MANAGED bool exists_drpsi1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_phi1d_1;
	AMREX_GPU_MANAGED int vars_size_phi1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_phi1d_1;
	AMREX_GPU_MANAGED int coord0_size_phi1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_phi1d_1;
	AMREX_GPU_MANAGED bool exists_phi1d_1;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_alpha1d_2;
	AMREX_GPU_MANAGED int vars_size_alpha1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_alpha1d_2;
	AMREX_GPU_MANAGED int coord0_size_alpha1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_alpha1d_2;
	AMREX_GPU_MANAGED bool exists_alpha1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_psi1d_2;
	AMREX_GPU_MANAGED int vars_size_psi1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_psi1d_2;
	AMREX_GPU_MANAGED int coord0_size_psi1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_psi1d_2;
	AMREX_GPU_MANAGED bool exists_psi1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_drphi1d_1;
	AMREX_GPU_MANAGED int vars_size_drphi1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_drphi1d_1;
	AMREX_GPU_MANAGED int coord0_size_drphi1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_drphi1d_1;
	AMREX_GPU_MANAGED bool exists_drphi1d_1;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_drphi1d_2;
	AMREX_GPU_MANAGED int vars_size_drphi1d_2;
	AMREX_GPU_MANAGED Real coord0_dx_drphi1d_2;
	AMREX_GPU_MANAGED int coord0_size_drphi1d_2;
	AMREX_GPU_MANAGED Real* coord0_data_drphi1d_2;
	AMREX_GPU_MANAGED bool exists_drphi1d_2;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_psi1d_1;
	AMREX_GPU_MANAGED int vars_size_psi1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_psi1d_1;
	AMREX_GPU_MANAGED int coord0_size_psi1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_psi1d_1;
	AMREX_GPU_MANAGED bool exists_psi1d_1;
	//Files to read
	AMREX_GPU_MANAGED Real* vars_data_alpha1d_1;
	AMREX_GPU_MANAGED int vars_size_alpha1d_1;
	AMREX_GPU_MANAGED Real coord0_dx_alpha1d_1;
	AMREX_GPU_MANAGED int coord0_size_alpha1d_1;
	AMREX_GPU_MANAGED Real* coord0_data_alpha1d_1;
	AMREX_GPU_MANAGED bool exists_alpha1d_1;


gsl_rng* Commons::r_var;

// constructor - reads in parameters from inputs file
//             - sizes multilevel arrays and data structures
//             - initializes BCRec boundary condition object
AmrCoreProblem::AmrCoreProblem ()
{
    ReadParameters();

    FillInterpolatorAlternativeSets();

    //Random initialization
    gsl_rng_env_setup();
    //Random for simulation
    int random_seed = random_seed_param*(ParallelDescriptor::MyProc() + 1);
    Commons::r_var = gsl_rng_alloc(gsl_rng_ranlxs0);
    gsl_rng_set(Commons::r_var, random_seed);

    // Geometry on all levels has been defined already.

    // No valid BoxArray and DistributionMapping have been defined.
    // But the arrays for them have been resized.

    int nlevs_max = max_level + 1;

    istep.resize(nlevs_max, 0);
    nsubsteps.resize(nlevs_max, 1);
    if (do_subcycle) {
        for (int lev = 1; lev <= max_level; ++lev) {
            nsubsteps[lev] = MaxRefRatio(lev-1);
        }
    }

    t_new.resize(nlevs_max, 0.0);
    t_old.resize(nlevs_max, -1.e100);
    dt.resize(nlevs_max, 1.e100);

    regions_var.resize(nlevs_max);
	unstate_var.resize(nlevs_max);
	rk1state_var.resize(nlevs_max);
	rk2state_var.resize(nlevs_max);
	rk3state_var.resize(nlevs_max);
	unp1state_var.resize(nlevs_max);
	unSyncAuxFieldGroup_var.resize(nlevs_max);
	analysisGroup_var.resize(nlevs_max);


    regrid_tag_var.resize(nlevs_max);

    bcs.resize(simflowny_vars::NumFields);
    for (int n = 0; n < simflowny_vars::NumFields; ++n)
    {
        for (int idim = 0; idim < AMREX_SPACEDIM; ++idim)
        {
            if (Geom(0).isPeriodic(idim))
            {
                bcs[n].setLo(idim, BCType::int_dir);
                bcs[n].setHi(idim, BCType::int_dir);
            }
            else
            {
                bcs[n].setLo(idim, BCType::ext_dir);
                bcs[n].setHi(idim, BCType::ext_dir);
            }
        }
    }

    //Interpolater
    mapper = new LagrangianPolynomic(false, spatialInterpolator);
    alternative_mapper = new LagrangianPolynomic(false, alternativeSpatialInterpolator);

    //Initialize index mappings
	mapVariableValues["pheI"] = simflowny_vars::pheI_plt;
	mapVarFieldIdx["pheI"] = simflowny_vars::pheI;
	mapVariableValues["trK"] = simflowny_vars::trK_plt;
	mapVarFieldIdx["trK"] = simflowny_vars::trK;
	mapVariableValues["phiI"] = simflowny_vars::phiI_plt;
	mapVarFieldIdx["phiI"] = simflowny_vars::phiI;
	mapVariableValues["theta"] = simflowny_vars::theta_plt;
	mapVarFieldIdx["theta"] = simflowny_vars::theta;
	mapVariableValues["Atd_yy"] = simflowny_vars::Atd_yy_plt;
	mapVarFieldIdx["Atd_yy"] = simflowny_vars::Atd_yy;
	mapVariableValues["Atd_yz"] = simflowny_vars::Atd_yz_plt;
	mapVarFieldIdx["Atd_yz"] = simflowny_vars::Atd_yz;
	mapVariableValues["phi2"] = simflowny_vars::phi2_plt;
	mapVarUnSyncAuxFieldIdx["phi2"] = simflowny_vars::phi2;
	mapVariableValues["Betau_y"] = simflowny_vars::Betau_y_plt;
	mapVarFieldIdx["Betau_y"] = simflowny_vars::Betau_y;
	mapVariableValues["Betau_x"] = simflowny_vars::Betau_x_plt;
	mapVarFieldIdx["Betau_x"] = simflowny_vars::Betau_x;
	mapVariableValues["gtd_yy"] = simflowny_vars::gtd_yy_plt;
	mapVarFieldIdx["gtd_yy"] = simflowny_vars::gtd_yy;
	mapVariableValues["gtd_yz"] = simflowny_vars::gtd_yz_plt;
	mapVarFieldIdx["gtd_yz"] = simflowny_vars::gtd_yz;
	mapVariableValues["Gamh_z"] = simflowny_vars::Gamh_z_plt;
	mapVarFieldIdx["Gamh_z"] = simflowny_vars::Gamh_z;
	mapVariableValues["Betau_z"] = simflowny_vars::Betau_z_plt;
	mapVarFieldIdx["Betau_z"] = simflowny_vars::Betau_z;
	mapVariableValues["chi"] = simflowny_vars::chi_plt;
	mapVarFieldIdx["chi"] = simflowny_vars::chi;
	mapVariableValues["Atd_xz"] = simflowny_vars::Atd_xz_plt;
	mapVarFieldIdx["Atd_xz"] = simflowny_vars::Atd_xz;
	mapVariableValues["pheR"] = simflowny_vars::pheR_plt;
	mapVarFieldIdx["pheR"] = simflowny_vars::pheR;
	mapVariableValues["piI"] = simflowny_vars::piI_plt;
	mapVarFieldIdx["piI"] = simflowny_vars::piI;
	mapVariableValues["Atd_xx"] = simflowny_vars::Atd_xx_plt;
	mapVarFieldIdx["Atd_xx"] = simflowny_vars::Atd_xx;
	mapVariableValues["Atd_zz"] = simflowny_vars::Atd_zz_plt;
	mapVarFieldIdx["Atd_zz"] = simflowny_vars::Atd_zz;
	mapVariableValues["Atd_xy"] = simflowny_vars::Atd_xy_plt;
	mapVarFieldIdx["Atd_xy"] = simflowny_vars::Atd_xy;
	mapVariableValues["phiR"] = simflowny_vars::phiR_plt;
	mapVarFieldIdx["phiR"] = simflowny_vars::phiR;
	mapVariableValues["peI"] = simflowny_vars::peI_plt;
	mapVarFieldIdx["peI"] = simflowny_vars::peI;
	mapVariableValues["Alpha"] = simflowny_vars::Alpha_plt;
	mapVarFieldIdx["Alpha"] = simflowny_vars::Alpha;
	mapVariableValues["piR"] = simflowny_vars::piR_plt;
	mapVarFieldIdx["piR"] = simflowny_vars::piR;
	mapVariableValues["gtd_xx"] = simflowny_vars::gtd_xx_plt;
	mapVarFieldIdx["gtd_xx"] = simflowny_vars::gtd_xx;
	mapVariableValues["gtd_zz"] = simflowny_vars::gtd_zz_plt;
	mapVarFieldIdx["gtd_zz"] = simflowny_vars::gtd_zz;
	mapVariableValues["Gamh_x"] = simflowny_vars::Gamh_x_plt;
	mapVarFieldIdx["Gamh_x"] = simflowny_vars::Gamh_x;
	mapVariableValues["peR"] = simflowny_vars::peR_plt;
	mapVarFieldIdx["peR"] = simflowny_vars::peR;
	mapVariableValues["gtd_xy"] = simflowny_vars::gtd_xy_plt;
	mapVarFieldIdx["gtd_xy"] = simflowny_vars::gtd_xy;
	mapVariableValues["Gamh_y"] = simflowny_vars::Gamh_y_plt;
	mapVarFieldIdx["Gamh_y"] = simflowny_vars::Gamh_y;
	mapVariableValues["gtd_xz"] = simflowny_vars::gtd_xz_plt;
	mapVarFieldIdx["gtd_xz"] = simflowny_vars::gtd_xz;
	mapVariableValues["Rscalar"] = simflowny_vars::Rscalar_plt;
	mapVarAnalysisIdx["Rscalar"] = simflowny_vars::Rscalar;
	mapVariableValues["HamCon"] = simflowny_vars::HamCon_plt;
	mapVarAnalysisIdx["HamCon"] = simflowny_vars::HamCon;
	mapVariableValues["MomCon_x"] = simflowny_vars::MomCon_x_plt;
	mapVarAnalysisIdx["MomCon_x"] = simflowny_vars::MomCon_x;
	mapVariableValues["MomCon_y"] = simflowny_vars::MomCon_y_plt;
	mapVarAnalysisIdx["MomCon_y"] = simflowny_vars::MomCon_y;
	mapVariableValues["MomCon_z"] = simflowny_vars::MomCon_z_plt;
	mapVarAnalysisIdx["MomCon_z"] = simflowny_vars::MomCon_z;
	mapVariableValues["trA"] = simflowny_vars::trA_plt;
	mapVarAnalysisIdx["trA"] = simflowny_vars::trA;
	mapVariableValues["detgtm1"] = simflowny_vars::detgtm1_plt;
	mapVarAnalysisIdx["detgtm1"] = simflowny_vars::detgtm1;
	mapVariableValues["M_ADM_surf"] = simflowny_vars::M_ADM_surf_plt;
	mapVarAnalysisIdx["M_ADM_surf"] = simflowny_vars::M_ADM_surf;
	mapVariableValues["Jz_ADM_surf"] = simflowny_vars::Jz_ADM_surf_plt;
	mapVarAnalysisIdx["Jz_ADM_surf"] = simflowny_vars::Jz_ADM_surf;
	mapVariableValues["N_Noetheri"] = simflowny_vars::N_Noetheri_plt;
	mapVarAnalysisIdx["N_Noetheri"] = simflowny_vars::N_Noetheri;
	mapVariableValues["N_Noethere"] = simflowny_vars::N_Noethere_plt;
	mapVarAnalysisIdx["N_Noethere"] = simflowny_vars::N_Noethere;
	mapVariableValues["M_Komar"] = simflowny_vars::M_Komar_plt;
	mapVarAnalysisIdx["M_Komar"] = simflowny_vars::M_Komar;
	mapVariableValues["Jz_Komar"] = simflowny_vars::Jz_Komar_plt;
	mapVarAnalysisIdx["Jz_Komar"] = simflowny_vars::Jz_Komar;
	mapVariableValues["psi4R"] = simflowny_vars::psi4R_plt;
	mapVarAnalysisIdx["psi4R"] = simflowny_vars::psi4R;
	mapVariableValues["psi4I"] = simflowny_vars::psi4I_plt;
	mapVarAnalysisIdx["psi4I"] = simflowny_vars::psi4I;
	mapVariableValues["Z_x"] = simflowny_vars::Z_x_plt;
	mapVarAnalysisIdx["Z_x"] = simflowny_vars::Z_x;
	mapVariableValues["Z_y"] = simflowny_vars::Z_y_plt;
	mapVarAnalysisIdx["Z_y"] = simflowny_vars::Z_y;
	mapVariableValues["Z_z"] = simflowny_vars::Z_z_plt;
	mapVarAnalysisIdx["Z_z"] = simflowny_vars::Z_z;
	mapVariableValues["regrid_tag"] = simflowny_vars::regrid_tag_plt;


	readTable("dralpha1d_2.h5", gpu_coord0_data_dralpha1d_2, coord0_size_dralpha1d_2, coord0_dx_dralpha1d_2, gpu_vars_data_dralpha1d_2, vars_size_dralpha1d_2, exists_dralpha1d_2);
	coord0_data_dralpha1d_2 = gpu_coord0_data_dralpha1d_2.data();
	vars_data_dralpha1d_2 = gpu_vars_data_dralpha1d_2.data();
	readTable("dralpha1d_1.h5", gpu_coord0_data_dralpha1d_1, coord0_size_dralpha1d_1, coord0_dx_dralpha1d_1, gpu_vars_data_dralpha1d_1, vars_size_dralpha1d_1, exists_dralpha1d_1);
	coord0_data_dralpha1d_1 = gpu_coord0_data_dralpha1d_1.data();
	vars_data_dralpha1d_1 = gpu_vars_data_dralpha1d_1.data();
	readTable("drpsi1d_1.h5", gpu_coord0_data_drpsi1d_1, coord0_size_drpsi1d_1, coord0_dx_drpsi1d_1, gpu_vars_data_drpsi1d_1, vars_size_drpsi1d_1, exists_drpsi1d_1);
	coord0_data_drpsi1d_1 = gpu_coord0_data_drpsi1d_1.data();
	vars_data_drpsi1d_1 = gpu_vars_data_drpsi1d_1.data();
	readTable("phi1d_2.h5", gpu_coord0_data_phi1d_2, coord0_size_phi1d_2, coord0_dx_phi1d_2, gpu_vars_data_phi1d_2, vars_size_phi1d_2, exists_phi1d_2);
	coord0_data_phi1d_2 = gpu_coord0_data_phi1d_2.data();
	vars_data_phi1d_2 = gpu_vars_data_phi1d_2.data();
	readTable("drpsi1d_2.h5", gpu_coord0_data_drpsi1d_2, coord0_size_drpsi1d_2, coord0_dx_drpsi1d_2, gpu_vars_data_drpsi1d_2, vars_size_drpsi1d_2, exists_drpsi1d_2);
	coord0_data_drpsi1d_2 = gpu_coord0_data_drpsi1d_2.data();
	vars_data_drpsi1d_2 = gpu_vars_data_drpsi1d_2.data();
	readTable("phi1d_1.h5", gpu_coord0_data_phi1d_1, coord0_size_phi1d_1, coord0_dx_phi1d_1, gpu_vars_data_phi1d_1, vars_size_phi1d_1, exists_phi1d_1);
	coord0_data_phi1d_1 = gpu_coord0_data_phi1d_1.data();
	vars_data_phi1d_1 = gpu_vars_data_phi1d_1.data();
	readTable("alpha1d_2.h5", gpu_coord0_data_alpha1d_2, coord0_size_alpha1d_2, coord0_dx_alpha1d_2, gpu_vars_data_alpha1d_2, vars_size_alpha1d_2, exists_alpha1d_2);
	coord0_data_alpha1d_2 = gpu_coord0_data_alpha1d_2.data();
	vars_data_alpha1d_2 = gpu_vars_data_alpha1d_2.data();
	readTable("psi1d_2.h5", gpu_coord0_data_psi1d_2, coord0_size_psi1d_2, coord0_dx_psi1d_2, gpu_vars_data_psi1d_2, vars_size_psi1d_2, exists_psi1d_2);
	coord0_data_psi1d_2 = gpu_coord0_data_psi1d_2.data();
	vars_data_psi1d_2 = gpu_vars_data_psi1d_2.data();
	readTable("drphi1d_1.h5", gpu_coord0_data_drphi1d_1, coord0_size_drphi1d_1, coord0_dx_drphi1d_1, gpu_vars_data_drphi1d_1, vars_size_drphi1d_1, exists_drphi1d_1);
	coord0_data_drphi1d_1 = gpu_coord0_data_drphi1d_1.data();
	vars_data_drphi1d_1 = gpu_vars_data_drphi1d_1.data();
	readTable("drphi1d_2.h5", gpu_coord0_data_drphi1d_2, coord0_size_drphi1d_2, coord0_dx_drphi1d_2, gpu_vars_data_drphi1d_2, vars_size_drphi1d_2, exists_drphi1d_2);
	coord0_data_drphi1d_2 = gpu_coord0_data_drphi1d_2.data();
	vars_data_drphi1d_2 = gpu_vars_data_drphi1d_2.data();
	readTable("psi1d_1.h5", gpu_coord0_data_psi1d_1, coord0_size_psi1d_1, coord0_dx_psi1d_1, gpu_vars_data_psi1d_1, vars_size_psi1d_1, exists_psi1d_1);
	coord0_data_psi1d_1 = gpu_coord0_data_psi1d_1.data();
	vars_data_psi1d_1 = gpu_vars_data_psi1d_1.data();
	readTable("alpha1d_1.h5", gpu_coord0_data_alpha1d_1, coord0_size_alpha1d_1, coord0_dx_alpha1d_1, gpu_vars_data_alpha1d_1, vars_size_alpha1d_1, exists_alpha1d_1);
	coord0_data_alpha1d_1 = gpu_coord0_data_alpha1d_1.data();
	vars_data_alpha1d_1 = gpu_vars_data_alpha1d_1.data();


    Commons::initialization();
}

AmrCoreProblem::~AmrCoreProblem ()
{
}

// advance solution to final time
void
AmrCoreProblem::Evolve ()
{
    Real cur_time = t_new[0];    
    int step = istep[0];
    ComputeDt();
    while(!checkFinalization(cur_time, dt[0])) {
        if (Verbose()) {
            amrex::Print() << "\nCoarse STEP " << step+1 << " starts ..." << std::endl;
        }

        ComputeDt();

        int lev = 0;
        int iteration = 0;
        if (do_subcycle)
            timeStepWithSubcycling(lev, cur_time, iteration);
        else
            timeStepNoSubcycling(cur_time, iteration);

        cur_time += dt[0];

        // sync up time
        for (lev = 0; lev <= finest_level; ++lev) {
            t_new[lev] = cur_time;
        }

        if (chk_int > 0 && (step+1) % chk_int == 0) {
            WriteCheckpointFile();
        }

#ifdef AMREX_MEM_PROFILING
        {
            std::ostringstream ss;
            ss << "[STEP " << step+1 << "]";
            MemProfiler::report(ss.str());
        }
#endif
        step++;
    }
}

/*
 * Checks the finalization conditions              
 */
bool AmrCoreProblem::checkFinalization(const double current_time, const double dt) {
if (greaterEq(current_time, parameters.tend)) { 
	return true;
}
return false;


}

// initializes multilevel data
void
AmrCoreProblem::InitData ()
{
    BL_PROFILE("AmrCoreProblem::InitData()");
    Real time;
    int cycle;
    if (restart_chkfile == "") {
        // start simulation from the beginning
        time = 0.0;
        cycle = 0;
        InitFromScratch(time);
        AverageDown();
        Simflowny_par const lparm = parameters;
        postInitialAnalysis(lparm);

        
        //Iteration counter
        for (int il = 0; il <= max_level; il++) {
            current_iteration.push_back(0);
        }
        //Plot iteration initialization
        next_mesh_dump_iteration = plot_int;
        for (int i=0; i<plot_integral_int.size(); i++) {
            next_integration_dump_iteration.push_back(plot_integral_int[i]);
        }
        for (int i=0; i<plot_point_int.size(); i++) {
            next_point_dump_iteration.push_back(plot_point_int[i]);
        }
		for (int i=0; i<plot_slice_int.size(); i++) {
			next_slice_dump_iteration.push_back(plot_slice_int[i]);
		}
		for (int i=0; i<plot_sphere_int.size(); i++) {
			next_sphere_dump_iteration.push_back(plot_sphere_int[i]);
		}

		if (chk_int > 0) {
            WriteCheckpointFile();
        }
    }
    else {
        // restart from a checkpoint
        ReadCheckpointFile();
        time = t_new[0];
        cycle = istep[0];
        for (int i = 0; i < max_level; i++) {
            int ratio = MaxRefRatio(i);
            cycle = cycle * ratio;
        }
        
        if (restart_field_reset) {
            ResetFields();
        }
        
        //Plot iteration initialization
        if (plot_integral_int.size() < plot_point_int.size()) {
            amrex::Abort("Number of integrations cannot be reduced after a checkpoint.");
        }
        for (int i=0; i<plot_integral_int.size(); i++) {
            if (i >= next_integration_dump_iteration.size()) {
                next_integration_dump_iteration.push_back(0);
            }
            if (next_integration_dump_iteration[i] == 0 && plot_integral_int[i] > 0) {
                next_integration_dump_iteration[i] = current_iteration[max_level] + plot_integral_int[i];
            }
        }
        //Plot iteration initialization
        if (plot_point_int.size() < next_point_dump_iteration.size()) {
            amrex::Abort("Number of point output cannot be reduced after a checkpoint.");
        }
        for (int i=0; i<plot_point_int.size(); i++) {
            if (i >= next_point_dump_iteration.size()) {
                next_point_dump_iteration.push_back(0);
            }
            if (next_point_dump_iteration[i] == 0 && plot_point_int[i] > 0) {
                next_point_dump_iteration[i] = current_iteration[max_level] + plot_point_int[i];
            }
        }
		//Plot iteration initialization
		if (plot_slice_int.size() < next_slice_dump_iteration.size()) {
			amrex::Abort("Number of slice output cannot be reduced after a checkpoint.");
		}
		for (int i=0; i<plot_slice_int.size(); i++) {
			if (i >= next_slice_dump_iteration.size()) {
				next_slice_dump_iteration.push_back(0);
			}
			if (next_slice_dump_iteration[i] == 0 && plot_slice_int[i] > 0) {
				next_slice_dump_iteration[i] = current_iteration[max_level] + plot_slice_int[i];
			}
		}
		//Plot iteration initialization
		if (plot_sphere_int.size() < next_sphere_dump_iteration.size()) {
			amrex::Abort("Number of sphere output cannot be reduced after a checkpoint.");
		}
		for (int i=0; i<plot_sphere_int.size(); i++) {
			if (i >= next_sphere_dump_iteration.size()) {
				next_sphere_dump_iteration.push_back(0);
			}
			if (next_sphere_dump_iteration[i] == 0 && plot_sphere_int[i] > 0) {
				next_sphere_dump_iteration[i] = current_iteration[max_level] + plot_sphere_int[i];
			}
		}

    }

    if (plot_int > 0) {
        WritePlotFile(cycle, time);
    }
    for (int i=0; i<plot_integral_int.size(); i++) {
        if (plot_integral_int[i] > 0) {
            WritePlotIntegralFile(i, time);
        }
    }
    for (int i=0; i<plot_point_int.size(); i++) {
        if (plot_point_int[i] > 0) {
            WritePlotPointFile(i, time);
        }
    }
	for (int i=0; i<plot_slice_int.size(); i++) {
		if (plot_slice_int[i] > 0) {
			WritePlotSliceFile(cycle, i, time);
		}
	}
	for (int i=0; i<plot_sphere_int.size(); i++) {
		if (plot_sphere_int[i] > 0) {
			WritePlotSphereFile(cycle, i, time);
		}
	}

}

// Make a new level using provided BoxArray and DistributionMapping and
// fill with interpolated coarse level data.
// overrides the pure virtual function in AmrCore
void
AmrCoreProblem::MakeNewLevelFromCoarse (int lev, Real time, const BoxArray& ba,
                                    const DistributionMapping& dm)
{
    BL_PROFILE("AmrCoreProblem::MakeNewLevelFromCoarse()");

    regions_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumRegions, NUM_GHOST_CELLS);
	unstate_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk2state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk3state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unp1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unSyncAuxFieldGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumUnSyncAuxFields, NUM_GHOST_CELLS);
	analysisGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumAnalysisVars, NUM_GHOST_CELLS);


    regrid_tag_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, NUM_GHOST_CELLS);

    //hierarchy_mask[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, 0);
    //hierarchy_mask[lev].setVal(1);
    //unmask coarser level cells overlaping this new box array.
    //calculateMask(lev, lev-1);

    t_new[lev] = time;
    t_old[lev] = time - 1.e200;

    FillCoarsePatch(lev, time, regions_var[lev], regions_var, 0, NumRegions);
	FillCoarsePatch(lev, time, unp1state_var[lev], unp1state_var, 0, NumFields);
	FillCoarsePatch(lev, time, unSyncAuxFieldGroup_var[lev], unSyncAuxFieldGroup_var, 0, NumUnSyncAuxFields);

}

// Remake an existing level using provided BoxArray and DistributionMapping and
// fill with existing fine and coarse data.
// overrides the pure virtual function in AmrCore
void
AmrCoreProblem::RemakeLevel (int lev, Real time, const BoxArray& ba,
                         const DistributionMapping& dm)
{
BL_PROFILE("AmrCoreProblem::RemakeLevel()");

    MultiFab new_regions(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumRegions, NUM_GHOST_CELLS);
    FillPatch(lev, time, new_regions, regions_var, NumRegions, std::set<int>());
    std::swap(new_regions, regions_var[lev]);

	MultiFab new_state(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	FillPatch(lev, time, new_state, unp1state_var, NumFields, state_interpolator_change);
	std::swap(new_state, unp1state_var[lev]);

	unstate_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk2state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk3state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	MultiFab new_unSyncAuxstate(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumUnSyncAuxFields, NUM_GHOST_CELLS);
	FillPatch(lev, time, new_unSyncAuxstate, unSyncAuxFieldGroup_var, NumUnSyncAuxFields, std::set<int>());
	std::swap(new_unSyncAuxstate, unSyncAuxFieldGroup_var[lev]);
	analysisGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumAnalysisVars, NUM_GHOST_CELLS);


    regrid_tag_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, NUM_GHOST_CELLS);

    //hierarchy_mask[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, 0);
    //hierarchy_mask[lev].setVal(1);
    //unmask coarser level cells overlaping this new box array.
    //if (lev > 0) {
    //    calculateMask(lev, lev-1);
    //}

    t_new[lev] = time;
    t_old[lev] = time - 1.e200;
}

// Delete level data
// overrides the pure virtual function in AmrCore
void
AmrCoreProblem::ClearLevel (int lev)
{
    regions_var[lev].clear();
	unstate_var[lev].clear();
	rk1state_var[lev].clear();
	rk2state_var[lev].clear();
	rk3state_var[lev].clear();
	unp1state_var[lev].clear();
	unSyncAuxFieldGroup_var[lev].clear();
	analysisGroup_var[lev].clear();


    regrid_tag_var[lev].clear();

    //hierarchy_mask[lev].clear();
    //if (lev > 0) {
    //    hierarchy_mask[lev-1].setVal(1);
    //}
}

// Make a new level from scratch using provided BoxArray and DistributionMapping.
// Only used during initialization.
// overrides the pure virtual function in AmrCore
void AmrCoreProblem::MakeNewLevelFromScratch (int lev, Real time, const BoxArray& ba,
                                          const DistributionMapping& dm)
{
    BL_PROFILE("AmrCoreProblem::MakeNewLevelFromScratch()");

    regions_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumRegions, NUM_GHOST_CELLS);
    regions_var[lev].setVal(0.0);
	unstate_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk2state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk3state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unp1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unSyncAuxFieldGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumUnSyncAuxFields, NUM_GHOST_CELLS);
	analysisGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumAnalysisVars, NUM_GHOST_CELLS);

	unstate_var[lev].setVal(0.0);
	rk1state_var[lev].setVal(0.0);
	rk2state_var[lev].setVal(0.0);
	rk3state_var[lev].setVal(0.0);
	unp1state_var[lev].setVal(0.0);
	unSyncAuxFieldGroup_var[lev].setVal(0.0);
	analysisGroup_var[lev].setVal(0.0);


    regrid_tag_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, NUM_GHOST_CELLS);

    //hierarchy_mask[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, 0);
    //hierarchy_mask[lev].setVal(1);
    //unmask coarser level cells overlaping this new box array.
    //if (lev > 0) {
    //    calculateMask(lev, lev-1);
    //}

    t_new[lev] = time;
    t_old[lev] = time - 1.e200;

    MultiFab& regions = regions_var[lev];
	MultiFab& state = unp1state_var[lev];
	MultiFab& unSync_state_aux = unSyncAuxFieldGroup_var[lev];


    const auto problo = Geom(lev).ProbLoArray();
    const auto dx     = Geom(lev).CellSizeArray();
    Simflowny_par const lparm = parameters;

    //Mapping regions
    mapDataOnPatch(time, lev, regions);

    //Calculates the extrapolation variables
    calculateExtrapolationVars(time, lev, regions);



#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
    for (MFIter mfi(state,TilingIfNotGPU()); mfi.isValid(); ++mfi)
    {
        Array4<Real> fab_regions = regions[mfi].array();
        Array4<Real> fab = state[mfi].array();
		Array4<Real> unSync_fab_aux = unSync_state_aux[mfi].array();

        const Box& box = mfi.tilebox();
        const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);

        amrex::launch(gbx,
        [=] AMREX_GPU_DEVICE (Box const& tbx)
        {
            initdata(tbx, fab_regions, fab, unSync_fab_aux, problo, dx, lparm);
        });
    }

    //Post-initialization Sync.
    postInitialization(lev, time, parameters);
}

//Post initialization boundaries and synchronization
void
AmrCoreProblem::postInitialization(const int lev, Real time, Simflowny_par const parameters)
{

	//Variables
	MultiFab& regions_mf = regions_var[lev];
	MultiFab& unSyncAuxFieldGroup_mf = unSyncAuxFieldGroup_var[lev];
	MultiFab& unp1_mf = unp1state_var[lev];
	//First synchronization from initialization
	FillPatch(lev, time, unSyncAuxFieldGroup_mf, unSyncAuxFieldGroup_var, NumUnSyncAuxFields, std::set<int>(), true);
	FillPatch(lev, time, unp1_mf, unp1state_var, NumFields, state_interpolator_change, true);
	Real current_time = time;
	Real level_ratio = 1;
	if (lev > 0) {
		level_ratio = refRatio(lev - 1).max();
	}
	double dt_lev = 0;

	auto dx = geom[lev].CellSizeArray();
	const auto prob_lo = geom[lev].ProbLoArray();
#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
	for (MFIter mfi(regions_mf); mfi.isValid(); ++mfi) {
		Array4<Real> unSyncAuxFieldGroup = unSyncAuxFieldGroup_mf.array(mfi);
		Array4<Real> unp1 = unp1_mf.array(mfi);

		const Box& vbx = mfi.validbox();
		const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
		const Box& bx = mfi.tilebox();
		const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);
		const auto bx_lo = amrex::lbound(gvbx);
		const auto bx_hi = amrex::ubound(gvbx);
		const auto tbx_lo = amrex::lbound(gbx);
		const auto tbx_hi = amrex::ubound(gbx);
		const auto bx_size = gbx.size();

		const auto& domain = geom[lev].Domain();
		const auto& dom_lo = amrex::lbound(domain);
		const auto& dom_hi = amrex::ubound(domain);
		const bool box_touches_lower_x_boundary = bx_lo.x + NUM_GHOST_CELLS == dom_lo.x;
		const bool box_touches_upper_x_boundary = bx_hi.x - NUM_GHOST_CELLS == dom_hi.x + 1;
		const bool box_touches_lower_y_boundary = bx_lo.y + NUM_GHOST_CELLS == dom_lo.y;
		const bool box_touches_upper_y_boundary = bx_hi.y - NUM_GHOST_CELLS == dom_hi.y + 1;
		const bool box_touches_lower_z_boundary = bx_lo.z + NUM_GHOST_CELLS == dom_lo.z;
		const bool box_touches_upper_z_boundary = bx_hi.z - NUM_GHOST_CELLS == dom_hi.z + 1;
		amrex::ParallelFor(gbx, [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {

			if (((i + 3 <= bx_hi.x || !box_touches_upper_x_boundary) && (i - 3 >= bx_lo.x || !box_touches_lower_x_boundary) && (j + 3 <= bx_hi.y || !box_touches_upper_y_boundary) && (j - 3 >= bx_lo.y || !box_touches_lower_y_boundary) && (k + 3 <= bx_hi.z || !box_touches_upper_z_boundary) && (k - 3 >= bx_lo.z || !box_touches_lower_z_boundary))) {
				unSyncAuxFieldGroup(i, j, k, phi2) = unp1(i, j, k, phiR) * unp1(i, j, k, phiR) + unp1(i, j, k, phiI) * unp1(i, j, k, phiI) + unp1(i, j, k, pheR) * unp1(i, j, k, pheR) + unp1(i, j, k, pheI) * unp1(i, j, k, pheI);
			}
		});
	}
#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
	for (MFIter mfi(regions_mf); mfi.isValid(); ++mfi) {
		Array4<Real> regions = regions_mf.array(mfi);
		Array4<Real> unp1 = unp1_mf.array(mfi);
		Array4<Real> unSyncAuxFieldGroup = unSyncAuxFieldGroup_mf.array(mfi);

		const Box& vbx = mfi.validbox();
		const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
		const Box& bx = mfi.tilebox();
		const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);
		const auto bx_lo = amrex::lbound(gvbx);
		const auto bx_hi = amrex::ubound(gvbx);
		const auto tbx_lo = amrex::lbound(gbx);
		const auto tbx_hi = amrex::ubound(gbx);
		const auto bx_size = gbx.size();

		const auto& domain = geom[lev].Domain();
		const auto& dom_lo = amrex::lbound(domain);
		const auto& dom_hi = amrex::ubound(domain);
		amrex::ParallelFor(gbx, [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {

			if (i < dom_lo.x) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
			if (i > dom_hi.x + 1) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
			if (j < dom_lo.y) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
			if (j > dom_hi.y + 1) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
			if (k < dom_lo.z) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
			if (k > dom_hi.z + 1) {
				//Region field extrapolations
				if ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0 || regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) != 0)) {
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, trK, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, theta, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_yz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Betau_z, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, chi, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, pheR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Atd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phiR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peI, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Alpha, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, piR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xx, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_zz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_x, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, peR, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xy, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, Gamh_y, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, gtd_xz, unp1, FOV_1, regions, dx);
					extrapolate_field(d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, i, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, j, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI, regions, k, phi2, unSyncAuxFieldGroup, FOV_1, regions, dx);
				}
			}
		});
	}
	//Last synchronization from initialization
	FillPatch(lev, time, unSyncAuxFieldGroup_mf, unSyncAuxFieldGroup_var, NumUnSyncAuxFields, std::set<int>(), true);
	FillPatch(lev, time, unp1_mf, unp1state_var, NumFields, state_interpolator_change, true);

}

// initial analysis calculation
void 
AmrCoreProblem::postInitialAnalysis(Simflowny_par const parameters)
{
    double current_time = 0;
		for (int level = 0; level <= max_level; level++) {
			MultiFab& un_mf = unstate_var[level];
			MultiFab& rk1_mf = rk1state_var[level];
			MultiFab& rk2_mf = rk2state_var[level];
			MultiFab& rk3_mf = rk3state_var[level];
			MultiFab& unp1_mf = unp1state_var[level];
			MultiFab& unSyncAuxFieldGroup_mf = unSyncAuxFieldGroup_var[level];
			MultiFab& analysisGroup_mf = analysisGroup_var[level];
			auto dx = geom[level].CellSizeArray();
			const auto prob_lo = geom[level].ProbLoArray();
			double dt_lev = dt[level];
	#ifdef AMREX_USE_OMP
	#pragma omp parallel if (Gpu::notInLaunchRegion())
	#endif
			{
				for (MFIter mfi(un_mf, TilingIfNotGPU()); mfi.isValid(); ++mfi) {
					Array4<Real> unp1 = unp1_mf.array(mfi);
					Array4<Real> analysisGroup = analysisGroup_mf.array(mfi);

					// Set up tileboxes and nodal tileboxes
					const Box& vbx = mfi.validbox();
					const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
					const Box& bx = mfi.tilebox();
					const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);

					const auto bx_lo = amrex::lbound(gvbx);
					const auto bx_hi = amrex::ubound(gvbx);
					const auto tbx_lo = amrex::lbound(gbx);
					const auto tbx_hi = amrex::ubound(gbx);
					const auto& domain = geom[level].Domain();
					const auto& dom_lo = amrex::lbound(domain);
					const auto& dom_hi = amrex::ubound(domain);
					amrex::ParallelFor(gbx, 
					[=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {
						Real d_Rpd_xx_o0_t0_m0_l0, d_Rpd_xx_o0_t1_m0_l0, d_Rpd_xx_o0_t2_m0_l0, d_Rpd_xx_o0_t3_m0_l0, d_Rpd_xx_o0_t4_m0_l0, d_Rpd_xx_o0_t5_m0_l0, d_Rpd_xx_o0_t6_m0_l0, d_Rpd_xx_o0_t7_m0_l0, d_Rpd_xx_o0_t8_m0_l0, d_Ctd_xxx_o0_t0_m0_l0, d_Ctd_xxy_o0_t0_m0_l0, d_Ctd_xxz_o0_t0_m0_l0, d_Rtd_xx_o0_t0_m0_l0, d_Rtd_xx_o0_t1_m0_l0, d_Rtd_xx_o0_t2_m0_l0, d_Rtd_xx_o0_t3_m0_l0, d_Rtd_xx_o0_t4_m0_l0, d_Rtd_xx_o0_t5_m0_l0, d_Rtd_xx_o0_t6_m0_l0, d_Rtd_xy_o0_t6_m0_l0, d_Rtd_xz_o0_t6_m0_l0, d_Rtd_xx_o0_t7_m0_l0, d_Rtd_xy_o0_t7_m0_l0, d_Rtd_xz_o0_t7_m0_l0, d_Rtd_xx_o0_t8_m0_l0, d_Rtd_xy_o0_t8_m0_l0, d_Rtd_xz_o0_t8_m0_l0, d_Ctd_yxx_o0_t0_m0_l0, d_Ctd_xyy_o0_t0_m0_l0, d_Ctd_xyz_o0_t1_m0_l0, d_Rtd_xy_o0_t0_m0_l0, d_Rtd_xy_o0_t1_m0_l0, d_Rtd_xy_o0_t2_m0_l0, d_Rtd_xy_o0_t3_m0_l0, d_Rtd_xy_o0_t4_m0_l0, d_Rtd_xy_o0_t5_m0_l0, d_Ctd_zxx_o0_t0_m0_l0, d_Ctd_xyz_o0_t0_m0_l0, d_Ctd_xzz_o0_t0_m0_l0, d_Rtd_xz_o0_t0_m0_l0, d_Rtd_xz_o0_t1_m0_l0, d_Rtd_xz_o0_t2_m0_l0, d_Rtd_xz_o0_t3_m0_l0, d_Rtd_xz_o0_t4_m0_l0, d_Rtd_xz_o0_t5_m0_l0, d_Ctd_xyy_o0_t1_m0_l0, d_Ctd_yyy_o0_t0_m0_l0, d_Ctd_yyz_o0_t0_m0_l0, d_Rtd_yy_o0_t0_m0_l0, d_Rtd_yy_o0_t1_m0_l0, d_Rtd_yy_o0_t2_m0_l0, d_Rtd_yy_o0_t3_m0_l0, d_Rtd_yy_o0_t4_m0_l0, d_Rtd_yy_o0_t5_m0_l0, d_Ctd_xyz_o0_t2_m0_l0, d_Ctd_zyy_o0_t0_m0_l0, d_Ctd_yzz_o0_t0_m0_l0, d_Rtd_yz_o0_t0_m0_l0, d_Rtd_yz_o0_t1_m0_l0, d_Rtd_yz_o0_t2_m0_l0, d_Rtd_yz_o0_t3_m0_l0, d_Rtd_yz_o0_t4_m0_l0, d_Rtd_yz_o0_t5_m0_l0, d_Ctd_xzz_o0_t1_m0_l0, d_Ctd_yzz_o0_t1_m0_l0, d_Ctd_zzz_o0_t0_m0_l0, d_Rtd_zz_o0_t0_m0_l0, d_Rtd_zz_o0_t1_m0_l0, d_Rtd_zz_o0_t2_m0_l0, d_Rtd_zz_o0_t3_m0_l0, d_Rtd_zz_o0_t4_m0_l0, d_Rtd_zz_o0_t5_m0_l0, d_Del_Kd_xxy_o0_t0_m0_l0, d_Del_Kd_yxy_o0_t0_m0_l0, d_Del_Kd_zxy_o0_t0_m0_l0, d_Del_Kd_xxy_o0_t2_m0_l0, d_Del_Kd_yxx_o0_t2_m0_l0, d_Del_Kd_zxx_o0_t2_m0_l0, d_Del_Kd_xxz_o0_t0_m0_l0, d_Del_Kd_yxz_o0_t0_m0_l0, d_Del_Kd_zxz_o0_t0_m0_l0, d_MomCon_x_o0_t3_m0_l0, d_Del_Kd_yxx_o0_t0_m0_l0, d_Del_Kd_zxx_o0_t0_m0_l0, d_Del_Kd_xyy_o0_t0_m0_l0, d_MomCon_y_o0_t7_m0_l0, d_Del_Kd_zyy_o0_t0_m0_l0, d_Del_Kd_xyz_o0_t0_m0_l0, d_Del_Kd_yyz_o0_t0_m0_l0, d_Del_Kd_zyz_o0_t0_m0_l0, d_Del_Kd_xzz_o0_t0_m0_l0, d_Del_Kd_yzz_o0_t0_m0_l0, d_MomCon_z_o0_t11_m0_l0, d_d_phiR4d_t_o0_t1_m0_l0, d_d_phiR4d_t_o0_t2_m0_l0, d_d_phiR4d_t_o0_t3_m0_l0, d_d_phiI4d_t_o0_t1_m0_l0, d_d_phiI4d_t_o0_t2_m0_l0, d_d_phiI4d_t_o0_t3_m0_l0, d_d_pheR4d_t_o0_t1_m0_l0, d_d_pheR4d_t_o0_t2_m0_l0, d_d_pheR4d_t_o0_t3_m0_l0, d_d_pheI4d_t_o0_t1_m0_l0, d_d_pheI4d_t_o0_t2_m0_l0, d_d_pheI4d_t_o0_t3_m0_l0, sfmass, inv_chi, g4d_zz, g4d_yz, g4d_yy, ur_z, ur_y, uph_z, d_d_phiR4d_t_o0_t0_m0_l0, m_d_phiR4d_t_o0_t6_l0, m_d_phiR4d_t_o0_t5_l0, m_d_phiR4d_t_o0_t4_l0, m_d_phiR4d_t_o0_t3_l0, m_d_phiR4d_t_o0_t2_l0, m_d_phiR4d_t_o0_t1_l0, d_phiR4d_t, d_phiR4d_x, d_phiR4d_z, d_phiR4d_y, uph_x, uph_y, ur_x, d_d_pheR4d_t_o0_t0_m0_l0, m_d_pheR4d_t_o0_t6_l0, m_d_pheR4d_t_o0_t5_l0, m_d_pheR4d_t_o0_t4_l0, m_d_pheR4d_t_o0_t3_l0, m_d_pheR4d_t_o0_t2_l0, m_d_pheR4d_t_o0_t1_l0, d_pheR4d_t, d_pheR4d_x, d_pheR4d_y, d_pheR4d_z, uthd_y, uthd_x, uthd_z, m_Ctd_zzz_o0_t0_l0, Ctd_zzz, wrr, m_Ctd_yyz_o0_t0_l0, Ctd_yyz, m_Ctd_zyy_o0_t1_l0, m_Ctd_zyy_o0_t0_l0, Ctd_zyy, m_Ctd_yyy_o0_t0_l0, Ctd_yyy, m_Ctd_xxz_o0_t0_l0, Ctd_xxz, m_Ctd_zxx_o0_t1_l0, m_Ctd_zxx_o0_t0_l0, Ctd_zxx, m_Ctd_xxx_o0_t0_l0, Ctd_xxx, m_Ctd_yzz_o0_t1_l0, m_Ctd_yzz_o0_t0_l0, Ctd_yzz, m_Ctd_zyz_o0_t0_l0, Ctd_zyz, m_Ctd_xxy_o0_t0_l0, Ctd_xxy, m_Ctd_yxx_o0_t1_l0, m_Ctd_yxx_o0_t0_l0, Ctd_yxx, m_Ctd_xyy_o0_t1_l0, m_Ctd_xyy_o0_t0_l0, Ctd_xyy, m_Ctd_yxy_o0_t0_l0, Ctd_yxy, m_Ctd_xyz_o0_t2_l0, m_Ctd_xyz_o0_t1_l0, m_Ctd_xyz_o0_t0_l0, Ctd_xyz, m_Ctd_yxz_o0_t2_l0, m_Ctd_yxz_o0_t0_l0, Ctd_yxz, m_Ctd_zxy_o0_t2_l0, Ctd_zxy, g4d_tt, g4d_ty, g4d_tx, g4d_tz, m_Ctd_xzz_o0_t1_l0, m_Ctd_xzz_o0_t0_l0, Ctd_xzz, m_Ctd_zxz_o0_t0_l0, Ctd_zxz, dSigma_y, dSigma_x, dSigma_z, d_d_phiI4d_t_o0_t0_m0_l0, m_d_phiI4d_t_o0_t6_l0, m_d_phiI4d_t_o0_t5_l0, m_d_phiI4d_t_o0_t4_l0, m_d_phiI4d_t_o0_t3_l0, m_d_phiI4d_t_o0_t2_l0, m_d_phiI4d_t_o0_t1_l0, d_phiI4d_t, d_phiI4d_x, d_phiI4d_y, d_phiI4d_z, g4u_tx, g4u_tz, g4u_ty, g4u_tt, d_phiR4u_t, d_phiI4u_t, d_pheR4u_t, sfVe, sfVi, detgtd, idetgtd, gtu_zz, g4u_zz, gtu_yy, g4u_yy, gtu_yz, g4u_yz, gtu_xx, g4u_xx, gtu_xz, g4u_xz, d_phiR4u_z, d_phiI4u_z, d_pheR4u_z, uth_z, Atud_zx, Atud_zz, Atud_zy, Atu_zz, Ct_zxz, Ct_zxy, Ct_zxx, Ct_zyz, Ct_zyy, Ct_zzz, gtu_xy, g4u_xy, d_phiR4u_y, d_phiR4u_x, d_phiI4u_y, d_phiI4u_x, dphi4sq, d_pheR4u_x, d_pheR4u_y, Ct_xzz, uth_x, uth_y, Gamt_z, Zu_z, Atud_yy, Atud_yx, Atud_yz, Atu_yz, Atud_xx, Atud_xz, Atud_xy, Atu_xz, Atu_yy, Ct_yxx, Ct_yxz, Ct_yxy, Atu_xy, Atu_xx, Ct_yyy, Ct_yyz, Ct_yzz, Gamt_y, Zu_y, Ct_xxy, Ct_xxx, Ct_xxz, d_Rtd_xx_o0_t9_m0_l0, m_Rtd_xx_o0_t8_l0, m_Rtd_xx_o0_t7_l0, m_Rtd_xx_o0_t6_l0, m_Rtd_xx_o0_t5_l0, m_Rtd_xx_o0_t4_l0, m_Rtd_xx_o0_t3_l0, m_Rtd_xx_o0_t2_l0, m_Rtd_xx_o0_t1_l0, m_Rtd_xx_o0_t0_l0, Rtd_xx, g4d_xy, wphph, vph_y, vph_z, vph_x, wthph, g4d_xx, wrph, vr_x, vr_y, vr_z, wthr, g4d_xz, wthth, vth_x, mmR_xx, mmI_xx, vth_y, mmR_yy, mmR_xy, mmI_xy, mmI_yy, vth_z, mmR_yz, mmR_zz, mmR_xz, mmI_zz, mmI_xz, mmI_yz, Ct_xyz, d_Rtd_xz_o0_t12_m0_l0, m_Rtd_xz_o0_t11_l0, m_Rtd_xz_o0_t10_l0, m_Rtd_xz_o0_t9_l0, m_Rtd_xz_o0_t8_l0, m_Rtd_xz_o0_t7_l0, m_Rtd_xz_o0_t6_l0, m_Rtd_xz_o0_t5_l0, m_Rtd_xz_o0_t4_l0, m_Rtd_xz_o0_t3_l0, m_Rtd_xz_o0_t2_l0, m_Rtd_xz_o0_t1_l0, m_Rtd_xz_o0_t0_l0, Rtd_xz, d_Rtd_zz_o0_t9_m0_l0, m_Rtd_zz_o0_t8_l0, m_Rtd_zz_o0_t7_l0, m_Rtd_zz_o0_t6_l0, m_Rtd_zz_o0_t5_l0, m_Rtd_zz_o0_t4_l0, m_Rtd_zz_o0_t3_l0, m_Rtd_zz_o0_t2_l0, m_Rtd_zz_o0_t1_l0, m_Rtd_zz_o0_t0_l0, Rtd_zz, Ct_xyy, Gamt_x, Zu_x, m_Rpd_xx_o0_t22_l0, m_Rpd_xx_o0_t21_l0, m_Rpd_xx_o0_t20_l0, m_Rpd_xx_o0_t19_l0, m_Rpd_xx_o0_t18_l0, m_Rpd_xx_o0_t17_l0, m_Rpd_xx_o0_t16_l0, m_Rpd_xx_o0_t15_l0, m_Rpd_xx_o0_t14_l0, m_Rpd_xx_o0_t13_l0, m_Rpd_xx_o0_t12_l0, m_Rpd_xx_o0_t11_l0, m_Rpd_xx_o0_t10_l0, m_Rpd_xx_o0_t9_l0, m_Rpd_xx_o0_t8_l0, m_Rpd_xx_o0_t7_l0, m_Rpd_xx_o0_t6_l0, m_Rpd_xx_o0_t5_l0, m_Rpd_xx_o0_t4_l0, m_Rpd_xx_o0_t3_l0, m_Rpd_xx_o0_t2_l0, m_Rpd_xx_o0_t1_l0, m_Rpd_xx_o0_t0_l0, Rpd_xx, d_Del_Kd_yxx_o0_t1_m0_l0, m_Del_Kd_yxx_o0_t6_l0, m_Del_Kd_yxx_o0_t5_l0, m_Del_Kd_yxx_o0_t4_l0, m_Del_Kd_yxx_o0_t3_l0, m_Del_Kd_yxx_o0_t2_l0, m_Del_Kd_yxx_o0_t0_l0, Del_Kd_yxx, d_Del_Kd_yxz_o0_t1_m0_l0, m_Del_Kd_yxz_o0_t10_l0, m_Del_Kd_yxz_o0_t9_l0, m_Del_Kd_yxz_o0_t8_l0, m_Del_Kd_yxz_o0_t7_l0, m_Del_Kd_yxz_o0_t6_l0, m_Del_Kd_yxz_o0_t5_l0, m_Del_Kd_yxz_o0_t4_l0, m_Del_Kd_yxz_o0_t3_l0, m_Del_Kd_yxz_o0_t2_l0, m_Del_Kd_yxz_o0_t0_l0, Del_Kd_yxz, EWeyl_xx, d_Del_Kd_zxx_o0_t1_m0_l0, m_Del_Kd_zxx_o0_t6_l0, m_Del_Kd_zxx_o0_t5_l0, m_Del_Kd_zxx_o0_t4_l0, m_Del_Kd_zxx_o0_t3_l0, m_Del_Kd_zxx_o0_t2_l0, m_Del_Kd_zxx_o0_t0_l0, Del_Kd_zxx, d_Del_Kd_zyz_o0_t1_m0_l0, m_Del_Kd_zyz_o0_t10_l0, m_Del_Kd_zyz_o0_t9_l0, m_Del_Kd_zyz_o0_t8_l0, m_Del_Kd_zyz_o0_t7_l0, m_Del_Kd_zyz_o0_t6_l0, m_Del_Kd_zyz_o0_t5_l0, m_Del_Kd_zyz_o0_t4_l0, m_Del_Kd_zyz_o0_t3_l0, m_Del_Kd_zyz_o0_t2_l0, m_Del_Kd_zyz_o0_t0_l0, Del_Kd_zyz, d_Del_Kd_zyy_o0_t1_m0_l0, m_Del_Kd_zyy_o0_t6_l0, m_Del_Kd_zyy_o0_t5_l0, m_Del_Kd_zyy_o0_t4_l0, m_Del_Kd_zyy_o0_t3_l0, m_Del_Kd_zyy_o0_t2_l0, m_Del_Kd_zyy_o0_t0_l0, Del_Kd_zyy, d_Del_Kd_zxy_o0_t1_m0_l0, m_Del_Kd_zxy_o0_t7_l0, m_Del_Kd_zxy_o0_t6_l0, m_Del_Kd_zxy_o0_t5_l0, m_Del_Kd_zxy_o0_t4_l0, m_Del_Kd_zxy_o0_t2_l0, m_Del_Kd_zxy_o0_t0_l0, Del_Kd_zxy, d_Del_Kd_zxz_o0_t1_m0_l0, m_Del_Kd_zxz_o0_t10_l0, m_Del_Kd_zxz_o0_t9_l0, m_Del_Kd_zxz_o0_t8_l0, m_Del_Kd_zxz_o0_t7_l0, m_Del_Kd_zxz_o0_t6_l0, m_Del_Kd_zxz_o0_t5_l0, m_Del_Kd_zxz_o0_t4_l0, m_Del_Kd_zxz_o0_t3_l0, m_Del_Kd_zxz_o0_t2_l0, m_Del_Kd_zxz_o0_t0_l0, Del_Kd_zxz, m_Rpd_yz_o0_t25_l0, m_Rpd_yz_o0_t24_l0, m_Rpd_yz_o0_t23_l0, m_Rpd_yz_o0_t22_l0, m_Rpd_yz_o0_t21_l0, m_Rpd_yz_o0_t20_l0, m_Rpd_yz_o0_t19_l0, m_Rpd_yz_o0_t18_l0, m_Rpd_yz_o0_t17_l0, m_Rpd_yz_o0_t16_l0, m_Rpd_yz_o0_t15_l0, m_Rpd_yz_o0_t14_l0, m_Rpd_yz_o0_t13_l0, m_Rpd_yz_o0_t12_l0, m_Rpd_yz_o0_t11_l0, m_Rpd_yz_o0_t10_l0, m_Rpd_yz_o0_t9_l0, m_Rpd_yz_o0_t8_l0, m_Rpd_yz_o0_t7_l0, m_Rpd_yz_o0_t6_l0, m_Rpd_yz_o0_t5_l0, m_Rpd_yz_o0_t4_l0, m_Rpd_yz_o0_t3_l0, m_Rpd_yz_o0_t2_l0, m_Rpd_yz_o0_t1_l0, m_Rpd_yz_o0_t0_l0, Rpd_yz, d_Del_Kd_yzz_o0_t1_m0_l0, m_Del_Kd_yzz_o0_t6_l0, m_Del_Kd_yzz_o0_t5_l0, m_Del_Kd_yzz_o0_t4_l0, m_Del_Kd_yzz_o0_t3_l0, m_Del_Kd_yzz_o0_t2_l0, m_Del_Kd_yzz_o0_t0_l0, Del_Kd_yzz, d_Del_Kd_xxy_o0_t1_m0_l0, m_Del_Kd_xxy_o0_t10_l0, m_Del_Kd_xxy_o0_t9_l0, m_Del_Kd_xxy_o0_t8_l0, m_Del_Kd_xxy_o0_t7_l0, m_Del_Kd_xxy_o0_t6_l0, m_Del_Kd_xxy_o0_t5_l0, m_Del_Kd_xxy_o0_t4_l0, m_Del_Kd_xxy_o0_t3_l0, m_Del_Kd_xxy_o0_t2_l0, m_Del_Kd_xxy_o0_t0_l0, Del_Kd_xxy, d_Del_Kd_xzz_o0_t1_m0_l0, m_Del_Kd_xzz_o0_t6_l0, m_Del_Kd_xzz_o0_t5_l0, m_Del_Kd_xzz_o0_t4_l0, m_Del_Kd_xzz_o0_t3_l0, m_Del_Kd_xzz_o0_t2_l0, m_Del_Kd_xzz_o0_t0_l0, Del_Kd_xzz, d_Del_Kd_xyz_o0_t1_m0_l0, m_Del_Kd_xyz_o0_t2_l0, m_Del_Kd_xyz_o0_t0_l0, Del_Kd_xyz, BWeyl_xz, BWeyl_zz, BWeyl_yz, d_Del_Kd_xyy_o0_t1_m0_l0, m_Del_Kd_xyy_o0_t6_l0, m_Del_Kd_xyy_o0_t5_l0, m_Del_Kd_xyy_o0_t4_l0, m_Del_Kd_xyy_o0_t3_l0, m_Del_Kd_xyy_o0_t2_l0, m_Del_Kd_xyy_o0_t0_l0, Del_Kd_xyy, d_Del_Kd_xxz_o0_t1_m0_l0, m_Del_Kd_xxz_o0_t10_l0, m_Del_Kd_xxz_o0_t9_l0, m_Del_Kd_xxz_o0_t8_l0, m_Del_Kd_xxz_o0_t7_l0, m_Del_Kd_xxz_o0_t6_l0, m_Del_Kd_xxz_o0_t5_l0, m_Del_Kd_xxz_o0_t4_l0, m_Del_Kd_xxz_o0_t3_l0, m_Del_Kd_xxz_o0_t2_l0, m_Del_Kd_xxz_o0_t0_l0, Del_Kd_xxz, BWeyl_xx, BWeyl_zx, BWeyl_yx, m_Rpd_xz_o0_t25_l0, m_Rpd_xz_o0_t24_l0, m_Rpd_xz_o0_t23_l0, m_Rpd_xz_o0_t22_l0, m_Rpd_xz_o0_t21_l0, m_Rpd_xz_o0_t20_l0, m_Rpd_xz_o0_t19_l0, m_Rpd_xz_o0_t18_l0, m_Rpd_xz_o0_t17_l0, m_Rpd_xz_o0_t16_l0, m_Rpd_xz_o0_t15_l0, m_Rpd_xz_o0_t14_l0, m_Rpd_xz_o0_t13_l0, m_Rpd_xz_o0_t12_l0, m_Rpd_xz_o0_t11_l0, m_Rpd_xz_o0_t10_l0, m_Rpd_xz_o0_t9_l0, m_Rpd_xz_o0_t8_l0, m_Rpd_xz_o0_t7_l0, m_Rpd_xz_o0_t6_l0, m_Rpd_xz_o0_t5_l0, m_Rpd_xz_o0_t4_l0, m_Rpd_xz_o0_t3_l0, m_Rpd_xz_o0_t2_l0, m_Rpd_xz_o0_t1_l0, m_Rpd_xz_o0_t0_l0, Rpd_xz, EWeyl_xz, m_Rpd_xy_o0_t25_l0, m_Rpd_xy_o0_t24_l0, m_Rpd_xy_o0_t23_l0, m_Rpd_xy_o0_t22_l0, m_Rpd_xy_o0_t21_l0, m_Rpd_xy_o0_t20_l0, m_Rpd_xy_o0_t19_l0, m_Rpd_xy_o0_t18_l0, m_Rpd_xy_o0_t17_l0, m_Rpd_xy_o0_t16_l0, m_Rpd_xy_o0_t15_l0, m_Rpd_xy_o0_t14_l0, m_Rpd_xy_o0_t13_l0, m_Rpd_xy_o0_t12_l0, m_Rpd_xy_o0_t11_l0, m_Rpd_xy_o0_t10_l0, m_Rpd_xy_o0_t9_l0, m_Rpd_xy_o0_t8_l0, m_Rpd_xy_o0_t7_l0, m_Rpd_xy_o0_t6_l0, m_Rpd_xy_o0_t5_l0, m_Rpd_xy_o0_t4_l0, m_Rpd_xy_o0_t3_l0, m_Rpd_xy_o0_t2_l0, m_Rpd_xy_o0_t1_l0, m_Rpd_xy_o0_t0_l0, Rpd_xy, m_Rpd_zz_o0_t22_l0, m_Rpd_zz_o0_t21_l0, m_Rpd_zz_o0_t20_l0, m_Rpd_zz_o0_t19_l0, m_Rpd_zz_o0_t18_l0, m_Rpd_zz_o0_t17_l0, m_Rpd_zz_o0_t16_l0, m_Rpd_zz_o0_t15_l0, m_Rpd_zz_o0_t14_l0, m_Rpd_zz_o0_t13_l0, m_Rpd_zz_o0_t12_l0, m_Rpd_zz_o0_t11_l0, m_Rpd_zz_o0_t10_l0, m_Rpd_zz_o0_t9_l0, m_Rpd_zz_o0_t8_l0, m_Rpd_zz_o0_t7_l0, m_Rpd_zz_o0_t6_l0, m_Rpd_zz_o0_t5_l0, m_Rpd_zz_o0_t4_l0, m_Rpd_zz_o0_t3_l0, m_Rpd_zz_o0_t2_l0, m_Rpd_zz_o0_t1_l0, m_Rpd_zz_o0_t0_l0, Rpd_zz, EWeyl_zz, d_Del_Kd_yyz_o0_t1_m0_l0, m_Del_Kd_yyz_o0_t10_l0, m_Del_Kd_yyz_o0_t9_l0, m_Del_Kd_yyz_o0_t8_l0, m_Del_Kd_yyz_o0_t7_l0, m_Del_Kd_yyz_o0_t6_l0, m_Del_Kd_yyz_o0_t5_l0, m_Del_Kd_yyz_o0_t4_l0, m_Del_Kd_yyz_o0_t3_l0, m_Del_Kd_yyz_o0_t2_l0, m_Del_Kd_yyz_o0_t0_l0, Del_Kd_yyz, d_Del_Kd_yxy_o0_t1_m0_l0, m_Del_Kd_yxy_o0_t10_l0, m_Del_Kd_yxy_o0_t9_l0, m_Del_Kd_yxy_o0_t8_l0, m_Del_Kd_yxy_o0_t7_l0, m_Del_Kd_yxy_o0_t6_l0, m_Del_Kd_yxy_o0_t5_l0, m_Del_Kd_yxy_o0_t4_l0, m_Del_Kd_yxy_o0_t3_l0, m_Del_Kd_yxy_o0_t2_l0, m_Del_Kd_yxy_o0_t0_l0, Del_Kd_yxy, BWeyl_xy, BWeyl_zy, BWeyl_yy, m_Rpd_yy_o0_t22_l0, m_Rpd_yy_o0_t21_l0, m_Rpd_yy_o0_t20_l0, m_Rpd_yy_o0_t19_l0, m_Rpd_yy_o0_t18_l0, m_Rpd_yy_o0_t17_l0, m_Rpd_yy_o0_t16_l0, m_Rpd_yy_o0_t15_l0, m_Rpd_yy_o0_t14_l0, m_Rpd_yy_o0_t13_l0, m_Rpd_yy_o0_t12_l0, m_Rpd_yy_o0_t11_l0, m_Rpd_yy_o0_t10_l0, m_Rpd_yy_o0_t9_l0, m_Rpd_yy_o0_t8_l0, m_Rpd_yy_o0_t7_l0, m_Rpd_yy_o0_t6_l0, m_Rpd_yy_o0_t5_l0, m_Rpd_yy_o0_t4_l0, m_Rpd_yy_o0_t3_l0, m_Rpd_yy_o0_t2_l0, m_Rpd_yy_o0_t1_l0, m_Rpd_yy_o0_t0_l0, Rpd_yy, d_Rtd_xy_o0_t12_m0_l0, m_Rtd_xy_o0_t11_l0, m_Rtd_xy_o0_t10_l0, m_Rtd_xy_o0_t9_l0, m_Rtd_xy_o0_t8_l0, m_Rtd_xy_o0_t7_l0, m_Rtd_xy_o0_t6_l0, m_Rtd_xy_o0_t5_l0, m_Rtd_xy_o0_t4_l0, m_Rtd_xy_o0_t3_l0, m_Rtd_xy_o0_t2_l0, m_Rtd_xy_o0_t1_l0, m_Rtd_xy_o0_t0_l0, Rtd_xy, EWeyl_xy, d_Rtd_yy_o0_t9_m0_l0, m_Rtd_yy_o0_t8_l0, m_Rtd_yy_o0_t7_l0, m_Rtd_yy_o0_t6_l0, m_Rtd_yy_o0_t5_l0, m_Rtd_yy_o0_t4_l0, m_Rtd_yy_o0_t3_l0, m_Rtd_yy_o0_t2_l0, m_Rtd_yy_o0_t1_l0, m_Rtd_yy_o0_t0_l0, Rtd_yy, EWeyl_yy, d_Rtd_yz_o0_t12_m0_l0, m_Rtd_yz_o0_t11_l0, m_Rtd_yz_o0_t10_l0, m_Rtd_yz_o0_t9_l0, m_Rtd_yz_o0_t8_l0, m_Rtd_yz_o0_t7_l0, m_Rtd_yz_o0_t6_l0, m_Rtd_yz_o0_t5_l0, m_Rtd_yz_o0_t4_l0, m_Rtd_yz_o0_t3_l0, m_Rtd_yz_o0_t2_l0, m_Rtd_yz_o0_t1_l0, m_Rtd_yz_o0_t0_l0, Rtd_yz, EWeyl_yz, d_d_pheI4d_t_o0_t0_m0_l0, m_d_pheI4d_t_o0_t6_l0, m_d_pheI4d_t_o0_t5_l0, m_d_pheI4d_t_o0_t4_l0, m_d_pheI4d_t_o0_t3_l0, m_d_pheI4d_t_o0_t2_l0, m_d_pheI4d_t_o0_t1_l0, d_pheI4d_t, d_pheI4d_z, d_pheI4d_y, d_pheI4d_x, d_pheI4u_t, d_pheI4u_z, d_pheI4u_y, d_pheI4u_x, dphe4sq, Tsfu_tt, Tu_tt, Tsfu_tz, Tu_tz, Tsfu_ty, Tu_ty, Tsfu_tx, Tu_tx, Jtd_ADM_z, Jtd_ADM_y, Jtd_ADM_x, Tsfu_zz, Tu_zz, Tsfu_yz, Tu_yz, Tsfu_yy, Tu_yy, Tsfu_xz, Tu_xz, Tsfu_xy, Tu_xy, Tsfu_xx, Tu_xx, Td_yz, Td_yy, Td_xz, Td_xy, Td_xx, Td_zz, TT, Td_tx, Td_tz, Td_ty, Td_tt, rho_ADM, d_Rscalar_o0_t0_m0_l0, d_HamCon_o0_t0_m0_l0, d_MomCon_x_o0_t13_m0_l0, d_MomCon_y_o0_t13_m0_l0, d_MomCon_z_o0_t13_m0_l0, d_trA_o0_t0_m0_l0, d_detgtm1_o0_t0_m0_l0, d_M_ADM_surf_o0_t9_m0_l0, d_Jz_ADM_surf_o0_t0_m0_l0, d_N_Noetheri_o0_t0_m0_l0, d_N_Noethere_o0_t0_m0_l0, d_M_Komar_o0_t0_m0_l0, d_Jz_Komar_o0_t0_m0_l0, d_psi4R_o0_t0_m0_l0, d_psi4I_o0_t0_m0_l0, d_Z_x_o0_t0_m0_l0, d_Z_y_o0_t0_m0_l0, d_Z_z_o0_t0_m0_l0, m_M_ADM_surf_o0_t8_l0, m_M_ADM_surf_o0_t7_l0, m_M_ADM_surf_o0_t6_l0, m_M_ADM_surf_o0_t5_l0, m_M_ADM_surf_o0_t4_l0, m_M_ADM_surf_o0_t3_l0, m_M_ADM_surf_o0_t2_l0, m_M_ADM_surf_o0_t1_l0, m_M_ADM_surf_o0_t0_l0, m_MomCon_z_o0_t12_l0, m_MomCon_z_o0_t11_l0, m_MomCon_z_o0_t10_l0, m_MomCon_z_o0_t9_l0, m_MomCon_z_o0_t8_l0, m_MomCon_z_o0_t7_l0, m_MomCon_z_o0_t6_l0, m_MomCon_z_o0_t5_l0, m_MomCon_z_o0_t4_l0, m_MomCon_z_o0_t3_l0, m_MomCon_z_o0_t2_l0, m_MomCon_z_o0_t1_l0, m_MomCon_z_o0_t0_l0, m_MomCon_y_o0_t12_l0, m_MomCon_y_o0_t11_l0, m_MomCon_y_o0_t10_l0, m_MomCon_y_o0_t9_l0, m_MomCon_y_o0_t8_l0, m_MomCon_y_o0_t7_l0, m_MomCon_y_o0_t6_l0, m_MomCon_y_o0_t5_l0, m_MomCon_y_o0_t4_l0, m_MomCon_y_o0_t3_l0, m_MomCon_y_o0_t2_l0, m_MomCon_y_o0_t1_l0, m_MomCon_y_o0_t0_l0, m_MomCon_x_o0_t12_l0, m_MomCon_x_o0_t11_l0, m_MomCon_x_o0_t10_l0, m_MomCon_x_o0_t9_l0, m_MomCon_x_o0_t8_l0, m_MomCon_x_o0_t7_l0, m_MomCon_x_o0_t6_l0, m_MomCon_x_o0_t5_l0, m_MomCon_x_o0_t4_l0, m_MomCon_x_o0_t3_l0, m_MomCon_x_o0_t2_l0, m_MomCon_x_o0_t1_l0, m_MomCon_x_o0_t0_l0;

						if ((i + 2 <= bx_hi.x && i - 2 >= bx_lo.x && j + 2 <= bx_hi.y && j - 2 >= bx_lo.y && k + 2 <= bx_hi.z && k - 2 >= bx_lo.z)) {
							d_Rpd_xx_o0_t0_m0_l0 = D1CDO4_i(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t1_m0_l0 = D1CDO4_j(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t2_m0_l0 = D1CDO4_k(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t3_m0_l0 = D2CDO4_i(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t4_m0_l0 = D1CDO4crossed_ij(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t5_m0_l0 = D1CDO4crossed_ik(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t6_m0_l0 = D2CDO4_j(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t7_m0_l0 = D1CDO4crossed_jk(chi, unp1, i, j, k, dx, dt_lev);
							d_Rpd_xx_o0_t8_m0_l0 = D2CDO4_k(chi, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xxx_o0_t0_m0_l0 = D1CDO4_i(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xxy_o0_t0_m0_l0 = D1CDO4_j(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xxz_o0_t0_m0_l0 = D1CDO4_k(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t0_m0_l0 = D2CDO4_i(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t3_m0_l0 = D2CDO4_j(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t5_m0_l0 = D2CDO4_k(gtd_xx, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t6_m0_l0 = D1CDO4_i(Gamh_x, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t6_m0_l0 = D1CDO4_j(Gamh_x, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t6_m0_l0 = D1CDO4_k(Gamh_x, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t7_m0_l0 = D1CDO4_i(Gamh_y, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t7_m0_l0 = D1CDO4_j(Gamh_y, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t7_m0_l0 = D1CDO4_k(Gamh_y, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xx_o0_t8_m0_l0 = D1CDO4_i(Gamh_z, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t8_m0_l0 = D1CDO4_j(Gamh_z, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t8_m0_l0 = D1CDO4_k(Gamh_z, unp1, i, j, k, dx, dt_lev);
							d_Ctd_yxx_o0_t0_m0_l0 = D1CDO4_i(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xyy_o0_t0_m0_l0 = D1CDO4_j(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xyz_o0_t1_m0_l0 = D1CDO4_k(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t0_m0_l0 = D2CDO4_i(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t3_m0_l0 = D2CDO4_j(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xy_o0_t5_m0_l0 = D2CDO4_k(gtd_xy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_zxx_o0_t0_m0_l0 = D1CDO4_i(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xyz_o0_t0_m0_l0 = D1CDO4_j(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xzz_o0_t0_m0_l0 = D1CDO4_k(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t0_m0_l0 = D2CDO4_i(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t3_m0_l0 = D2CDO4_j(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_xz_o0_t5_m0_l0 = D2CDO4_k(gtd_xz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xyy_o0_t1_m0_l0 = D1CDO4_i(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_yyy_o0_t0_m0_l0 = D1CDO4_j(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_yyz_o0_t0_m0_l0 = D1CDO4_k(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t0_m0_l0 = D2CDO4_i(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t3_m0_l0 = D2CDO4_j(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yy_o0_t5_m0_l0 = D2CDO4_k(gtd_yy, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xyz_o0_t2_m0_l0 = D1CDO4_i(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_zyy_o0_t0_m0_l0 = D1CDO4_j(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_yzz_o0_t0_m0_l0 = D1CDO4_k(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t0_m0_l0 = D2CDO4_i(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t3_m0_l0 = D2CDO4_j(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_yz_o0_t5_m0_l0 = D2CDO4_k(gtd_yz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_xzz_o0_t1_m0_l0 = D1CDO4_i(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_yzz_o0_t1_m0_l0 = D1CDO4_j(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Ctd_zzz_o0_t0_m0_l0 = D1CDO4_k(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t0_m0_l0 = D2CDO4_i(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t1_m0_l0 = D1CDO4crossed_ij(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t2_m0_l0 = D1CDO4crossed_ik(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t3_m0_l0 = D2CDO4_j(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t4_m0_l0 = D1CDO4crossed_jk(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Rtd_zz_o0_t5_m0_l0 = D2CDO4_k(gtd_zz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xxy_o0_t0_m0_l0 = D1CDO4_i(Atd_xy, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yxy_o0_t0_m0_l0 = D1CDO4_j(Atd_xy, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zxy_o0_t0_m0_l0 = D1CDO4_k(Atd_xy, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xxy_o0_t2_m0_l0 = D1CDO4_i(trK, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yxx_o0_t2_m0_l0 = D1CDO4_j(trK, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zxx_o0_t2_m0_l0 = D1CDO4_k(trK, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xxz_o0_t0_m0_l0 = D1CDO4_i(Atd_xz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yxz_o0_t0_m0_l0 = D1CDO4_j(Atd_xz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zxz_o0_t0_m0_l0 = D1CDO4_k(Atd_xz, unp1, i, j, k, dx, dt_lev);
							d_MomCon_x_o0_t3_m0_l0 = D1CDO4_i(Atd_xx, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yxx_o0_t0_m0_l0 = D1CDO4_j(Atd_xx, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zxx_o0_t0_m0_l0 = D1CDO4_k(Atd_xx, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xyy_o0_t0_m0_l0 = D1CDO4_i(Atd_yy, unp1, i, j, k, dx, dt_lev);
							d_MomCon_y_o0_t7_m0_l0 = D1CDO4_j(Atd_yy, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zyy_o0_t0_m0_l0 = D1CDO4_k(Atd_yy, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xyz_o0_t0_m0_l0 = D1CDO4_i(Atd_yz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yyz_o0_t0_m0_l0 = D1CDO4_j(Atd_yz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_zyz_o0_t0_m0_l0 = D1CDO4_k(Atd_yz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_xzz_o0_t0_m0_l0 = D1CDO4_i(Atd_zz, unp1, i, j, k, dx, dt_lev);
							d_Del_Kd_yzz_o0_t0_m0_l0 = D1CDO4_j(Atd_zz, unp1, i, j, k, dx, dt_lev);
							d_MomCon_z_o0_t11_m0_l0 = D1CDO4_k(Atd_zz, unp1, i, j, k, dx, dt_lev);
							d_d_phiR4d_t_o0_t1_m0_l0 = D1CDO4_i(phiR, unp1, i, j, k, dx, dt_lev);
							d_d_phiR4d_t_o0_t2_m0_l0 = D1CDO4_j(phiR, unp1, i, j, k, dx, dt_lev);
							d_d_phiR4d_t_o0_t3_m0_l0 = D1CDO4_k(phiR, unp1, i, j, k, dx, dt_lev);
							d_d_phiI4d_t_o0_t1_m0_l0 = D1CDO4_i(phiI, unp1, i, j, k, dx, dt_lev);
							d_d_phiI4d_t_o0_t2_m0_l0 = D1CDO4_j(phiI, unp1, i, j, k, dx, dt_lev);
							d_d_phiI4d_t_o0_t3_m0_l0 = D1CDO4_k(phiI, unp1, i, j, k, dx, dt_lev);
							d_d_pheR4d_t_o0_t1_m0_l0 = D1CDO4_i(pheR, unp1, i, j, k, dx, dt_lev);
							d_d_pheR4d_t_o0_t2_m0_l0 = D1CDO4_j(pheR, unp1, i, j, k, dx, dt_lev);
							d_d_pheR4d_t_o0_t3_m0_l0 = D1CDO4_k(pheR, unp1, i, j, k, dx, dt_lev);
							d_d_pheI4d_t_o0_t1_m0_l0 = D1CDO4_i(pheI, unp1, i, j, k, dx, dt_lev);
							d_d_pheI4d_t_o0_t2_m0_l0 = D1CDO4_j(pheI, unp1, i, j, k, dx, dt_lev);
							d_d_pheI4d_t_o0_t3_m0_l0 = D1CDO4_k(pheI, unp1, i, j, k, dx, dt_lev);
							sfmass = std::min(parameters.p_sfmass, parameters.p_sfmass * (pow((parameters.R_0 / std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k)))), parameters.eta_damping_exp)));
							inv_chi = 1.0 / fabs(unp1(i, j, k, chi));
							g4d_zz = inv_chi * unp1(i, j, k, gtd_zz);
							g4d_yz = inv_chi * unp1(i, j, k, gtd_yz);
							g4d_yy = inv_chi * unp1(i, j, k, gtd_yy);
							ur_z = zcoord(k);
							ur_y = ycoord(j);
							uph_z = 0.0;
							d_d_phiR4d_t_o0_t0_m0_l0 = -unp1(i, j, k, Alpha) * unp1(i, j, k, piR);
							m_d_phiR4d_t_o0_t6_l0 = std::min(0.0, unp1(i, j, k, Betau_z)) * d_d_phiR4d_t_o0_t3_m0_l0;
							m_d_phiR4d_t_o0_t5_l0 = std::min(0.0, unp1(i, j, k, Betau_y)) * d_d_phiR4d_t_o0_t2_m0_l0;
							m_d_phiR4d_t_o0_t4_l0 = std::min(0.0, unp1(i, j, k, Betau_x)) * d_d_phiR4d_t_o0_t1_m0_l0;
							m_d_phiR4d_t_o0_t3_l0 = std::max(0.0, unp1(i, j, k, Betau_z)) * d_d_phiR4d_t_o0_t3_m0_l0;
							m_d_phiR4d_t_o0_t2_l0 = std::max(0.0, unp1(i, j, k, Betau_y)) * d_d_phiR4d_t_o0_t2_m0_l0;
							m_d_phiR4d_t_o0_t1_l0 = std::max(0.0, unp1(i, j, k, Betau_x)) * d_d_phiR4d_t_o0_t1_m0_l0;
							d_phiR4d_t = (((((d_d_phiR4d_t_o0_t0_m0_l0 + m_d_phiR4d_t_o0_t1_l0) + m_d_phiR4d_t_o0_t2_l0) + m_d_phiR4d_t_o0_t3_l0) + m_d_phiR4d_t_o0_t4_l0) + m_d_phiR4d_t_o0_t5_l0) + m_d_phiR4d_t_o0_t6_l0;
							d_phiR4d_x = d_d_phiR4d_t_o0_t1_m0_l0;
							d_phiR4d_z = d_d_phiR4d_t_o0_t3_m0_l0;
							d_phiR4d_y = d_d_phiR4d_t_o0_t2_m0_l0;
							uph_x = -ycoord(j);
							uph_y = xcoord(i);
							ur_x = xcoord(i);
							d_d_pheR4d_t_o0_t0_m0_l0 = -unp1(i, j, k, Alpha) * unp1(i, j, k, peR);
							m_d_pheR4d_t_o0_t6_l0 = std::min(0.0, unp1(i, j, k, Betau_z)) * d_d_pheR4d_t_o0_t3_m0_l0;
							m_d_pheR4d_t_o0_t5_l0 = std::min(0.0, unp1(i, j, k, Betau_y)) * d_d_pheR4d_t_o0_t2_m0_l0;
							m_d_pheR4d_t_o0_t4_l0 = std::min(0.0, unp1(i, j, k, Betau_x)) * d_d_pheR4d_t_o0_t1_m0_l0;
							m_d_pheR4d_t_o0_t3_l0 = std::max(0.0, unp1(i, j, k, Betau_z)) * d_d_pheR4d_t_o0_t3_m0_l0;
							m_d_pheR4d_t_o0_t2_l0 = std::max(0.0, unp1(i, j, k, Betau_y)) * d_d_pheR4d_t_o0_t2_m0_l0;
							m_d_pheR4d_t_o0_t1_l0 = std::max(0.0, unp1(i, j, k, Betau_x)) * d_d_pheR4d_t_o0_t1_m0_l0;
							d_pheR4d_t = (((((d_d_pheR4d_t_o0_t0_m0_l0 + m_d_pheR4d_t_o0_t1_l0) + m_d_pheR4d_t_o0_t2_l0) + m_d_pheR4d_t_o0_t3_l0) + m_d_pheR4d_t_o0_t4_l0) + m_d_pheR4d_t_o0_t5_l0) + m_d_pheR4d_t_o0_t6_l0;
							d_pheR4d_x = d_d_pheR4d_t_o0_t1_m0_l0;
							d_pheR4d_y = d_d_pheR4d_t_o0_t2_m0_l0;
							d_pheR4d_z = d_d_pheR4d_t_o0_t3_m0_l0;
							uthd_y = ycoord(j) * zcoord(k);
							uthd_x = xcoord(i) * zcoord(k);
							uthd_z = (-xcoord(i) * xcoord(i)) + (-ycoord(j) * ycoord(j));
							m_Ctd_zzz_o0_t0_l0 = .5 * d_Ctd_zzz_o0_t0_m0_l0;
							Ctd_zzz = m_Ctd_zzz_o0_t0_l0;
							wrr = inv_chi * unp1(i, j, k, gtd_xx) * (ur_x * ur_x) + 2.0 * inv_chi * unp1(i, j, k, gtd_xy) * ur_x * ur_y + 2.0 * inv_chi * unp1(i, j, k, gtd_xz) * ur_x * ur_z + inv_chi * unp1(i, j, k, gtd_yy) * (ur_y * ur_y) + 2.0 * inv_chi * unp1(i, j, k, gtd_yz) * ur_y * ur_z + inv_chi * unp1(i, j, k, gtd_zz) * (ur_z * ur_z);
							m_Ctd_yyz_o0_t0_l0 = .5 * d_Ctd_yyz_o0_t0_m0_l0;
							Ctd_yyz = m_Ctd_yyz_o0_t0_l0;
							m_Ctd_zyy_o0_t1_l0 = (-.5) * d_Ctd_yyz_o0_t0_m0_l0;
							m_Ctd_zyy_o0_t0_l0 = 1.0 * d_Ctd_zyy_o0_t0_m0_l0;
							Ctd_zyy = m_Ctd_zyy_o0_t0_l0 + m_Ctd_zyy_o0_t1_l0;
							m_Ctd_yyy_o0_t0_l0 = .5 * d_Ctd_yyy_o0_t0_m0_l0;
							Ctd_yyy = m_Ctd_yyy_o0_t0_l0;
							m_Ctd_xxz_o0_t0_l0 = .5 * d_Ctd_xxz_o0_t0_m0_l0;
							Ctd_xxz = m_Ctd_xxz_o0_t0_l0;
							m_Ctd_zxx_o0_t1_l0 = (-.5) * d_Ctd_xxz_o0_t0_m0_l0;
							m_Ctd_zxx_o0_t0_l0 = 1.0 * d_Ctd_zxx_o0_t0_m0_l0;
							Ctd_zxx = m_Ctd_zxx_o0_t0_l0 + m_Ctd_zxx_o0_t1_l0;
							m_Ctd_xxx_o0_t0_l0 = .5 * d_Ctd_xxx_o0_t0_m0_l0;
							Ctd_xxx = m_Ctd_xxx_o0_t0_l0;
							m_Ctd_yzz_o0_t1_l0 = (-.5) * d_Ctd_yzz_o0_t1_m0_l0;
							m_Ctd_yzz_o0_t0_l0 = 1.0 * d_Ctd_yzz_o0_t0_m0_l0;
							Ctd_yzz = m_Ctd_yzz_o0_t0_l0 + m_Ctd_yzz_o0_t1_l0;
							m_Ctd_zyz_o0_t0_l0 = .5 * d_Ctd_yzz_o0_t1_m0_l0;
							Ctd_zyz = m_Ctd_zyz_o0_t0_l0;
							m_Ctd_xxy_o0_t0_l0 = .5 * d_Ctd_xxy_o0_t0_m0_l0;
							Ctd_xxy = m_Ctd_xxy_o0_t0_l0;
							m_Ctd_yxx_o0_t1_l0 = (-.5) * d_Ctd_xxy_o0_t0_m0_l0;
							m_Ctd_yxx_o0_t0_l0 = 1.0 * d_Ctd_yxx_o0_t0_m0_l0;
							Ctd_yxx = m_Ctd_yxx_o0_t0_l0 + m_Ctd_yxx_o0_t1_l0;
							m_Ctd_xyy_o0_t1_l0 = (-.5) * d_Ctd_xyy_o0_t1_m0_l0;
							m_Ctd_xyy_o0_t0_l0 = 1.0 * d_Ctd_xyy_o0_t0_m0_l0;
							Ctd_xyy = m_Ctd_xyy_o0_t0_l0 + m_Ctd_xyy_o0_t1_l0;
							m_Ctd_yxy_o0_t0_l0 = .5 * d_Ctd_xyy_o0_t1_m0_l0;
							Ctd_yxy = m_Ctd_yxy_o0_t0_l0;
							m_Ctd_xyz_o0_t2_l0 = (-.5) * d_Ctd_xyz_o0_t2_m0_l0;
							m_Ctd_xyz_o0_t1_l0 = .5 * d_Ctd_xyz_o0_t1_m0_l0;
							m_Ctd_xyz_o0_t0_l0 = .5 * d_Ctd_xyz_o0_t0_m0_l0;
							Ctd_xyz = (m_Ctd_xyz_o0_t0_l0 + m_Ctd_xyz_o0_t1_l0) + m_Ctd_xyz_o0_t2_l0;
							m_Ctd_yxz_o0_t2_l0 = (-.5) * d_Ctd_xyz_o0_t0_m0_l0;
							m_Ctd_yxz_o0_t0_l0 = .5 * d_Ctd_xyz_o0_t2_m0_l0;
							Ctd_yxz = (m_Ctd_yxz_o0_t0_l0 + m_Ctd_xyz_o0_t1_l0) + m_Ctd_yxz_o0_t2_l0;
							m_Ctd_zxy_o0_t2_l0 = (-.5) * d_Ctd_xyz_o0_t1_m0_l0;
							Ctd_zxy = (m_Ctd_yxz_o0_t0_l0 + m_Ctd_xyz_o0_t0_l0) + m_Ctd_zxy_o0_t2_l0;
							g4d_tt = (-unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)) + inv_chi * (unp1(i, j, k, gtd_xx) * (unp1(i, j, k, Betau_x) * unp1(i, j, k, Betau_x)) + 2.0 * unp1(i, j, k, gtd_xy) * unp1(i, j, k, Betau_y) * unp1(i, j, k, Betau_x) + 2.0 * unp1(i, j, k, gtd_xz) * unp1(i, j, k, Betau_z) * unp1(i, j, k, Betau_x) + unp1(i, j, k, gtd_yy) * (unp1(i, j, k, Betau_y) * unp1(i, j, k, Betau_y)) + 2.0 * unp1(i, j, k, gtd_yz) * unp1(i, j, k, Betau_z) * unp1(i, j, k, Betau_y) + unp1(i, j, k, gtd_zz) * (unp1(i, j, k, Betau_z) * unp1(i, j, k, Betau_z)));
							g4d_ty = inv_chi * (unp1(i, j, k, gtd_xy) * unp1(i, j, k, Betau_x) + unp1(i, j, k, gtd_yy) * unp1(i, j, k, Betau_y) + unp1(i, j, k, gtd_yz) * unp1(i, j, k, Betau_z));
							g4d_tx = inv_chi * (unp1(i, j, k, gtd_xx) * unp1(i, j, k, Betau_x) + unp1(i, j, k, gtd_xy) * unp1(i, j, k, Betau_y) + unp1(i, j, k, gtd_xz) * unp1(i, j, k, Betau_z));
							g4d_tz = inv_chi * (unp1(i, j, k, gtd_xz) * unp1(i, j, k, Betau_x) + unp1(i, j, k, gtd_yz) * unp1(i, j, k, Betau_y) + unp1(i, j, k, gtd_zz) * unp1(i, j, k, Betau_z));
							m_Ctd_xzz_o0_t1_l0 = (-.5) * d_Ctd_xzz_o0_t1_m0_l0;
							m_Ctd_xzz_o0_t0_l0 = 1.0 * d_Ctd_xzz_o0_t0_m0_l0;
							Ctd_xzz = m_Ctd_xzz_o0_t0_l0 + m_Ctd_xzz_o0_t1_l0;
							m_Ctd_zxz_o0_t0_l0 = .5 * d_Ctd_xzz_o0_t1_m0_l0;
							Ctd_zxz = m_Ctd_zxz_o0_t0_l0;
							dSigma_y = (1. * ur_y * std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k)))) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							dSigma_x = (1. * ur_x * std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k)))) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							dSigma_z = (1. * ur_z * std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k)))) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							d_d_phiI4d_t_o0_t0_m0_l0 = -unp1(i, j, k, Alpha) * unp1(i, j, k, piI);
							m_d_phiI4d_t_o0_t6_l0 = std::min(0.0, unp1(i, j, k, Betau_z)) * d_d_phiI4d_t_o0_t3_m0_l0;
							m_d_phiI4d_t_o0_t5_l0 = std::min(0.0, unp1(i, j, k, Betau_y)) * d_d_phiI4d_t_o0_t2_m0_l0;
							m_d_phiI4d_t_o0_t4_l0 = std::min(0.0, unp1(i, j, k, Betau_x)) * d_d_phiI4d_t_o0_t1_m0_l0;
							m_d_phiI4d_t_o0_t3_l0 = std::max(0.0, unp1(i, j, k, Betau_z)) * d_d_phiI4d_t_o0_t3_m0_l0;
							m_d_phiI4d_t_o0_t2_l0 = std::max(0.0, unp1(i, j, k, Betau_y)) * d_d_phiI4d_t_o0_t2_m0_l0;
							m_d_phiI4d_t_o0_t1_l0 = std::max(0.0, unp1(i, j, k, Betau_x)) * d_d_phiI4d_t_o0_t1_m0_l0;
							d_phiI4d_t = (((((d_d_phiI4d_t_o0_t0_m0_l0 + m_d_phiI4d_t_o0_t1_l0) + m_d_phiI4d_t_o0_t2_l0) + m_d_phiI4d_t_o0_t3_l0) + m_d_phiI4d_t_o0_t4_l0) + m_d_phiI4d_t_o0_t5_l0) + m_d_phiI4d_t_o0_t6_l0;
							d_phiI4d_x = d_d_phiI4d_t_o0_t1_m0_l0;
							d_phiI4d_y = d_d_phiI4d_t_o0_t2_m0_l0;
							d_phiI4d_z = d_d_phiI4d_t_o0_t3_m0_l0;
							g4u_tx = unp1(i, j, k, Betau_x) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha));
							g4u_tz = unp1(i, j, k, Betau_z) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha));
							g4u_ty = unp1(i, j, k, Betau_y) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha));
							g4u_tt = -1.0 / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha));
							d_phiR4u_t = g4u_tt * d_phiR4d_t + g4u_tx * d_phiR4d_x + g4u_ty * d_phiR4d_y + g4u_tz * d_phiR4d_z;
							d_phiI4u_t = g4u_tt * d_phiI4d_t + g4u_tx * d_phiI4d_x + g4u_ty * d_phiI4d_y + g4u_tz * d_phiI4d_z;
							d_pheR4u_t = g4u_tt * d_pheR4d_t + g4u_tx * d_pheR4d_x + g4u_ty * d_pheR4d_y + g4u_tz * d_pheR4d_z;
							sfVe = (sfmass * sfmass) * (unp1(i, j, k, pheR) * unp1(i, j, k, pheR) + unp1(i, j, k, pheI) * unp1(i, j, k, pheI)) * ((1.0 - 2.0 * (unp1(i, j, k, pheR) * unp1(i, j, k, pheR) + unp1(i, j, k, pheI) * unp1(i, j, k, pheI)) / (parameters.sfsigma * parameters.sfsigma)) * (1.0 - 2.0 * (unp1(i, j, k, pheR) * unp1(i, j, k, pheR) + unp1(i, j, k, pheI) * unp1(i, j, k, pheI)) / (parameters.sfsigma * parameters.sfsigma)));
							sfVi = (sfmass * sfmass) * (unp1(i, j, k, phiR) * unp1(i, j, k, phiR) + unp1(i, j, k, phiI) * unp1(i, j, k, phiI)) * ((1.0 - 2.0 * (unp1(i, j, k, phiR) * unp1(i, j, k, phiR) + unp1(i, j, k, phiI) * unp1(i, j, k, phiI)) / (parameters.sfsigma * parameters.sfsigma)) * (1.0 - 2.0 * (unp1(i, j, k, phiR) * unp1(i, j, k, phiR) + unp1(i, j, k, phiI) * unp1(i, j, k, phiI)) / (parameters.sfsigma * parameters.sfsigma)));
							detgtd = unp1(i, j, k, gtd_yy) * unp1(i, j, k, gtd_zz) * unp1(i, j, k, gtd_xx) + (-unp1(i, j, k, gtd_yy) * (unp1(i, j, k, gtd_xz) * unp1(i, j, k, gtd_xz))) + (-unp1(i, j, k, gtd_zz) * (unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_xy))) + (-(unp1(i, j, k, gtd_yz) * unp1(i, j, k, gtd_yz)) * unp1(i, j, k, gtd_xx)) + 2.0 * unp1(i, j, k, gtd_yz) * unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_xz);
							idetgtd = 1.0 / detgtd;
							gtu_zz = idetgtd * (unp1(i, j, k, gtd_xx) * unp1(i, j, k, gtd_yy) - unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_xy));
							g4u_zz = unp1(i, j, k, chi) * gtu_zz + (-(unp1(i, j, k, Betau_z) * unp1(i, j, k, Betau_z)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							gtu_yy = idetgtd * (unp1(i, j, k, gtd_xx) * unp1(i, j, k, gtd_zz) - unp1(i, j, k, gtd_xz) * unp1(i, j, k, gtd_xz));
							g4u_yy = unp1(i, j, k, chi) * gtu_yy + (-(unp1(i, j, k, Betau_y) * unp1(i, j, k, Betau_y)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							gtu_yz = idetgtd * ((-unp1(i, j, k, gtd_xx) * unp1(i, j, k, gtd_yz)) + unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_xz));
							g4u_yz = unp1(i, j, k, chi) * gtu_yz + (-(unp1(i, j, k, Betau_y) * unp1(i, j, k, Betau_z)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							gtu_xx = idetgtd * (unp1(i, j, k, gtd_yy) * unp1(i, j, k, gtd_zz) - unp1(i, j, k, gtd_yz) * unp1(i, j, k, gtd_yz));
							g4u_xx = unp1(i, j, k, chi) * gtu_xx + (-(unp1(i, j, k, Betau_x) * unp1(i, j, k, Betau_x)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							gtu_xz = idetgtd * (unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_yz) - unp1(i, j, k, gtd_yy) * unp1(i, j, k, gtd_xz));
							g4u_xz = unp1(i, j, k, chi) * gtu_xz + (-(unp1(i, j, k, Betau_x) * unp1(i, j, k, Betau_z)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							d_phiR4u_z = g4u_tz * d_phiR4d_t + g4u_xz * d_phiR4d_x + g4u_yz * d_phiR4d_y + g4u_zz * d_phiR4d_z;
							d_phiI4u_z = g4u_tz * d_phiI4d_t + g4u_xz * d_phiI4d_x + g4u_yz * d_phiI4d_y + g4u_zz * d_phiI4d_z;
							d_pheR4u_z = g4u_tz * d_pheR4d_t + g4u_xz * d_pheR4d_x + g4u_yz * d_pheR4d_y + g4u_zz * d_pheR4d_z;
							uth_z = unp1(i, j, k, chi) * gtu_xz * uthd_x + unp1(i, j, k, chi) * gtu_yz * uthd_y + unp1(i, j, k, chi) * gtu_zz * uthd_z;
							Atud_zx = gtu_xz * unp1(i, j, k, Atd_xx) + gtu_yz * unp1(i, j, k, Atd_xy) + gtu_zz * unp1(i, j, k, Atd_xz);
							Atud_zz = gtu_xz * unp1(i, j, k, Atd_xz) + gtu_yz * unp1(i, j, k, Atd_yz) + gtu_zz * unp1(i, j, k, Atd_zz);
							Atud_zy = gtu_xz * unp1(i, j, k, Atd_xy) + gtu_yz * unp1(i, j, k, Atd_yy) + gtu_zz * unp1(i, j, k, Atd_yz);
							Atu_zz = Atud_zx * gtu_xz + Atud_zy * gtu_yz + Atud_zz * gtu_zz;
							Ct_zxz = gtu_xz * Ctd_xxz + gtu_yz * Ctd_yxz + gtu_zz * Ctd_zxz;
							Ct_zxy = gtu_xz * Ctd_xxy + gtu_yz * Ctd_yxy + gtu_zz * Ctd_zxy;
							Ct_zxx = gtu_xz * Ctd_xxx + gtu_yz * Ctd_yxx + gtu_zz * Ctd_zxx;
							Ct_zyz = gtu_xz * Ctd_xyz + gtu_yz * Ctd_yyz + gtu_zz * Ctd_zyz;
							Ct_zyy = gtu_xz * Ctd_xyy + gtu_yz * Ctd_yyy + gtu_zz * Ctd_zyy;
							Ct_zzz = gtu_xz * Ctd_xzz + gtu_yz * Ctd_yzz + gtu_zz * Ctd_zzz;
							gtu_xy = idetgtd * ((-unp1(i, j, k, gtd_xy) * unp1(i, j, k, gtd_zz)) + unp1(i, j, k, gtd_yz) * unp1(i, j, k, gtd_xz));
							g4u_xy = unp1(i, j, k, chi) * gtu_xy + (-(unp1(i, j, k, Betau_x) * unp1(i, j, k, Betau_y)) / (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)));
							d_phiR4u_y = g4u_ty * d_phiR4d_t + g4u_xy * d_phiR4d_x + g4u_yy * d_phiR4d_y + g4u_yz * d_phiR4d_z;
							d_phiR4u_x = g4u_tx * d_phiR4d_t + g4u_xx * d_phiR4d_x + g4u_xy * d_phiR4d_y + g4u_xz * d_phiR4d_z;
							d_phiI4u_y = g4u_ty * d_phiI4d_t + g4u_xy * d_phiI4d_x + g4u_yy * d_phiI4d_y + g4u_yz * d_phiI4d_z;
							d_phiI4u_x = g4u_tx * d_phiI4d_t + g4u_xx * d_phiI4d_x + g4u_xy * d_phiI4d_y + g4u_xz * d_phiI4d_z;
							dphi4sq = d_phiR4u_t * d_phiR4d_t + d_phiR4u_x * d_phiR4d_x + d_phiR4u_y * d_phiR4d_y + d_phiR4u_z * d_phiR4d_z + d_phiI4u_t * d_phiI4d_t + d_phiI4u_x * d_phiI4d_x + d_phiI4u_y * d_phiI4d_y + d_phiI4u_z * d_phiI4d_z;
							d_pheR4u_x = g4u_tx * d_pheR4d_t + g4u_xx * d_pheR4d_x + g4u_xy * d_pheR4d_y + g4u_xz * d_pheR4d_z;
							d_pheR4u_y = g4u_ty * d_pheR4d_t + g4u_xy * d_pheR4d_x + g4u_yy * d_pheR4d_y + g4u_yz * d_pheR4d_z;
							Ct_xzz = gtu_xx * Ctd_xzz + gtu_xy * Ctd_yzz + gtu_xz * Ctd_zzz;
							uth_x = unp1(i, j, k, chi) * gtu_xx * uthd_x + unp1(i, j, k, chi) * gtu_xy * uthd_y + unp1(i, j, k, chi) * gtu_xz * uthd_z;
							uth_y = unp1(i, j, k, chi) * gtu_xy * uthd_x + unp1(i, j, k, chi) * gtu_yy * uthd_y + unp1(i, j, k, chi) * gtu_yz * uthd_z;
							Gamt_z = gtu_xx * Ct_zxx + 2.0 * gtu_xy * Ct_zxy + 2.0 * gtu_xz * Ct_zxz + gtu_yy * Ct_zyy + 2.0 * gtu_yz * Ct_zyz + gtu_zz * Ct_zzz;
							Zu_z = .5 * unp1(i, j, k, chi) * (unp1(i, j, k, Gamh_z) - Gamt_z);
							Atud_yy = gtu_xy * unp1(i, j, k, Atd_xy) + gtu_yy * unp1(i, j, k, Atd_yy) + gtu_yz * unp1(i, j, k, Atd_yz);
							Atud_yx = gtu_xy * unp1(i, j, k, Atd_xx) + gtu_yy * unp1(i, j, k, Atd_xy) + gtu_yz * unp1(i, j, k, Atd_xz);
							Atud_yz = gtu_xy * unp1(i, j, k, Atd_xz) + gtu_yy * unp1(i, j, k, Atd_yz) + gtu_yz * unp1(i, j, k, Atd_zz);
							Atu_yz = Atud_yx * gtu_xz + Atud_yy * gtu_yz + Atud_yz * gtu_zz;
							Atud_xx = gtu_xx * unp1(i, j, k, Atd_xx) + gtu_xy * unp1(i, j, k, Atd_xy) + gtu_xz * unp1(i, j, k, Atd_xz);
							Atud_xz = gtu_xx * unp1(i, j, k, Atd_xz) + gtu_xy * unp1(i, j, k, Atd_yz) + gtu_xz * unp1(i, j, k, Atd_zz);
							Atud_xy = gtu_xx * unp1(i, j, k, Atd_xy) + gtu_xy * unp1(i, j, k, Atd_yy) + gtu_xz * unp1(i, j, k, Atd_yz);
							Atu_xz = Atud_xx * gtu_xz + Atud_xy * gtu_yz + Atud_xz * gtu_zz;
							Atu_yy = Atud_yx * gtu_xy + Atud_yy * gtu_yy + Atud_yz * gtu_yz;
							Ct_yxx = gtu_xy * Ctd_xxx + gtu_yy * Ctd_yxx + gtu_yz * Ctd_zxx;
							Ct_yxz = gtu_xy * Ctd_xxz + gtu_yy * Ctd_yxz + gtu_yz * Ctd_zxz;
							Ct_yxy = gtu_xy * Ctd_xxy + gtu_yy * Ctd_yxy + gtu_yz * Ctd_zxy;
							Atu_xy = Atud_xx * gtu_xy + Atud_xy * gtu_yy + Atud_xz * gtu_yz;
							Atu_xx = Atud_xx * gtu_xx + Atud_xy * gtu_xy + Atud_xz * gtu_xz;
							Ct_yyy = gtu_xy * Ctd_xyy + gtu_yy * Ctd_yyy + gtu_yz * Ctd_zyy;
							Ct_yyz = gtu_xy * Ctd_xyz + gtu_yy * Ctd_yyz + gtu_yz * Ctd_zyz;
							Ct_yzz = gtu_xy * Ctd_xzz + gtu_yy * Ctd_yzz + gtu_yz * Ctd_zzz;
							Gamt_y = gtu_xx * Ct_yxx + 2.0 * gtu_xy * Ct_yxy + 2.0 * gtu_xz * Ct_yxz + gtu_yy * Ct_yyy + 2.0 * gtu_yz * Ct_yyz + gtu_zz * Ct_yzz;
							Zu_y = .5 * unp1(i, j, k, chi) * (unp1(i, j, k, Gamh_y) - Gamt_y);
							Ct_xxy = gtu_xx * Ctd_xxy + gtu_xy * Ctd_yxy + gtu_xz * Ctd_zxy;
							Ct_xxx = gtu_xx * Ctd_xxx + gtu_xy * Ctd_yxx + gtu_xz * Ctd_zxx;
							Ct_xxz = gtu_xx * Ctd_xxz + gtu_xy * Ctd_yxz + gtu_xz * Ctd_zxz;
							d_Rtd_xx_o0_t9_m0_l0 = gtu_zz * (3.0 * Ct_xxz * Ctd_xxz + 2.0 * Ct_yxz * Ctd_xyz + Ct_yxz * Ctd_yxz + 2.0 * Ct_zxz * Ctd_xzz + Ct_zxz * Ctd_zxz) + gtu_yz * (3.0 * Ct_xxy * Ctd_xxz + 2.0 * Ct_yxy * Ctd_xyz + Ct_yxy * Ctd_yxz + 2.0 * Ct_zxy * Ctd_xzz + Ct_zxy * Ctd_zxz) + gtu_xz * (3.0 * Ct_xxx * Ctd_xxz + 2.0 * Ct_yxx * Ctd_xyz + Ct_yxx * Ctd_yxz + 2.0 * Ct_zxx * Ctd_xzz + Ct_zxx * Ctd_zxz) + gtu_yz * (3.0 * Ct_xxz * Ctd_xxy + 2.0 * Ct_yxz * Ctd_xyy + Ct_yxz * Ctd_yxy + 2.0 * Ct_zxz * Ctd_xyz + Ct_zxz * Ctd_zxy) + gtu_yy * (3.0 * Ct_xxy * Ctd_xxy + 2.0 * Ct_yxy * Ctd_xyy + Ct_yxy * Ctd_yxy + 2.0 * Ct_zxy * Ctd_xyz + Ct_zxy * Ctd_zxy) + gtu_xy * (3.0 * Ct_xxx * Ctd_xxy + 2.0 * Ct_yxx * Ctd_xyy + Ct_yxx * Ctd_yxy + 2.0 * Ct_zxx * Ctd_xyz + Ct_zxx * Ctd_zxy) + gtu_xz * (3.0 * Ct_xxz * Ctd_xxx + 2.0 * Ct_yxz * Ctd_xxy + Ct_yxz * Ctd_yxx + 2.0 * Ct_zxz * Ctd_xxz + Ct_zxz * Ctd_zxx) + gtu_xy * (3.0 * Ct_xxy * Ctd_xxx + 2.0 * Ct_yxy * Ctd_xxy + Ct_yxy * Ctd_yxx + 2.0 * Ct_zxy * Ctd_xxz + Ct_zxy * Ctd_zxx) + gtu_xx * (3.0 * Ct_xxx * Ctd_xxx + 2.0 * Ct_yxx * Ctd_xxy + Ct_yxx * Ctd_yxx + 2.0 * Ct_zxx * Ctd_xxz + Ct_zxx * Ctd_zxx) + 1.0 * unp1(i, j, k, Gamh_z) * Ctd_xxz + 1.0 * unp1(i, j, k, Gamh_y) * Ctd_xxy + 1.0 * unp1(i, j, k, Gamh_x) * Ctd_xxx;
							m_Rtd_xx_o0_t8_l0 = 1.0 * unp1(i, j, k, gtd_xz) * d_Rtd_xx_o0_t8_m0_l0;
							m_Rtd_xx_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_xy) * d_Rtd_xx_o0_t7_m0_l0;
							m_Rtd_xx_o0_t6_l0 = 1.0 * unp1(i, j, k, gtd_xx) * d_Rtd_xx_o0_t6_m0_l0;
							m_Rtd_xx_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_xx_o0_t5_m0_l0;
							m_Rtd_xx_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_xx_o0_t4_m0_l0;
							m_Rtd_xx_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_xx_o0_t3_m0_l0;
							m_Rtd_xx_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_xx_o0_t2_m0_l0;
							m_Rtd_xx_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_xx_o0_t1_m0_l0;
							m_Rtd_xx_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_xx_o0_t0_m0_l0;
							Rtd_xx = ((((((((m_Rtd_xx_o0_t0_l0 + m_Rtd_xx_o0_t1_l0) + m_Rtd_xx_o0_t2_l0) + m_Rtd_xx_o0_t3_l0) + m_Rtd_xx_o0_t4_l0) + m_Rtd_xx_o0_t5_l0) + m_Rtd_xx_o0_t6_l0) + m_Rtd_xx_o0_t7_l0) + m_Rtd_xx_o0_t8_l0) + d_Rtd_xx_o0_t9_m0_l0;
							g4d_xy = inv_chi * unp1(i, j, k, gtd_xy);
							wphph = inv_chi * unp1(i, j, k, gtd_xx) * (uph_x * uph_x) + 2.0 * inv_chi * unp1(i, j, k, gtd_xy) * uph_x * uph_y + 2.0 * inv_chi * unp1(i, j, k, gtd_xz) * uph_x * uph_z + inv_chi * unp1(i, j, k, gtd_yy) * (uph_y * uph_y) + 2.0 * inv_chi * unp1(i, j, k, gtd_yz) * uph_y * uph_z + inv_chi * unp1(i, j, k, gtd_zz) * (uph_z * uph_z);
							vph_y = uph_y / std::max(.10e-3, sqrt(wphph));
							vph_z = uph_z / std::max(.10e-3, sqrt(wphph));
							vph_x = uph_x / std::max(.10e-3, sqrt(wphph));
							wthph = inv_chi * unp1(i, j, k, gtd_xx) * uth_x * vph_x + inv_chi * unp1(i, j, k, gtd_xy) * uth_x * vph_y + inv_chi * unp1(i, j, k, gtd_xz) * uth_x * vph_z + inv_chi * unp1(i, j, k, gtd_xy) * uth_y * vph_x + inv_chi * unp1(i, j, k, gtd_yy) * uth_y * vph_y + inv_chi * unp1(i, j, k, gtd_yz) * uth_y * vph_z + inv_chi * unp1(i, j, k, gtd_xz) * uth_z * vph_x + inv_chi * unp1(i, j, k, gtd_yz) * uth_z * vph_y + inv_chi * unp1(i, j, k, gtd_zz) * uth_z * vph_z;
							g4d_xx = inv_chi * unp1(i, j, k, gtd_xx);
							wrph = inv_chi * unp1(i, j, k, gtd_xx) * ur_x * vph_x + inv_chi * unp1(i, j, k, gtd_xy) * ur_x * vph_y + inv_chi * unp1(i, j, k, gtd_xz) * ur_x * vph_z + inv_chi * unp1(i, j, k, gtd_xy) * ur_y * vph_x + inv_chi * unp1(i, j, k, gtd_yy) * ur_y * vph_y + inv_chi * unp1(i, j, k, gtd_yz) * ur_y * vph_z + inv_chi * unp1(i, j, k, gtd_xz) * ur_z * vph_x + inv_chi * unp1(i, j, k, gtd_yz) * ur_z * vph_y + inv_chi * unp1(i, j, k, gtd_zz) * ur_z * vph_z;
							vr_x = ((-wrph * vph_x) + ur_x) / std::max(.10e-3, sqrt(wrr));
							vr_y = ((-wrph * vph_y) + ur_y) / std::max(.10e-3, sqrt(wrr));
							vr_z = ((-wrph * vph_z) + ur_z) / std::max(.10e-3, sqrt(wrr));
							wthr = inv_chi * unp1(i, j, k, gtd_xx) * uth_x * vr_x + inv_chi * unp1(i, j, k, gtd_xy) * uth_x * vr_y + inv_chi * unp1(i, j, k, gtd_xz) * uth_x * vr_z + inv_chi * unp1(i, j, k, gtd_xy) * uth_y * vr_x + inv_chi * unp1(i, j, k, gtd_yy) * uth_y * vr_y + inv_chi * unp1(i, j, k, gtd_yz) * uth_y * vr_z + inv_chi * unp1(i, j, k, gtd_xz) * uth_z * vr_x + inv_chi * unp1(i, j, k, gtd_yz) * uth_z * vr_y + inv_chi * unp1(i, j, k, gtd_zz) * uth_z * vr_z;
							g4d_xz = inv_chi * unp1(i, j, k, gtd_xz);
							wthth = inv_chi * unp1(i, j, k, gtd_xx) * (uth_x * uth_x) + 2.0 * inv_chi * unp1(i, j, k, gtd_xy) * uth_x * uth_y + 2.0 * inv_chi * unp1(i, j, k, gtd_xz) * uth_x * uth_z + inv_chi * unp1(i, j, k, gtd_yy) * (uth_y * uth_y) + 2.0 * inv_chi * unp1(i, j, k, gtd_yz) * uth_y * uth_z + inv_chi * unp1(i, j, k, gtd_zz) * (uth_z * uth_z);
							vth_x = ((-wthph * vph_x) + (-wthr * vr_x) + uth_x) / std::max(.10e-3, sqrt(wthth));
							mmR_xx = (-.5 * (vph_x * vph_x)) + .5 * (vth_x * vth_x);
							mmI_xx = -1.0 * vph_x * vth_x;
							vth_y = ((-wthph * vph_y) + (-wthr * vr_y) + uth_y) / std::max(.10e-3, sqrt(wthth));
							mmR_yy = (-.5 * (vph_y * vph_y)) + .5 * (vth_y * vth_y);
							mmR_xy = (-.5 * vph_x * vph_y) + .5 * vth_x * vth_y;
							mmI_xy = (-.5 * vph_x * vth_y) + (-.5 * vph_y * vth_x);
							mmI_yy = -1.0 * vph_y * vth_y;
							vth_z = ((-wthph * vph_z) + (-wthr * vr_z) + uth_z) / std::max(.10e-3, sqrt(wthth));
							mmR_yz = (-.5 * vph_y * vph_z) + .5 * vth_y * vth_z;
							mmR_zz = (-0.5 * (vph_z * vph_z)) + 0.5 * (vth_z * vth_z);
							mmR_xz = (-.5 * vph_x * vph_z) + .5 * vth_x * vth_z;
							mmI_zz = -1.0 * vph_z * vth_z;
							mmI_xz = (-.5 * vph_x * vth_z) + (-.5 * vph_z * vth_x);
							mmI_yz = (-.5 * vph_y * vth_z) + (-.5 * vph_z * vth_y);
							Ct_xyz = gtu_xx * Ctd_xyz + gtu_xy * Ctd_yyz + gtu_xz * Ctd_zyz;
							d_Rtd_xz_o0_t12_m0_l0 = gtu_zz * (Ct_xxz * Ctd_zxz + Ct_xxz * Ctd_xzz + Ct_xzz * Ctd_xxz + Ct_yxz * Ctd_zyz + Ct_yxz * Ctd_yzz + Ct_yzz * Ctd_xyz + 2.0 * Ct_zxz * Ctd_zzz + Ct_zzz * Ctd_xzz) + gtu_yz * (Ct_xxy * Ctd_zxz + Ct_xxy * Ctd_xzz + Ct_xyz * Ctd_xxz + Ct_yxy * Ctd_zyz + Ct_yxy * Ctd_yzz + Ct_yyz * Ctd_xyz + 2.0 * Ct_zxy * Ctd_zzz + Ct_zyz * Ctd_xzz) + gtu_xz * (Ct_xxx * Ctd_zxz + Ct_xxx * Ctd_xzz + Ct_xxz * Ctd_xxz + Ct_yxx * Ctd_zyz + Ct_yxx * Ctd_yzz + Ct_yxz * Ctd_xyz + 2.0 * Ct_zxx * Ctd_zzz + Ct_zxz * Ctd_xzz) + gtu_yz * (Ct_xxz * Ctd_zxy + Ct_xxz * Ctd_xyz + Ct_xzz * Ctd_xxy + Ct_yxz * Ctd_zyy + Ct_yxz * Ctd_yyz + Ct_yzz * Ctd_xyy + 2.0 * Ct_zxz * Ctd_zyz + Ct_zzz * Ctd_xyz) + gtu_yy * (Ct_xxy * Ctd_zxy + Ct_xxy * Ctd_xyz + Ct_xyz * Ctd_xxy + Ct_yxy * Ctd_zyy + Ct_yxy * Ctd_yyz + Ct_yyz * Ctd_xyy + 2.0 * Ct_zxy * Ctd_zyz + Ct_zyz * Ctd_xyz) + gtu_xy * (Ct_xxx * Ctd_zxy + Ct_xxx * Ctd_xyz + Ct_xxz * Ctd_xxy + Ct_yxx * Ctd_zyy + Ct_yxx * Ctd_yyz + Ct_yxz * Ctd_xyy + 2.0 * Ct_zxx * Ctd_zyz + Ct_zxz * Ctd_xyz) + gtu_xz * (Ct_xxz * Ctd_zxx + Ct_xxz * Ctd_xxz + Ct_xzz * Ctd_xxx + Ct_yxz * Ctd_zxy + Ct_yxz * Ctd_yxz + Ct_yzz * Ctd_xxy + 2.0 * Ct_zxz * Ctd_zxz + Ct_zzz * Ctd_xxz) + gtu_xy * (Ct_xxy * Ctd_zxx + Ct_xxy * Ctd_xxz + Ct_xyz * Ctd_xxx + Ct_yxy * Ctd_zxy + Ct_yxy * Ctd_yxz + Ct_yyz * Ctd_xxy + 2.0 * Ct_zxy * Ctd_zxz + Ct_zyz * Ctd_xxz) + gtu_xx * (Ct_xxx * Ctd_zxx + Ct_xxx * Ctd_xxz + Ct_xxz * Ctd_xxx + Ct_yxx * Ctd_zxy + Ct_yxx * Ctd_yxz + Ct_yxz * Ctd_xxy + 2.0 * Ct_zxx * Ctd_zxz + Ct_zxz * Ctd_xxz) + .5 * unp1(i, j, k, Gamh_z) * (Ctd_xzz + Ctd_zxz) + .5 * unp1(i, j, k, Gamh_y) * (Ctd_xyz + Ctd_zxy) + .5 * unp1(i, j, k, Gamh_x) * (Ctd_xxz + Ctd_zxx);
							m_Rtd_xz_o0_t11_l0 = .5 * unp1(i, j, k, gtd_zz) * d_Rtd_xx_o0_t8_m0_l0;
							m_Rtd_xz_o0_t10_l0 = .5 * unp1(i, j, k, gtd_yz) * d_Rtd_xx_o0_t7_m0_l0;
							m_Rtd_xz_o0_t9_l0 = .5 * unp1(i, j, k, gtd_xz) * d_Rtd_xx_o0_t6_m0_l0;
							m_Rtd_xz_o0_t8_l0 = .5 * unp1(i, j, k, gtd_xz) * d_Rtd_xz_o0_t8_m0_l0;
							m_Rtd_xz_o0_t7_l0 = .5 * unp1(i, j, k, gtd_xy) * d_Rtd_xz_o0_t7_m0_l0;
							m_Rtd_xz_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xx) * d_Rtd_xz_o0_t6_m0_l0;
							m_Rtd_xz_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_xz_o0_t5_m0_l0;
							m_Rtd_xz_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_xz_o0_t4_m0_l0;
							m_Rtd_xz_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_xz_o0_t3_m0_l0;
							m_Rtd_xz_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_xz_o0_t2_m0_l0;
							m_Rtd_xz_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_xz_o0_t1_m0_l0;
							m_Rtd_xz_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_xz_o0_t0_m0_l0;
							Rtd_xz = (((((((((((m_Rtd_xz_o0_t0_l0 + m_Rtd_xz_o0_t1_l0) + m_Rtd_xz_o0_t2_l0) + m_Rtd_xz_o0_t3_l0) + m_Rtd_xz_o0_t4_l0) + m_Rtd_xz_o0_t5_l0) + m_Rtd_xz_o0_t6_l0) + m_Rtd_xz_o0_t7_l0) + m_Rtd_xz_o0_t8_l0) + m_Rtd_xz_o0_t9_l0) + m_Rtd_xz_o0_t10_l0) + m_Rtd_xz_o0_t11_l0) + d_Rtd_xz_o0_t12_m0_l0;
							d_Rtd_zz_o0_t9_m0_l0 = gtu_zz * (2.0 * Ct_xzz * Ctd_zxz + Ct_xzz * Ctd_xzz + 2.0 * Ct_yzz * Ctd_zyz + Ct_yzz * Ctd_yzz + 3.0 * Ct_zzz * Ctd_zzz) + gtu_yz * (2.0 * Ct_xyz * Ctd_zxz + Ct_xyz * Ctd_xzz + 2.0 * Ct_yyz * Ctd_zyz + Ct_yyz * Ctd_yzz + 3.0 * Ct_zyz * Ctd_zzz) + gtu_xz * (2.0 * Ct_xxz * Ctd_zxz + Ct_xxz * Ctd_xzz + 2.0 * Ct_yxz * Ctd_zyz + Ct_yxz * Ctd_yzz + 3.0 * Ct_zxz * Ctd_zzz) + gtu_yz * (2.0 * Ct_xzz * Ctd_zxy + Ct_xzz * Ctd_xyz + 2.0 * Ct_yzz * Ctd_zyy + Ct_yzz * Ctd_yyz + 3.0 * Ct_zzz * Ctd_zyz) + gtu_yy * (2.0 * Ct_xyz * Ctd_zxy + Ct_xyz * Ctd_xyz + 2.0 * Ct_yyz * Ctd_zyy + Ct_yyz * Ctd_yyz + 3.0 * Ct_zyz * Ctd_zyz) + gtu_xy * (2.0 * Ct_xxz * Ctd_zxy + Ct_xxz * Ctd_xyz + 2.0 * Ct_yxz * Ctd_zyy + Ct_yxz * Ctd_yyz + 3.0 * Ct_zxz * Ctd_zyz) + gtu_xz * (2.0 * Ct_xzz * Ctd_zxx + Ct_xzz * Ctd_xxz + 2.0 * Ct_yzz * Ctd_zxy + Ct_yzz * Ctd_yxz + 3.0 * Ct_zzz * Ctd_zxz) + gtu_xy * (2.0 * Ct_xyz * Ctd_zxx + Ct_xyz * Ctd_xxz + 2.0 * Ct_yyz * Ctd_zxy + Ct_yyz * Ctd_yxz + 3.0 * Ct_zyz * Ctd_zxz) + gtu_xx * (2.0 * Ct_xxz * Ctd_zxx + Ct_xxz * Ctd_xxz + 2.0 * Ct_yxz * Ctd_zxy + Ct_yxz * Ctd_yxz + 3.0 * Ct_zxz * Ctd_zxz) + 1.0 * unp1(i, j, k, Gamh_z) * Ctd_zzz + 1.0 * unp1(i, j, k, Gamh_y) * Ctd_zyz + 1.0 * unp1(i, j, k, Gamh_x) * Ctd_zxz;
							m_Rtd_zz_o0_t8_l0 = 1.0 * unp1(i, j, k, gtd_zz) * d_Rtd_xz_o0_t8_m0_l0;
							m_Rtd_zz_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_yz) * d_Rtd_xz_o0_t7_m0_l0;
							m_Rtd_zz_o0_t6_l0 = 1.0 * unp1(i, j, k, gtd_xz) * d_Rtd_xz_o0_t6_m0_l0;
							m_Rtd_zz_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_zz_o0_t5_m0_l0;
							m_Rtd_zz_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_zz_o0_t4_m0_l0;
							m_Rtd_zz_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_zz_o0_t3_m0_l0;
							m_Rtd_zz_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_zz_o0_t2_m0_l0;
							m_Rtd_zz_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_zz_o0_t1_m0_l0;
							m_Rtd_zz_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_zz_o0_t0_m0_l0;
							Rtd_zz = ((((((((m_Rtd_zz_o0_t0_l0 + m_Rtd_zz_o0_t1_l0) + m_Rtd_zz_o0_t2_l0) + m_Rtd_zz_o0_t3_l0) + m_Rtd_zz_o0_t4_l0) + m_Rtd_zz_o0_t5_l0) + m_Rtd_zz_o0_t6_l0) + m_Rtd_zz_o0_t7_l0) + m_Rtd_zz_o0_t8_l0) + d_Rtd_zz_o0_t9_m0_l0;
							Ct_xyy = gtu_xx * Ctd_xyy + gtu_xy * Ctd_yyy + gtu_xz * Ctd_zyy;
							Gamt_x = gtu_xx * Ct_xxx + 2.0 * gtu_xy * Ct_xxy + 2.0 * gtu_xz * Ct_xxz + gtu_yy * Ct_xyy + 2.0 * gtu_yz * Ct_xyz + gtu_zz * Ct_xzz;
							Zu_x = .5 * unp1(i, j, k, chi) * (unp1(i, j, k, Gamh_x) - Gamt_x);
							m_Rpd_xx_o0_t22_l0 = .5 * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_xx_o0_t21_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xx) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xx_o0_t20_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xx) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xx_o0_t19_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xx) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t18_l0 = 2.0 * inv_chi * Zu_z * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t17_l0 = 2.0 * inv_chi * Zu_y * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t16_l0 = 2.0 * inv_chi * Zu_x * unp1(i, j, k, gtd_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xx) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xx_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xx) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xx) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t12_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t11_l0 = (-.5 * Ct_zxx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xx_o0_t10_l0 = (-.5 * Ct_yxx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xx_o0_t9_l0 = (-.5 * Ct_xxx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xx_o0_t8_l0 = .5 * unp1(i, j, k, gtd_xx) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_xx_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_xx) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_xx_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xx) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_xx_o0_t5_l0 = 1.0 * unp1(i, j, k, gtd_xx) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_xx_o0_t4_l0 = 1.0 * unp1(i, j, k, gtd_xx) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_xx_o0_t3_l0 = .5 * unp1(i, j, k, gtd_xx) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_xx_o0_t2_l0 = (-.5 * unp1(i, j, k, gtd_xx) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xx_o0_t1_l0 = (-.5 * unp1(i, j, k, gtd_xx) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xx_o0_t0_l0 = (-.5 * unp1(i, j, k, gtd_xx) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_xx = (((((((((((((((((((((m_Rpd_xx_o0_t0_l0 + m_Rpd_xx_o0_t1_l0) + m_Rpd_xx_o0_t2_l0) + m_Rpd_xx_o0_t3_l0) + m_Rpd_xx_o0_t4_l0) + m_Rpd_xx_o0_t5_l0) + m_Rpd_xx_o0_t6_l0) + m_Rpd_xx_o0_t7_l0) + m_Rpd_xx_o0_t8_l0) + m_Rpd_xx_o0_t9_l0) + m_Rpd_xx_o0_t10_l0) + m_Rpd_xx_o0_t11_l0) + m_Rpd_xx_o0_t12_l0) + m_Rpd_xx_o0_t13_l0) + m_Rpd_xx_o0_t14_l0) + m_Rpd_xx_o0_t15_l0) + m_Rpd_xx_o0_t16_l0) + m_Rpd_xx_o0_t17_l0) + m_Rpd_xx_o0_t18_l0) + m_Rpd_xx_o0_t19_l0) + m_Rpd_xx_o0_t20_l0) + m_Rpd_xx_o0_t21_l0) + m_Rpd_xx_o0_t22_l0;
							d_Del_Kd_yxx_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xx) * Ct_xxy + 2.0 * unp1(i, j, k, Atd_xy) * Ct_yxy + 2.0 * unp1(i, j, k, Atd_xz) * Ct_zxy);
							m_Del_Kd_yxx_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxx_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxx_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxx_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxx_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xx) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_Del_Kd_yxx_o0_t0_l0 = inv_chi * d_Del_Kd_yxx_o0_t0_m0_l0;
							Del_Kd_yxx = (((((m_Del_Kd_yxx_o0_t0_l0 + d_Del_Kd_yxx_o0_t1_m0_l0) + m_Del_Kd_yxx_o0_t2_l0) + m_Del_Kd_yxx_o0_t3_l0) + m_Del_Kd_yxx_o0_t4_l0) + m_Del_Kd_yxx_o0_t5_l0) + m_Del_Kd_yxx_o0_t6_l0;
							d_Del_Kd_yxz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xxy + unp1(i, j, k, Atd_xx) * Ct_xyz + unp1(i, j, k, Atd_yz) * Ct_yxy + unp1(i, j, k, Atd_xy) * Ct_yyz + unp1(i, j, k, Atd_zz) * Ct_zxy + unp1(i, j, k, Atd_xz) * Ct_zyz);
							m_Del_Kd_yxz_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxz_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxz_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxz_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxz_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxz_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxz_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxz_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xz) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_Del_Kd_yxz_o0_t0_l0 = inv_chi * d_Del_Kd_yxz_o0_t0_m0_l0;
							Del_Kd_yxz = (((((((((m_Del_Kd_yxz_o0_t0_l0 + d_Del_Kd_yxz_o0_t1_m0_l0) + m_Del_Kd_yxz_o0_t2_l0) + m_Del_Kd_yxz_o0_t3_l0) + m_Del_Kd_yxz_o0_t4_l0) + m_Del_Kd_yxz_o0_t5_l0) + m_Del_Kd_yxz_o0_t6_l0) + m_Del_Kd_yxz_o0_t7_l0) + m_Del_Kd_yxz_o0_t8_l0) + m_Del_Kd_yxz_o0_t9_l0) + m_Del_Kd_yxz_o0_t10_l0;
							EWeyl_xx = (-inv_chi * Rpd_xx) + (-Rtd_xx) + inv_chi * (unp1(i, j, k, Atd_xx) * Atud_xx + unp1(i, j, k, Atd_xy) * Atud_yx + unp1(i, j, k, Atd_xz) * Atud_zx) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_xx)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_xx) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							d_Del_Kd_zxx_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xx) * Ct_xxz + 2.0 * unp1(i, j, k, Atd_xy) * Ct_yxz + 2.0 * unp1(i, j, k, Atd_xz) * Ct_zxz);
							m_Del_Kd_zxx_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zxx_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zxx_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxx_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxx_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xx) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_Del_Kd_zxx_o0_t0_l0 = inv_chi * d_Del_Kd_zxx_o0_t0_m0_l0;
							Del_Kd_zxx = (((((m_Del_Kd_zxx_o0_t0_l0 + d_Del_Kd_zxx_o0_t1_m0_l0) + m_Del_Kd_zxx_o0_t2_l0) + m_Del_Kd_zxx_o0_t3_l0) + m_Del_Kd_zxx_o0_t4_l0) + m_Del_Kd_zxx_o0_t5_l0) + m_Del_Kd_zxx_o0_t6_l0;
							d_Del_Kd_zyz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xyz + unp1(i, j, k, Atd_xy) * Ct_xzz + unp1(i, j, k, Atd_yz) * Ct_yyz + unp1(i, j, k, Atd_yy) * Ct_yzz + unp1(i, j, k, Atd_zz) * Ct_zyz + unp1(i, j, k, Atd_yz) * Ct_zzz);
							m_Del_Kd_zyz_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zyz_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zyz_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zyz_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zyz_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zyz_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zyz_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zyz_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_zz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zyz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_yz) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_Del_Kd_zyz_o0_t0_l0 = inv_chi * d_Del_Kd_zyz_o0_t0_m0_l0;
							Del_Kd_zyz = (((((((((m_Del_Kd_zyz_o0_t0_l0 + d_Del_Kd_zyz_o0_t1_m0_l0) + m_Del_Kd_zyz_o0_t2_l0) + m_Del_Kd_zyz_o0_t3_l0) + m_Del_Kd_zyz_o0_t4_l0) + m_Del_Kd_zyz_o0_t5_l0) + m_Del_Kd_zyz_o0_t6_l0) + m_Del_Kd_zyz_o0_t7_l0) + m_Del_Kd_zyz_o0_t8_l0) + m_Del_Kd_zyz_o0_t9_l0) + m_Del_Kd_zyz_o0_t10_l0;
							d_Del_Kd_zyy_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xy) * Ct_xyz + 2.0 * unp1(i, j, k, Atd_yy) * Ct_yyz + 2.0 * unp1(i, j, k, Atd_yz) * Ct_zyz);
							m_Del_Kd_zyy_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zyy_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zyy_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zyy_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zyy_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_yy) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_Del_Kd_zyy_o0_t0_l0 = inv_chi * d_Del_Kd_zyy_o0_t0_m0_l0;
							Del_Kd_zyy = (((((m_Del_Kd_zyy_o0_t0_l0 + d_Del_Kd_zyy_o0_t1_m0_l0) + m_Del_Kd_zyy_o0_t2_l0) + m_Del_Kd_zyy_o0_t3_l0) + m_Del_Kd_zyy_o0_t4_l0) + m_Del_Kd_zyy_o0_t5_l0) + m_Del_Kd_zyy_o0_t6_l0;
							d_Del_Kd_zxy_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xy) * Ct_xxz + unp1(i, j, k, Atd_xx) * Ct_xyz + unp1(i, j, k, Atd_yy) * Ct_yxz + unp1(i, j, k, Atd_xy) * Ct_yyz + unp1(i, j, k, Atd_yz) * Ct_zxz + unp1(i, j, k, Atd_xz) * Ct_zyz);
							m_Del_Kd_zxy_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zxy_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zxy_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxy_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zxy_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xy) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_Del_Kd_zxy_o0_t0_l0 = inv_chi * d_Del_Kd_zxy_o0_t0_m0_l0;
							Del_Kd_zxy = (((((((((m_Del_Kd_zxy_o0_t0_l0 + d_Del_Kd_zxy_o0_t1_m0_l0) + m_Del_Kd_zxy_o0_t2_l0) + m_Del_Kd_yxz_o0_t3_l0) + m_Del_Kd_zxy_o0_t4_l0) + m_Del_Kd_zxy_o0_t5_l0) + m_Del_Kd_zxy_o0_t6_l0) + m_Del_Kd_zxy_o0_t7_l0) + m_Del_Kd_yxz_o0_t8_l0) + m_Del_Kd_yxz_o0_t9_l0) + m_Del_Kd_yxz_o0_t10_l0;
							d_Del_Kd_zxz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xxz + unp1(i, j, k, Atd_xx) * Ct_xzz + unp1(i, j, k, Atd_yz) * Ct_yxz + unp1(i, j, k, Atd_xy) * Ct_yzz + unp1(i, j, k, Atd_zz) * Ct_zxz + unp1(i, j, k, Atd_xz) * Ct_zzz);
							m_Del_Kd_zxz_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zxz_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zxz_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_zz) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxz_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zxz_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_zxz_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxz_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_zxz_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_zz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_zxz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xz) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_Del_Kd_zxz_o0_t0_l0 = inv_chi * d_Del_Kd_zxz_o0_t0_m0_l0;
							Del_Kd_zxz = (((((((((m_Del_Kd_zxz_o0_t0_l0 + d_Del_Kd_zxz_o0_t1_m0_l0) + m_Del_Kd_zxz_o0_t2_l0) + m_Del_Kd_zxz_o0_t3_l0) + m_Del_Kd_zxz_o0_t4_l0) + m_Del_Kd_zxz_o0_t5_l0) + m_Del_Kd_zxz_o0_t6_l0) + m_Del_Kd_zxz_o0_t7_l0) + m_Del_Kd_zxz_o0_t8_l0) + m_Del_Kd_zxz_o0_t9_l0) + m_Del_Kd_zxz_o0_t10_l0;
							m_Rpd_yz_o0_t25_l0 = .5 * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_yz_o0_t24_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yz) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t23_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yz) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t22_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yz) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yz_o0_t21_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_zz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t20_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t19_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t18_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t17_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_yy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t16_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yz) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yz) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yz_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yz) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yz_o0_t12_l0 = 1.0 * unp1(i, j, k, gtd_yz) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_yz_o0_t11_l0 = .5 * unp1(i, j, k, gtd_yz) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_yz_o0_t10_l0 = (-.5 * unp1(i, j, k, gtd_yz) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t9_l0 = (-.5 * unp1(i, j, k, gtd_yz) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t8_l0 = (-.5 * unp1(i, j, k, gtd_yz) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yz_o0_t7_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t6_l0 = .5 * unp1(i, j, k, gtd_yz) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_yz_o0_t5_l0 = 1.0 * unp1(i, j, k, gtd_yz) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_yz_o0_t4_l0 = .5 * unp1(i, j, k, gtd_yz) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_yz_o0_t3_l0 = 1.0 * unp1(i, j, k, gtd_yz) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_yz_o0_t2_l0 = (-.5 * Ct_zyz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yz_o0_t1_l0 = (-.5 * Ct_yyz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yz_o0_t0_l0 = (-.5 * Ct_xyz) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_yz = ((((((((((((((((((((((((m_Rpd_yz_o0_t0_l0 + m_Rpd_yz_o0_t1_l0) + m_Rpd_yz_o0_t2_l0) + m_Rpd_yz_o0_t3_l0) + m_Rpd_yz_o0_t4_l0) + m_Rpd_yz_o0_t5_l0) + m_Rpd_yz_o0_t6_l0) + m_Rpd_yz_o0_t7_l0) + m_Rpd_yz_o0_t8_l0) + m_Rpd_yz_o0_t9_l0) + m_Rpd_yz_o0_t10_l0) + m_Rpd_yz_o0_t11_l0) + m_Rpd_yz_o0_t12_l0) + m_Rpd_yz_o0_t13_l0) + m_Rpd_yz_o0_t14_l0) + m_Rpd_yz_o0_t15_l0) + m_Rpd_yz_o0_t16_l0) + m_Rpd_yz_o0_t17_l0) + m_Rpd_yz_o0_t18_l0) + m_Rpd_yz_o0_t19_l0) + m_Rpd_yz_o0_t20_l0) + m_Rpd_yz_o0_t21_l0) + m_Rpd_yz_o0_t22_l0) + m_Rpd_yz_o0_t23_l0) + m_Rpd_yz_o0_t24_l0) + m_Rpd_yz_o0_t25_l0;
							d_Del_Kd_yzz_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xz) * Ct_xyz + 2.0 * unp1(i, j, k, Atd_yz) * Ct_yyz + 2.0 * unp1(i, j, k, Atd_zz) * Ct_zyz);
							m_Del_Kd_yzz_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yzz_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yzz_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yzz_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yzz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_zz) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_Del_Kd_yzz_o0_t0_l0 = inv_chi * d_Del_Kd_yzz_o0_t0_m0_l0;
							Del_Kd_yzz = (((((m_Del_Kd_yzz_o0_t0_l0 + d_Del_Kd_yzz_o0_t1_m0_l0) + m_Del_Kd_yzz_o0_t2_l0) + m_Del_Kd_yzz_o0_t3_l0) + m_Del_Kd_yzz_o0_t4_l0) + m_Del_Kd_yzz_o0_t5_l0) + m_Del_Kd_yzz_o0_t6_l0;
							d_Del_Kd_xxy_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xy) * Ct_xxx + unp1(i, j, k, Atd_xx) * Ct_xxy + unp1(i, j, k, Atd_yy) * Ct_yxx + unp1(i, j, k, Atd_xy) * Ct_yxy + unp1(i, j, k, Atd_yz) * Ct_zxx + unp1(i, j, k, Atd_xz) * Ct_zxy);
							m_Del_Kd_xxy_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xxy_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xxy_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxy_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xxy_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xxy_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxy_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xxy_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxy_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xy) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_Del_Kd_xxy_o0_t0_l0 = inv_chi * d_Del_Kd_xxy_o0_t0_m0_l0;
							Del_Kd_xxy = (((((((((m_Del_Kd_xxy_o0_t0_l0 + d_Del_Kd_xxy_o0_t1_m0_l0) + m_Del_Kd_xxy_o0_t2_l0) + m_Del_Kd_xxy_o0_t3_l0) + m_Del_Kd_xxy_o0_t4_l0) + m_Del_Kd_xxy_o0_t5_l0) + m_Del_Kd_xxy_o0_t6_l0) + m_Del_Kd_xxy_o0_t7_l0) + m_Del_Kd_xxy_o0_t8_l0) + m_Del_Kd_xxy_o0_t9_l0) + m_Del_Kd_xxy_o0_t10_l0;
							d_Del_Kd_xzz_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xz) * Ct_xxz + 2.0 * unp1(i, j, k, Atd_yz) * Ct_yxz + 2.0 * unp1(i, j, k, Atd_zz) * Ct_zxz);
							m_Del_Kd_xzz_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xzz_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xzz_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xzz_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xzz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_zz) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_Del_Kd_xzz_o0_t0_l0 = inv_chi * d_Del_Kd_xzz_o0_t0_m0_l0;
							Del_Kd_xzz = (((((m_Del_Kd_xzz_o0_t0_l0 + d_Del_Kd_xzz_o0_t1_m0_l0) + m_Del_Kd_xzz_o0_t2_l0) + m_Del_Kd_xzz_o0_t3_l0) + m_Del_Kd_xzz_o0_t4_l0) + m_Del_Kd_xzz_o0_t5_l0) + m_Del_Kd_xzz_o0_t6_l0;
							d_Del_Kd_xyz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xxy + unp1(i, j, k, Atd_xy) * Ct_xxz + unp1(i, j, k, Atd_yz) * Ct_yxy + unp1(i, j, k, Atd_yy) * Ct_yxz + unp1(i, j, k, Atd_zz) * Ct_zxy + unp1(i, j, k, Atd_yz) * Ct_zxz);
							m_Del_Kd_xyz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_yz) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_Del_Kd_xyz_o0_t0_l0 = inv_chi * d_Del_Kd_xyz_o0_t0_m0_l0;
							Del_Kd_xyz = (((((((((m_Del_Kd_xyz_o0_t0_l0 + d_Del_Kd_xyz_o0_t1_m0_l0) + m_Del_Kd_xyz_o0_t2_l0) + m_Del_Kd_zxy_o0_t4_l0) + m_Del_Kd_yxz_o0_t4_l0) + m_Del_Kd_yxz_o0_t5_l0) + m_Del_Kd_yxz_o0_t6_l0) + m_Del_Kd_yxz_o0_t7_l0) + m_Del_Kd_zxy_o0_t5_l0) + m_Del_Kd_zxy_o0_t6_l0) + m_Del_Kd_zxy_o0_t7_l0;
							BWeyl_xz = (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yxz) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zxz + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_xyz + (-unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_zyz) + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_xzz) + unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_yzz;
							BWeyl_zz = (-unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_yxz) + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_zxz + unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_xyz + (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_zyz) + (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xzz) + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yzz;
							BWeyl_yz = (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_yxz) + unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_zxz + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xyz + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zyz) + (-unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_xzz) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_yzz;
							d_Del_Kd_xyy_o0_t1_m0_l0 = -inv_chi * (2.0 * unp1(i, j, k, Atd_xy) * Ct_xxy + 2.0 * unp1(i, j, k, Atd_yy) * Ct_yxy + 2.0 * unp1(i, j, k, Atd_yz) * Ct_zxy);
							m_Del_Kd_xyy_o0_t6_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xyy_o0_t5_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xyy_o0_t4_l0 = (-1.0 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xyy_o0_t3_l0 = 1.0 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xyy_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_yy) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_Del_Kd_xyy_o0_t0_l0 = inv_chi * d_Del_Kd_xyy_o0_t0_m0_l0;
							Del_Kd_xyy = (((((m_Del_Kd_xyy_o0_t0_l0 + d_Del_Kd_xyy_o0_t1_m0_l0) + m_Del_Kd_xyy_o0_t2_l0) + m_Del_Kd_xyy_o0_t3_l0) + m_Del_Kd_xyy_o0_t4_l0) + m_Del_Kd_xyy_o0_t5_l0) + m_Del_Kd_xyy_o0_t6_l0;
							d_Del_Kd_xxz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xxx + unp1(i, j, k, Atd_xx) * Ct_xxz + unp1(i, j, k, Atd_yz) * Ct_yxx + unp1(i, j, k, Atd_xy) * Ct_yxz + unp1(i, j, k, Atd_zz) * Ct_zxx + unp1(i, j, k, Atd_xz) * Ct_zxz);
							m_Del_Kd_xxz_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xxz_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xxz_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xz) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxz_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xxz_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_xxz_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xx) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxz_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_xxz_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_xxz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xz) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_Del_Kd_xxz_o0_t0_l0 = inv_chi * d_Del_Kd_xxz_o0_t0_m0_l0;
							Del_Kd_xxz = (((((((((m_Del_Kd_xxz_o0_t0_l0 + d_Del_Kd_xxz_o0_t1_m0_l0) + m_Del_Kd_xxz_o0_t2_l0) + m_Del_Kd_xxz_o0_t3_l0) + m_Del_Kd_xxz_o0_t4_l0) + m_Del_Kd_xxz_o0_t5_l0) + m_Del_Kd_xxz_o0_t6_l0) + m_Del_Kd_xxz_o0_t7_l0) + m_Del_Kd_xxz_o0_t8_l0) + m_Del_Kd_xxz_o0_t9_l0) + m_Del_Kd_xxz_o0_t10_l0;
							BWeyl_xx = (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yxx) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zxx + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_xxy + (-unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_zxy) + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_xxz) + unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_yxz;
							BWeyl_zx = (-unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_yxx) + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_zxx + unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_xxy + (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_zxy) + (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xxz) + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yxz;
							BWeyl_yx = (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_yxx) + unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_zxx + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xxy + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zxy) + (-unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_xxz) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_yxz;
							m_Rpd_xz_o0_t25_l0 = .5 * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_xz_o0_t24_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xz) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t23_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xz) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xz_o0_t22_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xz) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t21_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_zz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t20_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t19_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t18_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t17_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t16_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xz) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xz_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xz) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xz) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t12_l0 = (-.5 * Ct_zxz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t11_l0 = (-.5 * Ct_yxz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xz_o0_t10_l0 = (-.5 * Ct_xxz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xz_o0_t9_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t8_l0 = .5 * unp1(i, j, k, gtd_xz) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_xz_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_xz) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_xz_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xz) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_xz_o0_t5_l0 = 1.0 * unp1(i, j, k, gtd_xz) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_xz_o0_t4_l0 = 1.0 * unp1(i, j, k, gtd_xz) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_xz_o0_t3_l0 = .5 * unp1(i, j, k, gtd_xz) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_xz_o0_t2_l0 = (-.5 * unp1(i, j, k, gtd_xz) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xz_o0_t1_l0 = (-.5 * unp1(i, j, k, gtd_xz) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xz_o0_t0_l0 = (-.5 * unp1(i, j, k, gtd_xz) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_xz = ((((((((((((((((((((((((m_Rpd_xz_o0_t0_l0 + m_Rpd_xz_o0_t1_l0) + m_Rpd_xz_o0_t2_l0) + m_Rpd_xz_o0_t3_l0) + m_Rpd_xz_o0_t4_l0) + m_Rpd_xz_o0_t5_l0) + m_Rpd_xz_o0_t6_l0) + m_Rpd_xz_o0_t7_l0) + m_Rpd_xz_o0_t8_l0) + m_Rpd_xz_o0_t9_l0) + m_Rpd_xz_o0_t10_l0) + m_Rpd_xz_o0_t11_l0) + m_Rpd_xz_o0_t12_l0) + m_Rpd_xz_o0_t13_l0) + m_Rpd_xz_o0_t14_l0) + m_Rpd_xz_o0_t15_l0) + m_Rpd_xz_o0_t16_l0) + m_Rpd_xz_o0_t17_l0) + m_Rpd_xz_o0_t18_l0) + m_Rpd_xz_o0_t19_l0) + m_Rpd_xz_o0_t20_l0) + m_Rpd_xz_o0_t21_l0) + m_Rpd_xz_o0_t22_l0) + m_Rpd_xz_o0_t23_l0) + m_Rpd_xz_o0_t24_l0) + m_Rpd_xz_o0_t25_l0;
							EWeyl_xz = (-inv_chi * Rpd_xz) + (-Rtd_xz) + inv_chi * (unp1(i, j, k, Atd_xx) * Atud_xz + unp1(i, j, k, Atd_xy) * Atud_yz + unp1(i, j, k, Atd_xz) * Atud_zz) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_xz)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_xz) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							m_Rpd_xy_o0_t25_l0 = .5 * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_xy_o0_t24_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xy) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xy_o0_t23_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xy) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t22_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_xy) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t21_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t20_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_yy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t19_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t18_l0 = inv_chi * Zu_z * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t17_l0 = inv_chi * Zu_y * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t16_l0 = inv_chi * Zu_x * unp1(i, j, k, gtd_xx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xy) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xy) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_xy) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t12_l0 = (-.5 * Ct_zxy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xy_o0_t11_l0 = (-.5 * Ct_yxy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t10_l0 = (-.5 * Ct_xxy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_xy_o0_t9_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t8_l0 = .5 * unp1(i, j, k, gtd_xy) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_xy_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_xy) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_xy_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xy) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_xy_o0_t5_l0 = 1.0 * unp1(i, j, k, gtd_xy) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_xy_o0_t4_l0 = 1.0 * unp1(i, j, k, gtd_xy) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_xy_o0_t3_l0 = .5 * unp1(i, j, k, gtd_xy) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_xy_o0_t2_l0 = (-.5 * unp1(i, j, k, gtd_xy) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_xy_o0_t1_l0 = (-.5 * unp1(i, j, k, gtd_xy) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_xy_o0_t0_l0 = (-.5 * unp1(i, j, k, gtd_xy) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_xy = ((((((((((((((((((((((((m_Rpd_xy_o0_t0_l0 + m_Rpd_xy_o0_t1_l0) + m_Rpd_xy_o0_t2_l0) + m_Rpd_xy_o0_t3_l0) + m_Rpd_xy_o0_t4_l0) + m_Rpd_xy_o0_t5_l0) + m_Rpd_xy_o0_t6_l0) + m_Rpd_xy_o0_t7_l0) + m_Rpd_xy_o0_t8_l0) + m_Rpd_xy_o0_t9_l0) + m_Rpd_xy_o0_t10_l0) + m_Rpd_xy_o0_t11_l0) + m_Rpd_xy_o0_t12_l0) + m_Rpd_xy_o0_t13_l0) + m_Rpd_xy_o0_t14_l0) + m_Rpd_xy_o0_t15_l0) + m_Rpd_xy_o0_t16_l0) + m_Rpd_xy_o0_t17_l0) + m_Rpd_xy_o0_t18_l0) + m_Rpd_xy_o0_t19_l0) + m_Rpd_xy_o0_t20_l0) + m_Rpd_xy_o0_t21_l0) + m_Rpd_xy_o0_t22_l0) + m_Rpd_xy_o0_t23_l0) + m_Rpd_xy_o0_t24_l0) + m_Rpd_xy_o0_t25_l0;
							m_Rpd_zz_o0_t22_l0 = .5 * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_zz_o0_t21_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_zz) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t20_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_zz) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_zz_o0_t19_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_zz) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_zz_o0_t18_l0 = 2.0 * inv_chi * Zu_z * unp1(i, j, k, gtd_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t17_l0 = 2.0 * inv_chi * Zu_y * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t16_l0 = 2.0 * inv_chi * Zu_x * unp1(i, j, k, gtd_xz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_zz) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_zz_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_zz) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_zz_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_zz) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_zz_o0_t12_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t11_l0 = (-.5 * Ct_zzz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t10_l0 = (-.5 * Ct_yzz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_zz_o0_t9_l0 = (-.5 * Ct_xzz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_zz_o0_t8_l0 = .5 * unp1(i, j, k, gtd_zz) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_zz_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_zz) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_zz_o0_t6_l0 = .5 * unp1(i, j, k, gtd_zz) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_zz_o0_t5_l0 = 1.0 * unp1(i, j, k, gtd_zz) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_zz_o0_t4_l0 = 1.0 * unp1(i, j, k, gtd_zz) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_zz_o0_t3_l0 = .5 * unp1(i, j, k, gtd_zz) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_zz_o0_t2_l0 = (-.5 * unp1(i, j, k, gtd_zz) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_zz_o0_t1_l0 = (-.5 * unp1(i, j, k, gtd_zz) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_zz_o0_t0_l0 = (-.5 * unp1(i, j, k, gtd_zz) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_zz = (((((((((((((((((((((m_Rpd_zz_o0_t0_l0 + m_Rpd_zz_o0_t1_l0) + m_Rpd_zz_o0_t2_l0) + m_Rpd_zz_o0_t3_l0) + m_Rpd_zz_o0_t4_l0) + m_Rpd_zz_o0_t5_l0) + m_Rpd_zz_o0_t6_l0) + m_Rpd_zz_o0_t7_l0) + m_Rpd_zz_o0_t8_l0) + m_Rpd_zz_o0_t9_l0) + m_Rpd_zz_o0_t10_l0) + m_Rpd_zz_o0_t11_l0) + m_Rpd_zz_o0_t12_l0) + m_Rpd_zz_o0_t13_l0) + m_Rpd_zz_o0_t14_l0) + m_Rpd_zz_o0_t15_l0) + m_Rpd_zz_o0_t16_l0) + m_Rpd_zz_o0_t17_l0) + m_Rpd_zz_o0_t18_l0) + m_Rpd_zz_o0_t19_l0) + m_Rpd_zz_o0_t20_l0) + m_Rpd_zz_o0_t21_l0) + m_Rpd_zz_o0_t22_l0;
							EWeyl_zz = (-inv_chi * Rpd_zz) + (-Rtd_zz) + inv_chi * (unp1(i, j, k, Atd_xz) * Atud_xz + unp1(i, j, k, Atd_yz) * Atud_yz + unp1(i, j, k, Atd_zz) * Atud_zz) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_zz)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_zz) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							d_Del_Kd_yyz_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xz) * Ct_xyy + unp1(i, j, k, Atd_xy) * Ct_xyz + unp1(i, j, k, Atd_yz) * Ct_yyy + unp1(i, j, k, Atd_yy) * Ct_yyz + unp1(i, j, k, Atd_zz) * Ct_zyy + unp1(i, j, k, Atd_yz) * Ct_zyz);
							m_Del_Kd_yyz_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yyz_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yyz_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yz) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yyz_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yyz_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yyz_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yyz_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yyz_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yyz_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_yz) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_Del_Kd_yyz_o0_t0_l0 = inv_chi * d_Del_Kd_yyz_o0_t0_m0_l0;
							Del_Kd_yyz = (((((((((m_Del_Kd_yyz_o0_t0_l0 + d_Del_Kd_yyz_o0_t1_m0_l0) + m_Del_Kd_yyz_o0_t2_l0) + m_Del_Kd_yyz_o0_t3_l0) + m_Del_Kd_yyz_o0_t4_l0) + m_Del_Kd_yyz_o0_t5_l0) + m_Del_Kd_yyz_o0_t6_l0) + m_Del_Kd_yyz_o0_t7_l0) + m_Del_Kd_yyz_o0_t8_l0) + m_Del_Kd_yyz_o0_t9_l0) + m_Del_Kd_yyz_o0_t10_l0;
							d_Del_Kd_yxy_o0_t1_m0_l0 = -inv_chi * (unp1(i, j, k, Atd_xy) * Ct_xxy + unp1(i, j, k, Atd_xx) * Ct_xyy + unp1(i, j, k, Atd_yy) * Ct_yxy + unp1(i, j, k, Atd_xy) * Ct_yyy + unp1(i, j, k, Atd_yz) * Ct_zxy + unp1(i, j, k, Atd_xz) * Ct_zyy);
							m_Del_Kd_yxy_o0_t10_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxy_o0_t9_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxy_o0_t8_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_yy) * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxy_o0_t7_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Del_Kd_yxy_o0_t6_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxy_o0_t5_l0 = (-.5 * (inv_chi * inv_chi) * unp1(i, j, k, gtd_xy) * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxy_o0_t4_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_xy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Del_Kd_yxy_o0_t3_l0 = .5 * (inv_chi * inv_chi) * unp1(i, j, k, Atd_yy) * d_Rpd_xx_o0_t0_m0_l0;
							m_Del_Kd_yxy_o0_t2_l0 = .3333333333333333 * inv_chi * unp1(i, j, k, gtd_xy) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_Del_Kd_yxy_o0_t0_l0 = inv_chi * d_Del_Kd_yxy_o0_t0_m0_l0;
							Del_Kd_yxy = (((((((((m_Del_Kd_yxy_o0_t0_l0 + d_Del_Kd_yxy_o0_t1_m0_l0) + m_Del_Kd_yxy_o0_t2_l0) + m_Del_Kd_yxy_o0_t3_l0) + m_Del_Kd_yxy_o0_t4_l0) + m_Del_Kd_yxy_o0_t5_l0) + m_Del_Kd_yxy_o0_t6_l0) + m_Del_Kd_yxy_o0_t7_l0) + m_Del_Kd_yxy_o0_t8_l0) + m_Del_Kd_yxy_o0_t9_l0) + m_Del_Kd_yxy_o0_t10_l0;
							BWeyl_xy = (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yxy) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zxy + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_xyy + (-unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_zyy) + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_xyz) + unp1(i, j, k, gtd_xx) * (pow(inv_chi, 2.5)) * Del_Kd_yyz;
							BWeyl_zy = (-unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_yxy) + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_zxy + unp1(i, j, k, gtd_zz) * (pow(inv_chi, 2.5)) * Del_Kd_xyy + (-unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_zyy) + (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xyz) + unp1(i, j, k, gtd_xz) * (pow(inv_chi, 2.5)) * Del_Kd_yyz;
							BWeyl_yy = (-unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_yxy) + unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_zxy + unp1(i, j, k, gtd_yz) * (pow(inv_chi, 2.5)) * Del_Kd_xyy + (-unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_zyy) + (-unp1(i, j, k, gtd_yy) * (pow(inv_chi, 2.5)) * Del_Kd_xyz) + unp1(i, j, k, gtd_xy) * (pow(inv_chi, 2.5)) * Del_Kd_yyz;
							m_Rpd_yy_o0_t22_l0 = .5 * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_yy_o0_t21_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yy) * gtu_zz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yy_o0_t20_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yy) * gtu_yy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t19_l0 = (-.7500000000000000 * unp1(i, j, k, gtd_yy) * gtu_xx * inv_chi) * d_Rpd_xx_o0_t0_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yy_o0_t18_l0 = 2.0 * inv_chi * Zu_z * unp1(i, j, k, gtd_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t17_l0 = 2.0 * inv_chi * Zu_y * unp1(i, j, k, gtd_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t16_l0 = 2.0 * inv_chi * Zu_x * unp1(i, j, k, gtd_xy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t15_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yy) * gtu_yz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t14_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yy) * gtu_xz * inv_chi) * d_Rpd_xx_o0_t2_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yy_o0_t13_l0 = (-1.500000000000000 * unp1(i, j, k, gtd_yy) * gtu_xy * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yy_o0_t12_l0 = .5 * unp1(i, j, k, gtd_yy) * gtu_zz * d_Rpd_xx_o0_t8_m0_l0;
							m_Rpd_yy_o0_t11_l0 = 1.0 * unp1(i, j, k, gtd_yy) * gtu_yz * d_Rpd_xx_o0_t7_m0_l0;
							m_Rpd_yy_o0_t10_l0 = .5 * unp1(i, j, k, gtd_yy) * gtu_yy * d_Rpd_xx_o0_t6_m0_l0;
							m_Rpd_yy_o0_t9_l0 = 1.0 * unp1(i, j, k, gtd_yy) * gtu_xz * d_Rpd_xx_o0_t5_m0_l0;
							m_Rpd_yy_o0_t8_l0 = 1.0 * unp1(i, j, k, gtd_yy) * gtu_xy * d_Rpd_xx_o0_t4_m0_l0;
							m_Rpd_yy_o0_t7_l0 = .5 * unp1(i, j, k, gtd_yy) * gtu_xx * d_Rpd_xx_o0_t3_m0_l0;
							m_Rpd_yy_o0_t6_l0 = (-.5 * unp1(i, j, k, gtd_yy) * unp1(i, j, k, Gamh_z)) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yy_o0_t5_l0 = (-.5 * unp1(i, j, k, gtd_yy) * unp1(i, j, k, Gamh_y)) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t4_l0 = (-.5 * unp1(i, j, k, gtd_yy) * unp1(i, j, k, Gamh_x)) * d_Rpd_xx_o0_t0_m0_l0;
							m_Rpd_yy_o0_t3_l0 = (-.2500000000000000 * inv_chi) * d_Rpd_xx_o0_t1_m0_l0 * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t2_l0 = (-.5 * Ct_zyy) * d_Rpd_xx_o0_t2_m0_l0;
							m_Rpd_yy_o0_t1_l0 = (-.5 * Ct_yyy) * d_Rpd_xx_o0_t1_m0_l0;
							m_Rpd_yy_o0_t0_l0 = (-.5 * Ct_xyy) * d_Rpd_xx_o0_t0_m0_l0;
							Rpd_yy = (((((((((((((((((((((m_Rpd_yy_o0_t0_l0 + m_Rpd_yy_o0_t1_l0) + m_Rpd_yy_o0_t2_l0) + m_Rpd_yy_o0_t3_l0) + m_Rpd_yy_o0_t4_l0) + m_Rpd_yy_o0_t5_l0) + m_Rpd_yy_o0_t6_l0) + m_Rpd_yy_o0_t7_l0) + m_Rpd_yy_o0_t8_l0) + m_Rpd_yy_o0_t9_l0) + m_Rpd_yy_o0_t10_l0) + m_Rpd_yy_o0_t11_l0) + m_Rpd_yy_o0_t12_l0) + m_Rpd_yy_o0_t13_l0) + m_Rpd_yy_o0_t14_l0) + m_Rpd_yy_o0_t15_l0) + m_Rpd_yy_o0_t16_l0) + m_Rpd_yy_o0_t17_l0) + m_Rpd_yy_o0_t18_l0) + m_Rpd_yy_o0_t19_l0) + m_Rpd_yy_o0_t20_l0) + m_Rpd_yy_o0_t21_l0) + m_Rpd_yy_o0_t22_l0;
							d_Rtd_xy_o0_t12_m0_l0 = gtu_zz * (Ct_xxz * Ctd_yxz + Ct_xxz * Ctd_xyz + Ct_xyz * Ctd_xxz + 2.0 * Ct_yxz * Ctd_yyz + Ct_yyz * Ctd_xyz + Ct_zxz * Ctd_yzz + Ct_zxz * Ctd_zyz + Ct_zyz * Ctd_xzz) + gtu_yz * (Ct_xxy * Ctd_yxz + Ct_xxy * Ctd_xyz + Ct_xyy * Ctd_xxz + 2.0 * Ct_yxy * Ctd_yyz + Ct_yyy * Ctd_xyz + Ct_zxy * Ctd_yzz + Ct_zxy * Ctd_zyz + Ct_zyy * Ctd_xzz) + gtu_xz * (Ct_xxx * Ctd_yxz + Ct_xxx * Ctd_xyz + Ct_xxy * Ctd_xxz + 2.0 * Ct_yxx * Ctd_yyz + Ct_yxy * Ctd_xyz + Ct_zxx * Ctd_yzz + Ct_zxx * Ctd_zyz + Ct_zxy * Ctd_xzz) + gtu_yz * (Ct_xxz * Ctd_yxy + Ct_xxz * Ctd_xyy + Ct_xyz * Ctd_xxy + 2.0 * Ct_yxz * Ctd_yyy + Ct_yyz * Ctd_xyy + Ct_zxz * Ctd_yyz + Ct_zxz * Ctd_zyy + Ct_zyz * Ctd_xyz) + gtu_yy * (Ct_xxy * Ctd_yxy + Ct_xxy * Ctd_xyy + Ct_xyy * Ctd_xxy + 2.0 * Ct_yxy * Ctd_yyy + Ct_yyy * Ctd_xyy + Ct_zxy * Ctd_yyz + Ct_zxy * Ctd_zyy + Ct_zyy * Ctd_xyz) + gtu_xy * (Ct_xxx * Ctd_yxy + Ct_xxx * Ctd_xyy + Ct_xxy * Ctd_xxy + 2.0 * Ct_yxx * Ctd_yyy + Ct_yxy * Ctd_xyy + Ct_zxx * Ctd_yyz + Ct_zxx * Ctd_zyy + Ct_zxy * Ctd_xyz) + gtu_xz * (Ct_xxz * Ctd_yxx + Ct_xxz * Ctd_xxy + Ct_xyz * Ctd_xxx + 2.0 * Ct_yxz * Ctd_yxy + Ct_yyz * Ctd_xxy + Ct_zxz * Ctd_yxz + Ct_zxz * Ctd_zxy + Ct_zyz * Ctd_xxz) + gtu_xy * (Ct_xxy * Ctd_yxx + Ct_xxy * Ctd_xxy + Ct_xyy * Ctd_xxx + 2.0 * Ct_yxy * Ctd_yxy + Ct_yyy * Ctd_xxy + Ct_zxy * Ctd_yxz + Ct_zxy * Ctd_zxy + Ct_zyy * Ctd_xxz) + gtu_xx * (Ct_xxx * Ctd_yxx + Ct_xxx * Ctd_xxy + Ct_xxy * Ctd_xxx + 2.0 * Ct_yxx * Ctd_yxy + Ct_yxy * Ctd_xxy + Ct_zxx * Ctd_yxz + Ct_zxx * Ctd_zxy + Ct_zxy * Ctd_xxz) + .5 * unp1(i, j, k, Gamh_z) * (Ctd_xyz + Ctd_yxz) + .5 * unp1(i, j, k, Gamh_y) * (Ctd_xyy + Ctd_yxy) + .5 * unp1(i, j, k, Gamh_x) * (Ctd_xxy + Ctd_yxx);
							m_Rtd_xy_o0_t11_l0 = .5 * unp1(i, j, k, gtd_yz) * d_Rtd_xx_o0_t8_m0_l0;
							m_Rtd_xy_o0_t10_l0 = .5 * unp1(i, j, k, gtd_yy) * d_Rtd_xx_o0_t7_m0_l0;
							m_Rtd_xy_o0_t9_l0 = .5 * unp1(i, j, k, gtd_xy) * d_Rtd_xx_o0_t6_m0_l0;
							m_Rtd_xy_o0_t8_l0 = .5 * unp1(i, j, k, gtd_xz) * d_Rtd_xy_o0_t8_m0_l0;
							m_Rtd_xy_o0_t7_l0 = .5 * unp1(i, j, k, gtd_xy) * d_Rtd_xy_o0_t7_m0_l0;
							m_Rtd_xy_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xx) * d_Rtd_xy_o0_t6_m0_l0;
							m_Rtd_xy_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_xy_o0_t5_m0_l0;
							m_Rtd_xy_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_xy_o0_t4_m0_l0;
							m_Rtd_xy_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_xy_o0_t3_m0_l0;
							m_Rtd_xy_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_xy_o0_t2_m0_l0;
							m_Rtd_xy_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_xy_o0_t1_m0_l0;
							m_Rtd_xy_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_xy_o0_t0_m0_l0;
							Rtd_xy = (((((((((((m_Rtd_xy_o0_t0_l0 + m_Rtd_xy_o0_t1_l0) + m_Rtd_xy_o0_t2_l0) + m_Rtd_xy_o0_t3_l0) + m_Rtd_xy_o0_t4_l0) + m_Rtd_xy_o0_t5_l0) + m_Rtd_xy_o0_t6_l0) + m_Rtd_xy_o0_t7_l0) + m_Rtd_xy_o0_t8_l0) + m_Rtd_xy_o0_t9_l0) + m_Rtd_xy_o0_t10_l0) + m_Rtd_xy_o0_t11_l0) + d_Rtd_xy_o0_t12_m0_l0;
							EWeyl_xy = (-inv_chi * Rpd_xy) + (-Rtd_xy) + inv_chi * (unp1(i, j, k, Atd_xx) * Atud_xy + unp1(i, j, k, Atd_xy) * Atud_yy + unp1(i, j, k, Atd_xz) * Atud_zy) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_xy)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_xy) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							d_Rtd_yy_o0_t9_m0_l0 = gtu_zz * (2.0 * Ct_xyz * Ctd_yxz + Ct_xyz * Ctd_xyz + 3.0 * Ct_yyz * Ctd_yyz + 2.0 * Ct_zyz * Ctd_yzz + Ct_zyz * Ctd_zyz) + gtu_yz * (2.0 * Ct_xyy * Ctd_yxz + Ct_xyy * Ctd_xyz + 3.0 * Ct_yyy * Ctd_yyz + 2.0 * Ct_zyy * Ctd_yzz + Ct_zyy * Ctd_zyz) + gtu_xz * (2.0 * Ct_xxy * Ctd_yxz + Ct_xxy * Ctd_xyz + 3.0 * Ct_yxy * Ctd_yyz + 2.0 * Ct_zxy * Ctd_yzz + Ct_zxy * Ctd_zyz) + gtu_yz * (2.0 * Ct_xyz * Ctd_yxy + Ct_xyz * Ctd_xyy + 3.0 * Ct_yyz * Ctd_yyy + 2.0 * Ct_zyz * Ctd_yyz + Ct_zyz * Ctd_zyy) + gtu_yy * (2.0 * Ct_xyy * Ctd_yxy + Ct_xyy * Ctd_xyy + 3.0 * Ct_yyy * Ctd_yyy + 2.0 * Ct_zyy * Ctd_yyz + Ct_zyy * Ctd_zyy) + gtu_xy * (2.0 * Ct_xxy * Ctd_yxy + Ct_xxy * Ctd_xyy + 3.0 * Ct_yxy * Ctd_yyy + 2.0 * Ct_zxy * Ctd_yyz + Ct_zxy * Ctd_zyy) + gtu_xz * (2.0 * Ct_xyz * Ctd_yxx + Ct_xyz * Ctd_xxy + 3.0 * Ct_yyz * Ctd_yxy + 2.0 * Ct_zyz * Ctd_yxz + Ct_zyz * Ctd_zxy) + gtu_xy * (2.0 * Ct_xyy * Ctd_yxx + Ct_xyy * Ctd_xxy + 3.0 * Ct_yyy * Ctd_yxy + 2.0 * Ct_zyy * Ctd_yxz + Ct_zyy * Ctd_zxy) + gtu_xx * (2.0 * Ct_xxy * Ctd_yxx + Ct_xxy * Ctd_xxy + 3.0 * Ct_yxy * Ctd_yxy + 2.0 * Ct_zxy * Ctd_yxz + Ct_zxy * Ctd_zxy) + 1.0 * unp1(i, j, k, Gamh_z) * Ctd_yyz + 1.0 * unp1(i, j, k, Gamh_y) * Ctd_yyy + 1.0 * unp1(i, j, k, Gamh_x) * Ctd_yxy;
							m_Rtd_yy_o0_t8_l0 = 1.0 * unp1(i, j, k, gtd_yz) * d_Rtd_xy_o0_t8_m0_l0;
							m_Rtd_yy_o0_t7_l0 = 1.0 * unp1(i, j, k, gtd_yy) * d_Rtd_xy_o0_t7_m0_l0;
							m_Rtd_yy_o0_t6_l0 = 1.0 * unp1(i, j, k, gtd_xy) * d_Rtd_xy_o0_t6_m0_l0;
							m_Rtd_yy_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_yy_o0_t5_m0_l0;
							m_Rtd_yy_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_yy_o0_t4_m0_l0;
							m_Rtd_yy_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_yy_o0_t3_m0_l0;
							m_Rtd_yy_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_yy_o0_t2_m0_l0;
							m_Rtd_yy_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_yy_o0_t1_m0_l0;
							m_Rtd_yy_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_yy_o0_t0_m0_l0;
							Rtd_yy = ((((((((m_Rtd_yy_o0_t0_l0 + m_Rtd_yy_o0_t1_l0) + m_Rtd_yy_o0_t2_l0) + m_Rtd_yy_o0_t3_l0) + m_Rtd_yy_o0_t4_l0) + m_Rtd_yy_o0_t5_l0) + m_Rtd_yy_o0_t6_l0) + m_Rtd_yy_o0_t7_l0) + m_Rtd_yy_o0_t8_l0) + d_Rtd_yy_o0_t9_m0_l0;
							EWeyl_yy = (-inv_chi * Rpd_yy) + (-Rtd_yy) + inv_chi * (unp1(i, j, k, Atd_xy) * Atud_xy + unp1(i, j, k, Atd_yy) * Atud_yy + unp1(i, j, k, Atd_yz) * Atud_zy) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_yy)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_yy) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							d_Rtd_yz_o0_t12_m0_l0 = gtu_zz * (Ct_xyz * Ctd_zxz + Ct_xyz * Ctd_xzz + Ct_xzz * Ctd_yxz + Ct_yyz * Ctd_zyz + Ct_yyz * Ctd_yzz + Ct_yzz * Ctd_yyz + 2.0 * Ct_zyz * Ctd_zzz + Ct_zzz * Ctd_yzz) + gtu_yz * (Ct_xyy * Ctd_zxz + Ct_xyy * Ctd_xzz + Ct_xyz * Ctd_yxz + Ct_yyy * Ctd_zyz + Ct_yyy * Ctd_yzz + Ct_yyz * Ctd_yyz + 2.0 * Ct_zyy * Ctd_zzz + Ct_zyz * Ctd_yzz) + gtu_xz * (Ct_xxy * Ctd_zxz + Ct_xxy * Ctd_xzz + Ct_xxz * Ctd_yxz + Ct_yxy * Ctd_zyz + Ct_yxy * Ctd_yzz + Ct_yxz * Ctd_yyz + 2.0 * Ct_zxy * Ctd_zzz + Ct_zxz * Ctd_yzz) + gtu_yz * (Ct_xyz * Ctd_zxy + Ct_xyz * Ctd_xyz + Ct_xzz * Ctd_yxy + Ct_yyz * Ctd_zyy + Ct_yyz * Ctd_yyz + Ct_yzz * Ctd_yyy + 2.0 * Ct_zyz * Ctd_zyz + Ct_zzz * Ctd_yyz) + gtu_yy * (Ct_xyy * Ctd_zxy + Ct_xyy * Ctd_xyz + Ct_xyz * Ctd_yxy + Ct_yyy * Ctd_zyy + Ct_yyy * Ctd_yyz + Ct_yyz * Ctd_yyy + 2.0 * Ct_zyy * Ctd_zyz + Ct_zyz * Ctd_yyz) + gtu_xy * (Ct_xxy * Ctd_zxy + Ct_xxy * Ctd_xyz + Ct_xxz * Ctd_yxy + Ct_yxy * Ctd_zyy + Ct_yxy * Ctd_yyz + Ct_yxz * Ctd_yyy + 2.0 * Ct_zxy * Ctd_zyz + Ct_zxz * Ctd_yyz) + gtu_xz * (Ct_xyz * Ctd_zxx + Ct_xyz * Ctd_xxz + Ct_xzz * Ctd_yxx + Ct_yyz * Ctd_zxy + Ct_yyz * Ctd_yxz + Ct_yzz * Ctd_yxy + 2.0 * Ct_zyz * Ctd_zxz + Ct_zzz * Ctd_yxz) + gtu_xy * (Ct_xyy * Ctd_zxx + Ct_xyy * Ctd_xxz + Ct_xyz * Ctd_yxx + Ct_yyy * Ctd_zxy + Ct_yyy * Ctd_yxz + Ct_yyz * Ctd_yxy + 2.0 * Ct_zyy * Ctd_zxz + Ct_zyz * Ctd_yxz) + gtu_xx * (Ct_xxy * Ctd_zxx + Ct_xxy * Ctd_xxz + Ct_xxz * Ctd_yxx + Ct_yxy * Ctd_zxy + Ct_yxy * Ctd_yxz + Ct_yxz * Ctd_yxy + 2.0 * Ct_zxy * Ctd_zxz + Ct_zxz * Ctd_yxz) + .5 * unp1(i, j, k, Gamh_z) * (Ctd_yzz + Ctd_zyz) + .5 * unp1(i, j, k, Gamh_y) * (Ctd_yyz + Ctd_zyy) + .5 * unp1(i, j, k, Gamh_x) * (Ctd_yxz + Ctd_zxy);
							m_Rtd_yz_o0_t11_l0 = .5 * unp1(i, j, k, gtd_zz) * d_Rtd_xy_o0_t8_m0_l0;
							m_Rtd_yz_o0_t10_l0 = .5 * unp1(i, j, k, gtd_yz) * d_Rtd_xy_o0_t7_m0_l0;
							m_Rtd_yz_o0_t9_l0 = .5 * unp1(i, j, k, gtd_xz) * d_Rtd_xy_o0_t6_m0_l0;
							m_Rtd_yz_o0_t8_l0 = .5 * unp1(i, j, k, gtd_yz) * d_Rtd_xz_o0_t8_m0_l0;
							m_Rtd_yz_o0_t7_l0 = .5 * unp1(i, j, k, gtd_yy) * d_Rtd_xz_o0_t7_m0_l0;
							m_Rtd_yz_o0_t6_l0 = .5 * unp1(i, j, k, gtd_xy) * d_Rtd_xz_o0_t6_m0_l0;
							m_Rtd_yz_o0_t5_l0 = (-.5 * gtu_zz) * d_Rtd_yz_o0_t5_m0_l0;
							m_Rtd_yz_o0_t4_l0 = (-1.0 * gtu_yz) * d_Rtd_yz_o0_t4_m0_l0;
							m_Rtd_yz_o0_t3_l0 = (-.5 * gtu_yy) * d_Rtd_yz_o0_t3_m0_l0;
							m_Rtd_yz_o0_t2_l0 = (-1.0 * gtu_xz) * d_Rtd_yz_o0_t2_m0_l0;
							m_Rtd_yz_o0_t1_l0 = (-1.0 * gtu_xy) * d_Rtd_yz_o0_t1_m0_l0;
							m_Rtd_yz_o0_t0_l0 = (-.5 * gtu_xx) * d_Rtd_yz_o0_t0_m0_l0;
							Rtd_yz = (((((((((((m_Rtd_yz_o0_t0_l0 + m_Rtd_yz_o0_t1_l0) + m_Rtd_yz_o0_t2_l0) + m_Rtd_yz_o0_t3_l0) + m_Rtd_yz_o0_t4_l0) + m_Rtd_yz_o0_t5_l0) + m_Rtd_yz_o0_t6_l0) + m_Rtd_yz_o0_t7_l0) + m_Rtd_yz_o0_t8_l0) + m_Rtd_yz_o0_t9_l0) + m_Rtd_yz_o0_t10_l0) + m_Rtd_yz_o0_t11_l0) + d_Rtd_yz_o0_t12_m0_l0;
							EWeyl_yz = (-inv_chi * Rpd_yz) + (-Rtd_yz) + inv_chi * (unp1(i, j, k, Atd_xy) * Atud_xz + unp1(i, j, k, Atd_yy) * Atud_yz + unp1(i, j, k, Atd_yz) * Atud_zz) + (-.3333333333333333 * inv_chi * unp1(i, j, k, trK) * unp1(i, j, k, Atd_yz)) + (-.2222222222222222 * inv_chi * unp1(i, j, k, gtd_yz) * (unp1(i, j, k, trK) * unp1(i, j, k, trK)));
							d_d_pheI4d_t_o0_t0_m0_l0 = -unp1(i, j, k, Alpha) * unp1(i, j, k, peI);
							m_d_pheI4d_t_o0_t6_l0 = std::min(0.0, unp1(i, j, k, Betau_z)) * d_d_pheI4d_t_o0_t3_m0_l0;
							m_d_pheI4d_t_o0_t5_l0 = std::min(0.0, unp1(i, j, k, Betau_y)) * d_d_pheI4d_t_o0_t2_m0_l0;
							m_d_pheI4d_t_o0_t4_l0 = std::min(0.0, unp1(i, j, k, Betau_x)) * d_d_pheI4d_t_o0_t1_m0_l0;
							m_d_pheI4d_t_o0_t3_l0 = std::max(0.0, unp1(i, j, k, Betau_z)) * d_d_pheI4d_t_o0_t3_m0_l0;
							m_d_pheI4d_t_o0_t2_l0 = std::max(0.0, unp1(i, j, k, Betau_y)) * d_d_pheI4d_t_o0_t2_m0_l0;
							m_d_pheI4d_t_o0_t1_l0 = std::max(0.0, unp1(i, j, k, Betau_x)) * d_d_pheI4d_t_o0_t1_m0_l0;
							d_pheI4d_t = (((((d_d_pheI4d_t_o0_t0_m0_l0 + m_d_pheI4d_t_o0_t1_l0) + m_d_pheI4d_t_o0_t2_l0) + m_d_pheI4d_t_o0_t3_l0) + m_d_pheI4d_t_o0_t4_l0) + m_d_pheI4d_t_o0_t5_l0) + m_d_pheI4d_t_o0_t6_l0;
							d_pheI4d_z = d_d_pheI4d_t_o0_t3_m0_l0;
							d_pheI4d_y = d_d_pheI4d_t_o0_t2_m0_l0;
							d_pheI4d_x = d_d_pheI4d_t_o0_t1_m0_l0;
							d_pheI4u_t = g4u_tt * d_pheI4d_t + g4u_tx * d_pheI4d_x + g4u_ty * d_pheI4d_y + g4u_tz * d_pheI4d_z;
							d_pheI4u_z = g4u_tz * d_pheI4d_t + g4u_xz * d_pheI4d_x + g4u_yz * d_pheI4d_y + g4u_zz * d_pheI4d_z;
							d_pheI4u_y = g4u_ty * d_pheI4d_t + g4u_xy * d_pheI4d_x + g4u_yy * d_pheI4d_y + g4u_yz * d_pheI4d_z;
							d_pheI4u_x = g4u_tx * d_pheI4d_t + g4u_xx * d_pheI4d_x + g4u_xy * d_pheI4d_y + g4u_xz * d_pheI4d_z;
							dphe4sq = d_pheR4u_t * d_pheR4d_t + d_pheR4u_x * d_pheR4d_x + d_pheR4u_y * d_pheR4d_y + d_pheR4u_z * d_pheR4d_z + d_pheI4u_t * d_pheI4d_t + d_pheI4u_x * d_pheI4d_x + d_pheI4u_y * d_pheI4d_y + d_pheI4u_z * d_pheI4d_z;
							Tsfu_tt = 2.0 * (d_phiR4u_t * d_phiR4u_t) + 2.0 * (d_phiI4u_t * d_phiI4u_t) + (-g4u_tt * (dphi4sq + sfVi)) + 2.0 * (d_pheR4u_t * d_pheR4u_t) + 2.0 * (d_pheI4u_t * d_pheI4u_t) + (-g4u_tt * (dphe4sq + sfVe));
							Tu_tt = Tsfu_tt;
							Tsfu_tz = 2.0 * d_phiR4u_t * d_phiR4u_z + 2.0 * d_phiI4u_t * d_phiI4u_z + (-g4u_tz * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_t * d_pheR4u_z + 2.0 * d_pheI4u_t * d_pheI4u_z + (-g4u_tz * (dphe4sq + sfVe));
							Tu_tz = Tsfu_tz;
							Tsfu_ty = 2.0 * d_phiR4u_t * d_phiR4u_y + 2.0 * d_phiI4u_t * d_phiI4u_y + (-g4u_ty * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_t * d_pheR4u_y + 2.0 * d_pheI4u_t * d_pheI4u_y + (-g4u_ty * (dphe4sq + sfVe));
							Tu_ty = Tsfu_ty;
							Tsfu_tx = 2.0 * d_phiR4u_t * d_phiR4u_x + 2.0 * d_phiI4u_t * d_phiI4u_x + (-g4u_tx * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_t * d_pheR4u_x + 2.0 * d_pheI4u_t * d_pheI4u_x + (-g4u_tx * (dphe4sq + sfVe));
							Tu_tx = Tsfu_tx;
							Jtd_ADM_z = unp1(i, j, k, Alpha) * ((unp1(i, j, k, Betau_x) * Tu_tt + Tu_tx) * unp1(i, j, k, gtd_xz) + (unp1(i, j, k, Betau_y) * Tu_tt + Tu_ty) * unp1(i, j, k, gtd_yz) + (unp1(i, j, k, Betau_z) * Tu_tt + Tu_tz) * unp1(i, j, k, gtd_zz));
							Jtd_ADM_y = unp1(i, j, k, Alpha) * ((unp1(i, j, k, Betau_x) * Tu_tt + Tu_tx) * unp1(i, j, k, gtd_xy) + (unp1(i, j, k, Betau_y) * Tu_tt + Tu_ty) * unp1(i, j, k, gtd_yy) + (unp1(i, j, k, Betau_z) * Tu_tt + Tu_tz) * unp1(i, j, k, gtd_yz));
							Jtd_ADM_x = unp1(i, j, k, Alpha) * ((unp1(i, j, k, Betau_x) * Tu_tt + Tu_tx) * unp1(i, j, k, gtd_xx) + (unp1(i, j, k, Betau_y) * Tu_tt + Tu_ty) * unp1(i, j, k, gtd_xy) + (unp1(i, j, k, Betau_z) * Tu_tt + Tu_tz) * unp1(i, j, k, gtd_xz));
							Tsfu_zz = 2.0 * (d_phiR4u_z * d_phiR4u_z) + 2.0 * (d_phiI4u_z * d_phiI4u_z) + (-g4u_zz * (dphi4sq + sfVi)) + 2.0 * (d_pheR4u_z * d_pheR4u_z) + 2.0 * (d_pheI4u_z * d_pheI4u_z) + (-g4u_zz * (dphe4sq + sfVe));
							Tu_zz = Tsfu_zz;
							Tsfu_yz = 2.0 * d_phiR4u_y * d_phiR4u_z + 2.0 * d_phiI4u_y * d_phiI4u_z + (-g4u_yz * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_y * d_pheR4u_z + 2.0 * d_pheI4u_y * d_pheI4u_z + (-g4u_yz * (dphe4sq + sfVe));
							Tu_yz = Tsfu_yz;
							Tsfu_yy = 2.0 * (d_phiR4u_y * d_phiR4u_y) + 2.0 * (d_phiI4u_y * d_phiI4u_y) + (-g4u_yy * (dphi4sq + sfVi)) + 2.0 * (d_pheR4u_y * d_pheR4u_y) + 2.0 * (d_pheI4u_y * d_pheI4u_y) + (-g4u_yy * (dphe4sq + sfVe));
							Tu_yy = Tsfu_yy;
							Tsfu_xz = 2.0 * d_phiR4u_x * d_phiR4u_z + 2.0 * d_phiI4u_x * d_phiI4u_z + (-g4u_xz * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_x * d_pheR4u_z + 2.0 * d_pheI4u_x * d_pheI4u_z + (-g4u_xz * (dphe4sq + sfVe));
							Tu_xz = Tsfu_xz;
							Tsfu_xy = 2.0 * d_phiR4u_x * d_phiR4u_y + 2.0 * d_phiI4u_x * d_phiI4u_y + (-g4u_xy * (dphi4sq + sfVi)) + 2.0 * d_pheR4u_x * d_pheR4u_y + 2.0 * d_pheI4u_x * d_pheI4u_y + (-g4u_xy * (dphe4sq + sfVe));
							Tu_xy = Tsfu_xy;
							Tsfu_xx = 2.0 * (d_phiR4u_x * d_phiR4u_x) + 2.0 * (d_phiI4u_x * d_phiI4u_x) + (-g4u_xx * (dphi4sq + sfVi)) + 2.0 * (d_pheR4u_x * d_pheR4u_x) + 2.0 * (d_pheI4u_x * d_pheI4u_x) + (-g4u_xx * (dphe4sq + sfVe));
							Tu_xx = Tsfu_xx;
							Td_yz = g4d_ty * g4d_tz * Tu_tt + g4d_xy * g4d_tz * Tu_tx + g4d_yy * g4d_tz * Tu_ty + g4d_yz * g4d_tz * Tu_tz + g4d_ty * g4d_xz * Tu_tx + g4d_xy * g4d_xz * Tu_xx + g4d_yy * g4d_xz * Tu_xy + g4d_yz * g4d_xz * Tu_xz + g4d_ty * g4d_yz * Tu_ty + g4d_xy * g4d_yz * Tu_xy + g4d_yy * g4d_yz * Tu_yy + (g4d_yz * g4d_yz) * Tu_yz + g4d_ty * g4d_zz * Tu_tz + g4d_xy * g4d_zz * Tu_xz + g4d_yy * g4d_zz * Tu_yz + g4d_yz * g4d_zz * Tu_zz;
							Td_yy = (g4d_ty * g4d_ty) * Tu_tt + 2.0 * g4d_xy * g4d_ty * Tu_tx + 2.0 * g4d_yy * g4d_ty * Tu_ty + 2.0 * g4d_yz * g4d_ty * Tu_tz + (g4d_xy * g4d_xy) * Tu_xx + 2.0 * g4d_yy * g4d_xy * Tu_xy + 2.0 * g4d_yz * g4d_xy * Tu_xz + (g4d_yy * g4d_yy) * Tu_yy + 2.0 * g4d_yz * g4d_yy * Tu_yz + (g4d_yz * g4d_yz) * Tu_zz;
							Td_xz = g4d_tx * g4d_tz * Tu_tt + g4d_xx * g4d_tz * Tu_tx + g4d_xy * g4d_tz * Tu_ty + g4d_xz * g4d_tz * Tu_tz + g4d_tx * g4d_xz * Tu_tx + g4d_xx * g4d_xz * Tu_xx + g4d_xy * g4d_xz * Tu_xy + (g4d_xz * g4d_xz) * Tu_xz + g4d_tx * g4d_yz * Tu_ty + g4d_xx * g4d_yz * Tu_xy + g4d_xy * g4d_yz * Tu_yy + g4d_xz * g4d_yz * Tu_yz + g4d_tx * g4d_zz * Tu_tz + g4d_xx * g4d_zz * Tu_xz + g4d_xy * g4d_zz * Tu_yz + g4d_xz * g4d_zz * Tu_zz;
							Td_xy = g4d_tx * g4d_ty * Tu_tt + g4d_xx * g4d_ty * Tu_tx + g4d_xy * g4d_ty * Tu_ty + g4d_xz * g4d_ty * Tu_tz + g4d_tx * g4d_xy * Tu_tx + g4d_xx * g4d_xy * Tu_xx + (g4d_xy * g4d_xy) * Tu_xy + g4d_xz * g4d_xy * Tu_xz + g4d_tx * g4d_yy * Tu_ty + g4d_xx * g4d_yy * Tu_xy + g4d_xy * g4d_yy * Tu_yy + g4d_xz * g4d_yy * Tu_yz + g4d_tx * g4d_yz * Tu_tz + g4d_xx * g4d_yz * Tu_xz + g4d_xy * g4d_yz * Tu_yz + g4d_xz * g4d_yz * Tu_zz;
							Td_xx = (g4d_tx * g4d_tx) * Tu_tt + 2.0 * g4d_xx * g4d_tx * Tu_tx + 2.0 * g4d_xy * g4d_tx * Tu_ty + 2.0 * g4d_xz * g4d_tx * Tu_tz + (g4d_xx * g4d_xx) * Tu_xx + 2.0 * g4d_xy * g4d_xx * Tu_xy + 2.0 * g4d_xz * g4d_xx * Tu_xz + (g4d_xy * g4d_xy) * Tu_yy + 2.0 * g4d_xz * g4d_xy * Tu_yz + (g4d_xz * g4d_xz) * Tu_zz;
							Td_zz = (g4d_tz * g4d_tz) * Tu_tt + 2.0 * g4d_xz * g4d_tz * Tu_tx + 2.0 * g4d_yz * g4d_tz * Tu_ty + 2.0 * g4d_zz * g4d_tz * Tu_tz + (g4d_xz * g4d_xz) * Tu_xx + 2.0 * g4d_yz * g4d_xz * Tu_xy + 2.0 * g4d_zz * g4d_xz * Tu_xz + (g4d_yz * g4d_yz) * Tu_yy + 2.0 * g4d_zz * g4d_yz * Tu_yz + (g4d_zz * g4d_zz) * Tu_zz;
							TT = g4d_tt * Tu_tt + 2.0 * g4d_tx * Tu_tx + 2.0 * g4d_ty * Tu_ty + 2.0 * g4d_tz * Tu_tz + g4d_xx * Tu_xx + 2.0 * g4d_xy * Tu_xy + 2.0 * g4d_xz * Tu_xz + g4d_yy * Tu_yy + 2.0 * g4d_yz * Tu_yz + g4d_zz * Tu_zz;
							Td_tx = g4d_tt * g4d_tx * Tu_tt + (g4d_tx * g4d_tx) * Tu_tx + g4d_ty * g4d_tx * Tu_ty + g4d_tz * g4d_tx * Tu_tz + g4d_tt * g4d_xx * Tu_tx + g4d_tx * g4d_xx * Tu_xx + g4d_ty * g4d_xx * Tu_xy + g4d_tz * g4d_xx * Tu_xz + g4d_tt * g4d_xy * Tu_ty + g4d_tx * g4d_xy * Tu_xy + g4d_ty * g4d_xy * Tu_yy + g4d_tz * g4d_xy * Tu_yz + g4d_tt * g4d_xz * Tu_tz + g4d_tx * g4d_xz * Tu_xz + g4d_ty * g4d_xz * Tu_yz + g4d_tz * g4d_xz * Tu_zz;
							Td_tz = g4d_tt * g4d_tz * Tu_tt + g4d_tx * g4d_tz * Tu_tx + g4d_ty * g4d_tz * Tu_ty + (g4d_tz * g4d_tz) * Tu_tz + g4d_tt * g4d_xz * Tu_tx + g4d_tx * g4d_xz * Tu_xx + g4d_ty * g4d_xz * Tu_xy + g4d_tz * g4d_xz * Tu_xz + g4d_tt * g4d_yz * Tu_ty + g4d_tx * g4d_yz * Tu_xy + g4d_ty * g4d_yz * Tu_yy + g4d_tz * g4d_yz * Tu_yz + g4d_tt * g4d_zz * Tu_tz + g4d_tx * g4d_zz * Tu_xz + g4d_ty * g4d_zz * Tu_yz + g4d_tz * g4d_zz * Tu_zz;
							Td_ty = g4d_tt * g4d_ty * Tu_tt + g4d_tx * g4d_ty * Tu_tx + (g4d_ty * g4d_ty) * Tu_ty + g4d_tz * g4d_ty * Tu_tz + g4d_tt * g4d_xy * Tu_tx + g4d_tx * g4d_xy * Tu_xx + g4d_ty * g4d_xy * Tu_xy + g4d_tz * g4d_xy * Tu_xz + g4d_tt * g4d_yy * Tu_ty + g4d_tx * g4d_yy * Tu_xy + g4d_ty * g4d_yy * Tu_yy + g4d_tz * g4d_yy * Tu_yz + g4d_tt * g4d_yz * Tu_tz + g4d_tx * g4d_yz * Tu_xz + g4d_ty * g4d_yz * Tu_yz + g4d_tz * g4d_yz * Tu_zz;
							Td_tt = (g4d_tt * g4d_tt) * Tu_tt + 2.0 * g4d_tx * g4d_tt * Tu_tx + 2.0 * g4d_ty * g4d_tt * Tu_ty + 2.0 * g4d_tz * g4d_tt * Tu_tz + (g4d_tx * g4d_tx) * Tu_xx + 2.0 * g4d_ty * g4d_tx * Tu_xy + 2.0 * g4d_tz * g4d_tx * Tu_xz + (g4d_ty * g4d_ty) * Tu_yy + 2.0 * g4d_tz * g4d_ty * Tu_yz + (g4d_tz * g4d_tz) * Tu_zz;
							rho_ADM = (unp1(i, j, k, Alpha) * unp1(i, j, k, Alpha)) * Tu_tt;
							d_Rscalar_o0_t0_m0_l0 = (unp1(i, j, k, chi) * Rtd_xx + Rpd_xx) * gtu_xx + 2.0 * (unp1(i, j, k, chi) * Rtd_xy + Rpd_xy) * gtu_xy + 2.0 * (unp1(i, j, k, chi) * Rtd_xz + Rpd_xz) * gtu_xz + (unp1(i, j, k, chi) * Rtd_yy + Rpd_yy) * gtu_yy + 2.0 * (unp1(i, j, k, chi) * Rtd_yz + Rpd_yz) * gtu_yz + (unp1(i, j, k, chi) * Rtd_zz + Rpd_zz) * gtu_zz;
							d_HamCon_o0_t0_m0_l0 = (-16.0 * M_PI * rho_ADM) + (unp1(i, j, k, chi) * Rtd_xx + Rpd_xx) * gtu_xx + 2.0 * (unp1(i, j, k, chi) * Rtd_xy + Rpd_xy) * gtu_xy + 2.0 * (unp1(i, j, k, chi) * Rtd_xz + Rpd_xz) * gtu_xz + (unp1(i, j, k, chi) * Rtd_yy + Rpd_yy) * gtu_yy + 2.0 * (unp1(i, j, k, chi) * Rtd_yz + Rpd_yz) * gtu_yz + (unp1(i, j, k, chi) * Rtd_zz + Rpd_zz) * gtu_zz + (-unp1(i, j, k, Atd_xx) * Atu_xx) + (-2.0 * unp1(i, j, k, Atd_xy) * Atu_xy) + (-2.0 * unp1(i, j, k, Atd_xz) * Atu_xz) + (-unp1(i, j, k, Atd_yy) * Atu_yy) + (-2.0 * unp1(i, j, k, Atd_yz) * Atu_yz) + (-unp1(i, j, k, Atd_zz) * Atu_zz) + .6666666666666667 * (unp1(i, j, k, trK) * unp1(i, j, k, trK));
							d_MomCon_x_o0_t13_m0_l0 = (-Gamt_z * unp1(i, j, k, Atd_xz)) + (-Gamt_y * unp1(i, j, k, Atd_xy)) + (-Gamt_x * unp1(i, j, k, Atd_xx)) + (-Ct_zxz * Atud_zz) + (-Ct_zxy * Atud_yz) + (-Ct_zxx * Atud_xz) + (-Ct_yxz * Atud_zy) + (-Ct_yxy * Atud_yy) + (-Ct_yxx * Atud_xy) + (-Ct_xxz * Atud_zx) + (-Ct_xxy * Atud_yx) + (-Ct_xxx * Atud_xx) + (-8.0 * M_PI * Jtd_ADM_x * inv_chi);
							d_MomCon_y_o0_t13_m0_l0 = (-Gamt_z * unp1(i, j, k, Atd_yz)) + (-Gamt_y * unp1(i, j, k, Atd_yy)) + (-Gamt_x * unp1(i, j, k, Atd_xy)) + (-Ct_zyz * Atud_zz) + (-Ct_zyy * Atud_yz) + (-Ct_zxy * Atud_xz) + (-Ct_yyz * Atud_zy) + (-Ct_yyy * Atud_yy) + (-Ct_yxy * Atud_xy) + (-Ct_xyz * Atud_zx) + (-Ct_xyy * Atud_yx) + (-Ct_xxy * Atud_xx) + (-8.0 * M_PI * Jtd_ADM_y * inv_chi);
							d_MomCon_z_o0_t13_m0_l0 = (-Ct_yzz * Atud_zy) + (-Ct_yyz * Atud_yy) + (-Ct_yxz * Atud_xy) + (-Ct_xzz * Atud_zx) + (-Ct_xyz * Atud_yx) + (-Ct_xxz * Atud_xx) + (-8.0 * M_PI * Jtd_ADM_z * inv_chi) + (-Gamt_z * unp1(i, j, k, Atd_zz)) + (-Gamt_y * unp1(i, j, k, Atd_yz)) + (-Gamt_x * unp1(i, j, k, Atd_xz)) + (-Ct_zzz * Atud_zz) + (-Ct_zyz * Atud_yz) + (-Ct_zxz * Atud_xz);
							d_trA_o0_t0_m0_l0 = Atud_xx + Atud_yy + Atud_zz;
							d_detgtm1_o0_t0_m0_l0 = detgtd + (-1.0);
							d_M_ADM_surf_o0_t9_m0_l0 = .1989436788648692e-1 * unp1(i, j, k, chi) * Gamt_z * dSigma_z + .1989436788648692e-1 * unp1(i, j, k, chi) * Gamt_y * dSigma_y + .1989436788648692e-1 * unp1(i, j, k, chi) * Gamt_x * dSigma_x;
							d_Jz_ADM_surf_o0_t0_m0_l0 = .3978873577297384e-1 * unp1(i, j, k, chi) * (uph_x * (.3333333333333333 * unp1(i, j, k, trK) * gtu_xx + Atu_xx) + uph_y * (.3333333333333333 * unp1(i, j, k, trK) * gtu_xy + Atu_xy) + uph_z * (.3333333333333333 * unp1(i, j, k, trK) * gtu_xz + Atu_xz)) * dSigma_x + .3978873577297384e-1 * unp1(i, j, k, chi) * (uph_x * (.3333333333333333 * unp1(i, j, k, trK) * gtu_xy + Atu_xy) + uph_y * (.3333333333333333 * unp1(i, j, k, trK) * gtu_yy + Atu_yy) + uph_z * (.3333333333333333 * unp1(i, j, k, trK) * gtu_yz + Atu_yz)) * dSigma_y + .3978873577297384e-1 * unp1(i, j, k, chi) * (uph_x * (.3333333333333333 * unp1(i, j, k, trK) * gtu_xz + Atu_xz) + uph_y * (.3333333333333333 * unp1(i, j, k, trK) * gtu_yz + Atu_yz) + uph_z * (.3333333333333333 * unp1(i, j, k, trK) * gtu_zz + Atu_zz)) * dSigma_z;
							d_N_Noetheri_o0_t0_m0_l0 = -2.0 * (1. * unp1(i, j, k, phiI) * unp1(i, j, k, piR) - 1. * unp1(i, j, k, phiR) * unp1(i, j, k, piI)) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							d_N_Noethere_o0_t0_m0_l0 = -2.0 * ((-1. * unp1(i, j, k, peI) * unp1(i, j, k, pheR)) + 1. * unp1(i, j, k, peR) * unp1(i, j, k, pheI)) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							d_M_Komar_o0_t0_m0_l0 = 2.0 * (((-.5 * TT * g4d_tt) + Td_tt) / unp1(i, j, k, Alpha) + (-(((-.5 * TT * g4d_tx) + Td_tx) * unp1(i, j, k, Betau_x)) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_ty) + Td_ty) * unp1(i, j, k, Betau_y)) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_tz) + Td_tz) * unp1(i, j, k, Betau_z)) / unp1(i, j, k, Alpha))) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							d_Jz_Komar_o0_t0_m0_l0 = -((((-.5 * TT * g4d_tx) + Td_tx) * vph_x) / unp1(i, j, k, Alpha) + (-(((-.5 * TT * g4d_xx) + Td_xx) * unp1(i, j, k, Betau_x) * vph_x) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_xy) + Td_xy) * unp1(i, j, k, Betau_y) * vph_x) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_xz) + Td_xz) * unp1(i, j, k, Betau_z) * vph_x) / unp1(i, j, k, Alpha)) + (((-.5 * TT * g4d_ty) + Td_ty) * vph_y) / unp1(i, j, k, Alpha) + (-(((-.5 * TT * g4d_xy) + Td_xy) * unp1(i, j, k, Betau_x) * vph_y) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_yy) + Td_yy) * unp1(i, j, k, Betau_y) * vph_y) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_yz) + Td_yz) * unp1(i, j, k, Betau_z) * vph_y) / unp1(i, j, k, Alpha)) + (((-.5 * TT * g4d_tz) + Td_tz) * vph_z) / unp1(i, j, k, Alpha) + (-(((-.5 * TT * g4d_xz) + Td_xz) * unp1(i, j, k, Betau_x) * vph_z) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_yz) + Td_yz) * unp1(i, j, k, Betau_y) * vph_z) / unp1(i, j, k, Alpha)) + (-(((-.5 * TT * g4d_zz) + Td_zz) * unp1(i, j, k, Betau_z) * vph_z) / unp1(i, j, k, Alpha))) / (pow(unp1(i, j, k, chi), 1.500000000000000));
							d_psi4R_o0_t0_m0_l0 = std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k))) * ((-BWeyl_xx * mmI_xx) + EWeyl_xx * mmR_xx + (-BWeyl_yx * mmI_xy) + 2.0 * EWeyl_xy * mmR_xy + (-BWeyl_zx * mmI_xz) + 2.0 * EWeyl_xz * mmR_xz + (-BWeyl_xy * mmI_xy) + (-BWeyl_yy * mmI_yy) + EWeyl_yy * mmR_yy + (-BWeyl_zy * mmI_yz) + 2.0 * EWeyl_yz * mmR_yz + (-BWeyl_xz * mmI_xz) + (-BWeyl_yz * mmI_yz) + (-BWeyl_zz * mmI_zz) + EWeyl_zz * mmR_zz);
							d_psi4I_o0_t0_m0_l0 = std::max(.1e-9, sqrt(xcoord(i) * xcoord(i) + ycoord(j) * ycoord(j) + zcoord(k) * zcoord(k))) * (BWeyl_xx * mmR_xx + EWeyl_xx * mmI_xx + BWeyl_yx * mmR_xy + 2.0 * EWeyl_xy * mmI_xy + BWeyl_zx * mmR_xz + 2.0 * EWeyl_xz * mmI_xz + BWeyl_xy * mmR_xy + BWeyl_yy * mmR_yy + EWeyl_yy * mmI_yy + BWeyl_zy * mmR_yz + 2.0 * EWeyl_yz * mmI_yz + BWeyl_xz * mmR_xz + BWeyl_yz * mmR_yz + BWeyl_zz * mmR_zz + EWeyl_zz * mmI_zz);
							d_Z_x_o0_t0_m0_l0 = Zu_x;
							d_Z_y_o0_t0_m0_l0 = Zu_y;
							d_Z_z_o0_t0_m0_l0 = Zu_z;
							m_M_ADM_surf_o0_t8_l0 = .1250000000000000 * (gtu_zz * dSigma_z) / M_PI * d_Rpd_xx_o0_t2_m0_l0;
							m_M_ADM_surf_o0_t7_l0 = .1250000000000000 * (gtu_yz * dSigma_z) / M_PI * d_Rpd_xx_o0_t1_m0_l0;
							m_M_ADM_surf_o0_t6_l0 = .1250000000000000 * (gtu_xz * dSigma_z) / M_PI * d_Rpd_xx_o0_t0_m0_l0;
							m_M_ADM_surf_o0_t5_l0 = .1250000000000000 * (gtu_yz * dSigma_y) / M_PI * d_Rpd_xx_o0_t2_m0_l0;
							m_M_ADM_surf_o0_t4_l0 = .1250000000000000 * (gtu_yy * dSigma_y) / M_PI * d_Rpd_xx_o0_t1_m0_l0;
							m_M_ADM_surf_o0_t3_l0 = .1250000000000000 * (gtu_xy * dSigma_y) / M_PI * d_Rpd_xx_o0_t0_m0_l0;
							m_M_ADM_surf_o0_t2_l0 = .1250000000000000 * (gtu_xz * dSigma_x) / M_PI * d_Rpd_xx_o0_t2_m0_l0;
							m_M_ADM_surf_o0_t1_l0 = .1250000000000000 * (gtu_xy * dSigma_x) / M_PI * d_Rpd_xx_o0_t1_m0_l0;
							m_M_ADM_surf_o0_t0_l0 = .1250000000000000 * (gtu_xx * dSigma_x) / M_PI * d_Rpd_xx_o0_t0_m0_l0;
							m_MomCon_z_o0_t12_l0 = (-.6666666666666667) * d_Del_Kd_zxx_o0_t2_m0_l0;
							m_MomCon_z_o0_t11_l0 = gtu_zz * d_MomCon_z_o0_t11_m0_l0;
							m_MomCon_z_o0_t10_l0 = gtu_yz * d_Del_Kd_zyz_o0_t0_m0_l0;
							m_MomCon_z_o0_t9_l0 = gtu_xz * d_Del_Kd_zxz_o0_t0_m0_l0;
							m_MomCon_z_o0_t8_l0 = gtu_yz * d_Del_Kd_yzz_o0_t0_m0_l0;
							m_MomCon_z_o0_t7_l0 = gtu_yy * d_Del_Kd_yyz_o0_t0_m0_l0;
							m_MomCon_z_o0_t6_l0 = gtu_xy * d_Del_Kd_yxz_o0_t0_m0_l0;
							m_MomCon_z_o0_t5_l0 = gtu_xz * d_Del_Kd_xzz_o0_t0_m0_l0;
							m_MomCon_z_o0_t4_l0 = gtu_xy * d_Del_Kd_xyz_o0_t0_m0_l0;
							m_MomCon_z_o0_t3_l0 = gtu_xx * d_Del_Kd_xxz_o0_t0_m0_l0;
							m_MomCon_z_o0_t2_l0 = (-1.500000000000000 * inv_chi * Atud_zz) * d_Rpd_xx_o0_t2_m0_l0;
							m_MomCon_z_o0_t1_l0 = (-1.500000000000000 * inv_chi * Atud_yz) * d_Rpd_xx_o0_t1_m0_l0;
							m_MomCon_z_o0_t0_l0 = (-1.500000000000000 * inv_chi * Atud_xz) * d_Rpd_xx_o0_t0_m0_l0;
							m_MomCon_y_o0_t12_l0 = (-.6666666666666667) * d_Del_Kd_yxx_o0_t2_m0_l0;
							m_MomCon_y_o0_t11_l0 = gtu_zz * d_Del_Kd_zyz_o0_t0_m0_l0;
							m_MomCon_y_o0_t10_l0 = gtu_yz * d_Del_Kd_zyy_o0_t0_m0_l0;
							m_MomCon_y_o0_t9_l0 = gtu_xz * d_Del_Kd_zxy_o0_t0_m0_l0;
							m_MomCon_y_o0_t8_l0 = gtu_yz * d_Del_Kd_yyz_o0_t0_m0_l0;
							m_MomCon_y_o0_t7_l0 = gtu_yy * d_MomCon_y_o0_t7_m0_l0;
							m_MomCon_y_o0_t6_l0 = gtu_xy * d_Del_Kd_yxy_o0_t0_m0_l0;
							m_MomCon_y_o0_t5_l0 = gtu_xz * d_Del_Kd_xyz_o0_t0_m0_l0;
							m_MomCon_y_o0_t4_l0 = gtu_xy * d_Del_Kd_xyy_o0_t0_m0_l0;
							m_MomCon_y_o0_t3_l0 = gtu_xx * d_Del_Kd_xxy_o0_t0_m0_l0;
							m_MomCon_y_o0_t2_l0 = (-1.500000000000000 * inv_chi * Atud_yy) * d_Rpd_xx_o0_t1_m0_l0;
							m_MomCon_y_o0_t1_l0 = (-1.500000000000000 * inv_chi * Atud_zy) * d_Rpd_xx_o0_t2_m0_l0;
							m_MomCon_y_o0_t0_l0 = (-1.500000000000000 * inv_chi * Atud_xy) * d_Rpd_xx_o0_t0_m0_l0;
							m_MomCon_x_o0_t12_l0 = (-.6666666666666667) * d_Del_Kd_xxy_o0_t2_m0_l0;
							m_MomCon_x_o0_t11_l0 = gtu_zz * d_Del_Kd_zxz_o0_t0_m0_l0;
							m_MomCon_x_o0_t10_l0 = gtu_yz * d_Del_Kd_zxy_o0_t0_m0_l0;
							m_MomCon_x_o0_t9_l0 = gtu_xz * d_Del_Kd_zxx_o0_t0_m0_l0;
							m_MomCon_x_o0_t8_l0 = gtu_yz * d_Del_Kd_yxz_o0_t0_m0_l0;
							m_MomCon_x_o0_t7_l0 = gtu_yy * d_Del_Kd_yxy_o0_t0_m0_l0;
							m_MomCon_x_o0_t6_l0 = gtu_xy * d_Del_Kd_yxx_o0_t0_m0_l0;
							m_MomCon_x_o0_t5_l0 = gtu_xz * d_Del_Kd_xxz_o0_t0_m0_l0;
							m_MomCon_x_o0_t4_l0 = gtu_xy * d_Del_Kd_xxy_o0_t0_m0_l0;
							m_MomCon_x_o0_t3_l0 = gtu_xx * d_MomCon_x_o0_t3_m0_l0;
							m_MomCon_x_o0_t2_l0 = (-1.500000000000000 * inv_chi * Atud_yx) * d_Rpd_xx_o0_t1_m0_l0;
							m_MomCon_x_o0_t1_l0 = (-1.500000000000000 * inv_chi * Atud_xx) * d_Rpd_xx_o0_t0_m0_l0;
							m_MomCon_x_o0_t0_l0 = (-1.500000000000000 * inv_chi * Atud_zx) * d_Rpd_xx_o0_t2_m0_l0;
							analysisGroup(i, j, k, Rscalar) = d_Rscalar_o0_t0_m0_l0;
							analysisGroup(i, j, k, HamCon) = d_HamCon_o0_t0_m0_l0;
							analysisGroup(i, j, k, MomCon_x) = ((((((((((((m_MomCon_x_o0_t0_l0 + m_MomCon_x_o0_t1_l0) + m_MomCon_x_o0_t2_l0) + m_MomCon_x_o0_t3_l0) + m_MomCon_x_o0_t4_l0) + m_MomCon_x_o0_t5_l0) + m_MomCon_x_o0_t6_l0) + m_MomCon_x_o0_t7_l0) + m_MomCon_x_o0_t8_l0) + m_MomCon_x_o0_t9_l0) + m_MomCon_x_o0_t10_l0) + m_MomCon_x_o0_t11_l0) + m_MomCon_x_o0_t12_l0) + d_MomCon_x_o0_t13_m0_l0;
							analysisGroup(i, j, k, MomCon_y) = ((((((((((((m_MomCon_y_o0_t0_l0 + m_MomCon_y_o0_t1_l0) + m_MomCon_y_o0_t2_l0) + m_MomCon_y_o0_t3_l0) + m_MomCon_y_o0_t4_l0) + m_MomCon_y_o0_t5_l0) + m_MomCon_y_o0_t6_l0) + m_MomCon_y_o0_t7_l0) + m_MomCon_y_o0_t8_l0) + m_MomCon_y_o0_t9_l0) + m_MomCon_y_o0_t10_l0) + m_MomCon_y_o0_t11_l0) + m_MomCon_y_o0_t12_l0) + d_MomCon_y_o0_t13_m0_l0;
							analysisGroup(i, j, k, MomCon_z) = ((((((((((((m_MomCon_z_o0_t0_l0 + m_MomCon_z_o0_t1_l0) + m_MomCon_z_o0_t2_l0) + m_MomCon_z_o0_t3_l0) + m_MomCon_z_o0_t4_l0) + m_MomCon_z_o0_t5_l0) + m_MomCon_z_o0_t6_l0) + m_MomCon_z_o0_t7_l0) + m_MomCon_z_o0_t8_l0) + m_MomCon_z_o0_t9_l0) + m_MomCon_z_o0_t10_l0) + m_MomCon_z_o0_t11_l0) + m_MomCon_z_o0_t12_l0) + d_MomCon_z_o0_t13_m0_l0;
							analysisGroup(i, j, k, trA) = d_trA_o0_t0_m0_l0;
							analysisGroup(i, j, k, detgtm1) = d_detgtm1_o0_t0_m0_l0;
							analysisGroup(i, j, k, M_ADM_surf) = ((((((((m_M_ADM_surf_o0_t0_l0 + m_M_ADM_surf_o0_t1_l0) + m_M_ADM_surf_o0_t2_l0) + m_M_ADM_surf_o0_t3_l0) + m_M_ADM_surf_o0_t4_l0) + m_M_ADM_surf_o0_t5_l0) + m_M_ADM_surf_o0_t6_l0) + m_M_ADM_surf_o0_t7_l0) + m_M_ADM_surf_o0_t8_l0) + d_M_ADM_surf_o0_t9_m0_l0;
							analysisGroup(i, j, k, Jz_ADM_surf) = d_Jz_ADM_surf_o0_t0_m0_l0;
							analysisGroup(i, j, k, N_Noetheri) = d_N_Noetheri_o0_t0_m0_l0;
							analysisGroup(i, j, k, N_Noethere) = d_N_Noethere_o0_t0_m0_l0;
							analysisGroup(i, j, k, M_Komar) = d_M_Komar_o0_t0_m0_l0;
							analysisGroup(i, j, k, Jz_Komar) = d_Jz_Komar_o0_t0_m0_l0;
							analysisGroup(i, j, k, psi4R) = d_psi4R_o0_t0_m0_l0;
							analysisGroup(i, j, k, psi4I) = d_psi4I_o0_t0_m0_l0;
							analysisGroup(i, j, k, Z_x) = d_Z_x_o0_t0_m0_l0;
							analysisGroup(i, j, k, Z_y) = d_Z_y_o0_t0_m0_l0;
							analysisGroup(i, j, k, Z_z) = d_Z_z_o0_t0_m0_l0;
						}
					});
				}
			}
		}

}


void
AmrCoreProblem::calculateExtrapolationVars(Real time, const int lev, MultiFab& regions_mf)
{
	#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
	for (MFIter mfi(regions_mf,TilingIfNotGPU()); mfi.isValid(); ++mfi) {
		Array4<Real> regions = regions_mf[mfi].array();
		const Box& vbx = mfi.validbox();
		const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
		const Box& bx = mfi.tilebox();
		const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);
		amrex::ParallelFor(gbx, [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {
			regions(i, j, k, stalled_1) = calculateExtrapolationDistance(gvbx, regions, i, j, k, FOV_1);
			regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = 0;
			regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = 0;
			regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = 0;
		});
	}
	FillPatch(lev, time, regions_mf, regions_var, NumRegions, std::set<int>(), true);
#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
	for (MFIter mfi(regions_mf,TilingIfNotGPU()); mfi.isValid(); ++mfi) {
		Array4<Real> regions = regions_mf[mfi].array();
		const Box& vbx = mfi.validbox();
		const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
		const Box& bx = mfi.tilebox();
		const Box& gbx = mfi.growntilebox(NUM_GHOST_CELLS);
		const auto bx_lo = amrex::lbound(gvbx);
		const auto bx_hi = amrex::ubound(gvbx);
		int reGrid_i = 1;
		if (lev > 0) {
			reGrid_i = refRatio(lev-1)[0];
		}
		int reGrid_j = 1;
		if (lev > 0) {
			reGrid_j = refRatio(lev-1)[1];
		}
		int reGrid_k = 1;
		if (lev > 0) {
			reGrid_k = refRatio(lev-1)[2];
		}

		amrex::ParallelFor(gbx, [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {
			int dist_i_tmp, dist_i_tmp_p, dist_i_tmp_m, dist_i_tmp_r;
			int dist_j_tmp, dist_j_tmp_p, dist_j_tmp_m, dist_j_tmp_r;
			int dist_k_tmp, dist_k_tmp_p, dist_k_tmp_m, dist_k_tmp_r;
			int dist_r, dist_r_tmp;

			if (regions(i, j, k, FOV_zUpper) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
			if (regions(i, j, k, FOV_zLower) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
			if (regions(i, j, k, FOV_yUpper) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
			if (regions(i, j, k, FOV_yLower) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
			if (regions(i, j, k, FOV_xUpper) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
			if (regions(i, j, k, FOV_xLower) > 0) {
				dist_i_tmp = 999;
				dist_i_tmp_p = 999;
				dist_i_tmp_m = 999;
				dist_i_tmp_r = 999;
				dist_j_tmp = 999;
				dist_j_tmp_p = 999;
				dist_j_tmp_m = 999;
				dist_j_tmp_r = 999;
				dist_k_tmp = 999;
				dist_k_tmp_p = 999;
				dist_k_tmp_m = 999;
				dist_k_tmp_r = 999;
				dist_r = 999;
				for (int Dist_i = -3 * reGrid_i; Dist_i <= 3 * reGrid_i; Dist_i++) {
					for (int Dist_j = -3 * reGrid_j; Dist_j <= 3 * reGrid_j; Dist_j++) {
						for (int Dist_k = -3 * reGrid_k; Dist_k <= 3 * reGrid_k; Dist_k++) {
							if (i + Dist_i >= bx_lo.x && i + Dist_i <= bx_hi.x && j + Dist_j >= bx_lo.y && j + Dist_j <= bx_hi.y && k + Dist_k >= bx_lo.z && k + Dist_k <= bx_hi.z && (!regions(i + Dist_i, j + Dist_j, k + Dist_k, stalled_1))) {
								if (Dist_i < 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_m)) {
									dist_i_tmp_m = -Dist_i;
								}
								if (Dist_i > 0 && Dist_j == 0 && Dist_k == 0 && fabs(Dist_i) < fabs(dist_i_tmp_p)) {
									dist_i_tmp_p = -Dist_i;
								}
								if (Dist_j < 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_m)) {
									dist_j_tmp_m = -Dist_j;
								}
								if (Dist_j > 0 && Dist_i == 0 && Dist_k == 0 && fabs(Dist_j) < fabs(dist_j_tmp_p)) {
									dist_j_tmp_p = -Dist_j;
								}
								if (Dist_k < 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_m)) {
									dist_k_tmp_m = -Dist_k;
								}
								if (Dist_k > 0 && Dist_i == 0 && Dist_j == 0 && fabs(Dist_k) < fabs(dist_k_tmp_p)) {
									dist_k_tmp_p = -Dist_k;
								}
								dist_r_tmp = Dist_i * Dist_i + Dist_j * Dist_j + Dist_k * Dist_k;
								if (dist_r_tmp < dist_r) {
									dist_r = dist_r_tmp;
									dist_i_tmp_r = -Dist_i;
									dist_j_tmp_r = -Dist_j;
									dist_k_tmp_r = -Dist_k;
								}
							}
						}
					}
				}
				if (fabs(dist_i_tmp_m) == fabs(dist_i_tmp_p) && dist_i_tmp_m < 999) {
					dist_i_tmp = 0;
					bool enough = true;
					for (int Dist_i = 0; Dist_i <= 2 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_p + Dist_i <= bx_hi.x && (!regions(i - dist_i_tmp_p + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_p;
					}
					enough = true;
					for (int Dist_i = -2; Dist_i <= 0 && enough; Dist_i++) {
						if (!(i - dist_i_tmp_m + Dist_i >= bx_lo.x && (!regions(i - dist_i_tmp_m + Dist_i, j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_i_tmp = dist_i_tmp_m;
					}
				} else if (fabs(dist_i_tmp_m) < fabs(dist_i_tmp_p)) {
					dist_i_tmp = dist_i_tmp_m;
				} else {
					dist_i_tmp = dist_i_tmp_p;
				}
				if (fabs(dist_j_tmp_m) == fabs(dist_j_tmp_p) && dist_j_tmp_m < 999) {
					dist_j_tmp = 0;
					bool enough = true;
					for (int Dist_j = 0; Dist_j <= 2 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_p + Dist_j <= bx_hi.y && (!regions(i, j - dist_j_tmp_p + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_p;
					}
					enough = true;
					for (int Dist_j = -2; Dist_j <= 0 && enough; Dist_j++) {
						if (!(j - dist_j_tmp_m + Dist_j >= bx_lo.y && (!regions(i, j - dist_j_tmp_m + Dist_j, k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_j_tmp = dist_j_tmp_m;
					}
				} else if (fabs(dist_j_tmp_m) < fabs(dist_j_tmp_p)) {
					dist_j_tmp = dist_j_tmp_m;
				} else {
					dist_j_tmp = dist_j_tmp_p;
				}
				if (fabs(dist_k_tmp_m) == fabs(dist_k_tmp_p) && dist_k_tmp_m < 999) {
					dist_k_tmp = 0;
					bool enough = true;
					for (int Dist_k = 0; Dist_k <= 2 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_p + Dist_k <= bx_hi.z && (!regions(i, j, k - dist_k_tmp_p + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_p;
					}
					enough = true;
					for (int Dist_k = -2; Dist_k <= 0 && enough; Dist_k++) {
						if (!(k - dist_k_tmp_m + Dist_k >= bx_lo.z && (!regions(i, j, k - dist_k_tmp_m + Dist_k, stalled_1)))) {
							enough = false;
						}
					}
					if (enough) {
						dist_k_tmp = dist_k_tmp_m;
					}
				} else if (fabs(dist_k_tmp_m) < fabs(dist_k_tmp_p)) {
					dist_k_tmp = dist_k_tmp_m;
				} else {
					dist_k_tmp = dist_k_tmp_p;
				}
				if (dist_i_tmp == 999 && dist_j_tmp == 999 && dist_k_tmp == 999) {
					dist_i_tmp = dist_i_tmp_r;
					dist_j_tmp = dist_j_tmp_r;
					dist_k_tmp = dist_k_tmp_r;
				}
				if (dist_i_tmp != 999 && ((regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_i_tmp) < fabs(regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_i_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_i_tmp;
				}
				if (dist_j_tmp != 999 && ((regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_j_tmp) < fabs(regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_j_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_j_tmp;
				}
				if (dist_k_tmp != 999 && ((regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) == 0) || (fabs(dist_k_tmp) < fabs(regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI))))) {
					regions(i, j, k, d_k_phi2_gtd_xx_gtd_xy_gtd_xz_gtd_yy_gtd_yz_gtd_zz_Atd_xx_Atd_xy_Atd_xz_Atd_yy_Atd_yz_Atd_zz_Gamh_x_Gamh_y_Betau_x_Betau_y_Betau_z_Gamh_z_Alpha_chi_trK_theta_phiR_pheR_phiI_pheI_piR_peR_piI_peI) = dist_k_tmp;
				}
			}
		});
	}
	FillPatch(lev, time, regions_mf, regions_var, NumRegions, std::set<int>(), true);
}


/*
 * Calculates distance to region if extrapolation is needed
 */
AMREX_GPU_DEVICE
AMREX_FORCE_INLINE
bool AmrCoreProblem::calculateExtrapolationDistance(const Box& gbx, Array4<Real> regions, int i, int j, int k, int FOV) {
	const auto bx_lo = amrex::lbound(gbx);
	const auto bx_hi = amrex::ubound(gbx);
	int stencilAcc, stencilAccMax_i, stencilAccMax_j, stencilAccMax_k;
	bool notEnoughStencil = false;
	int FOV_threshold = 0;
	if (regions(i, j, k, FOV) <= FOV_threshold) {
		notEnoughStencil = true;
	} else {
		stencilAcc = 0;
		stencilAccMax_i = 0;
		for (int it1 = MAX(i-REGION_THICKNESS, bx_lo.x); it1 <= MIN(i+REGION_THICKNESS, bx_hi.x); it1++) {
			if (regions(it1, j, k, FOV) > FOV_threshold) {
				stencilAcc++;
			} else {
				stencilAccMax_i = MAX(stencilAccMax_i, stencilAcc);
				stencilAcc = 0;
			}
		}
		stencilAccMax_i = MAX(stencilAccMax_i, stencilAcc);
		stencilAcc = 0;
		stencilAccMax_j = 0;
		for (int jt1 = MAX(j-REGION_THICKNESS, bx_lo.y); jt1 <= MIN(j+REGION_THICKNESS, bx_hi.y); jt1++) {
			if (regions(i, jt1, k, FOV) > FOV_threshold) {
				stencilAcc++;
			} else {
				stencilAccMax_j = MAX(stencilAccMax_j, stencilAcc);
				stencilAcc = 0;
			}
		}
		stencilAccMax_j = MAX(stencilAccMax_j, stencilAcc);
		stencilAcc = 0;
		stencilAccMax_k = 0;
		for (int kt1 = MAX(k-REGION_THICKNESS, bx_lo.z); kt1 <= MIN(k+REGION_THICKNESS, bx_hi.z); kt1++) {
			if (regions(i, j, kt1, FOV) > FOV_threshold) {
				stencilAcc++;
			} else {
				stencilAccMax_k = MAX(stencilAccMax_k, stencilAcc);
				stencilAcc = 0;
			}
		}
		stencilAccMax_k = MAX(stencilAccMax_k, stencilAcc);
		if ((stencilAccMax_i < REGION_THICKNESS) || (stencilAccMax_j < REGION_THICKNESS) || (stencilAccMax_k < REGION_THICKNESS)) {
			notEnoughStencil = true;
		}
	}
	return notEnoughStencil;
}




//Map from regions into mesh cells
void
AmrCoreProblem::mapDataOnPatch(Real time, const int lev, MultiFab& regions_mf)
{
#ifdef AMREX_USE_OMP
#pragma omp parallel if (Gpu::notInLaunchRegion())
#endif
    for (MFIter mfi(regions_mf,TilingIfNotGPU()); mfi.isValid(); ++mfi)
    {
        Array4<Real> regions = regions_mf[mfi].array();
        const Box& vbx = mfi.validbox();
        const Box& gvbx = amrex::grow(vbx, NUM_GHOST_CELLS);
        const Box& bx = mfi.tilebox();
        const Box& gbx = amrex::grow(bx, NUM_GHOST_CELLS);

        const auto& domain = geom[lev].Domain();
        const auto& dom_lo = amrex::lbound(domain);
        const auto& dom_hi = amrex::ubound(domain);

        amrex::ParallelFor(gbx, 
            [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept {
			if(k <= dom_hi.z + 1 && k >= dom_lo.z && j <= dom_hi.y + 1 && j >= dom_lo.y && i <= dom_hi.x + 1 && i >= dom_lo.x) {
				regions(i, j, k, FOV_1) = 100;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zLower) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}
			if (k > dom_hi.z + 1) {
				regions(i, j, k, FOV_zUpper) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zLower) = 0;
			}
			if (k < dom_lo.z) {
				regions(i, j, k, FOV_zLower) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}
			if (j > dom_hi.y + 1) {
				regions(i, j, k, FOV_yUpper) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_zLower) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}
			if (j < dom_lo.y) {
				regions(i, j, k, FOV_yLower) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zLower) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}
			if (i > dom_hi.x + 1) {
				regions(i, j, k, FOV_xUpper) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xLower) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zLower) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}
			if (i < dom_lo.x) {
				regions(i, j, k, FOV_xLower) = 100;
				regions(i, j, k, FOV_1) = 0;
				regions(i, j, k, FOV_xUpper) = 0;
				regions(i, j, k, FOV_yLower) = 0;
				regions(i, j, k, FOV_yUpper) = 0;
				regions(i, j, k, FOV_zLower) = 0;
				regions(i, j, k, FOV_zUpper) = 0;
			}

        });
    }
    FillPatch(lev, time, regions_mf, regions_var, NumRegions, std::set<int>(), true);
}

void
AmrCoreProblem::InitFromScratch (Real time)
{
    
    // define coarse level BoxArray and DistributionMap
    {
        finest_level = 0;

        //Create level 0 grid
        const BoxArray& ba = MakeBaseGrids();
        DistributionMapping dm(ba);
        const auto old_num_setdm = num_setdm;
        const auto old_num_setba = num_setba;

        //Initialize level 0 grid
        MakeNewLevelFromScratch(0, time, ba, dm);

        if (old_num_setba == num_setba) {
            SetBoxArray(0, ba);
        }
        if (old_num_setdm == num_setdm) {
            SetDistributionMap(0, dm);
        }
    }

    use_fixed_coarse_grids = fmr_levels > 0;
    use_fixed_upto_level = fmr_levels;

    if (max_level > 0) // build fine levels
    {
        Vector<BoxArray> new_grids(max_level+1);
        new_grids[0] = grids[0];
        do
        {
            int new_finest;

            if (finest_level < fmr_levels) {
                Box bx = Box(IntVect(fmr_lo[finest_level]), IntVect(fmr_hi[finest_level]));
                bx.refine(refRatio(finest_level));
                BoxList bl;
                bl.push_back(bx);
                new_grids[finest_level + 1].define(bl);
            }

            // Add (at most) one level at a time.
            MakeNewGrids(finest_level,time,new_finest,new_grids);

            if (new_finest <= finest_level) break;
            finest_level = new_finest;

            DistributionMapping dm(new_grids[finest_level]);
            const auto old_num_setdm = num_setdm;

            MakeNewLevelFromScratch(new_finest, time, new_grids[finest_level], dm);

            SetBoxArray(new_finest, new_grids[new_finest]);
            if (old_num_setdm == num_setdm) {
                SetDistributionMap(new_finest, dm);
            }
        }
        while (finest_level < max_level);

        // Iterate grids to ensure fine grids encompass all interesting junk.
        iterate_on_new_grids = true;
        if (iterate_on_new_grids)
        {
            for (int it=0; it<4; ++it)  // try at most 4 times
            {
                for (int i = 1; i <= finest_level; ++i) {
                    new_grids[i] = grids[i];
                }

                int new_finest;
                MakeNewGrids(0, time, new_finest, new_grids);

                if (new_finest < finest_level) break;
                finest_level = new_finest;

                bool grids_the_same = true;
                for (int lev = 1; lev <= new_finest; ++lev) {
                    if (new_grids[lev] != grids[lev]) {
                        grids_the_same = false;
                        DistributionMapping dm(new_grids[lev]);
                        const auto old_num_setdm = num_setdm;

                        MakeNewLevelFromScratch(lev, time, new_grids[lev], dm);

                        SetBoxArray(lev, new_grids[lev]);
                        if (old_num_setdm == num_setdm) {
                            SetDistributionMap(lev, dm);
                        }
                    }
                }
                if (grids_the_same) break;
            }
        }
    }
}

void
AmrCoreProblem::regrid (int lbase, Real time, bool)
{
    if (lbase >= max_level) return;

    use_fixed_coarse_grids = fmr_levels > 0;
    use_fixed_upto_level = fmr_levels;

    int new_finest;
    Vector<BoxArray> new_grids(finest_level+2);

    for (int i = 0; i < fmr_levels; i++) {
        Box bx = Box(IntVect(fmr_lo[i]), IntVect(fmr_hi[i]));
        bx.refine(refRatio(i));
        BoxList bl;
        bl.push_back(bx);
        new_grids[i + 1].define(bl);
    }

    MakeNewGrids(lbase, time, new_finest, new_grids);

    BL_ASSERT(new_finest <= finest_level+1);

    bool coarse_ba_changed = false;
    for (int lev = use_fixed_upto_level+1; lev <= new_finest; ++lev)
    {
        if (lev <= finest_level) // an old level
        {
            bool ba_changed = (new_grids[lev] != grids[lev]);
            if (ba_changed || coarse_ba_changed) {
                BoxArray level_grids = grids[lev];
                DistributionMapping level_dmap = dmap[lev];
                if (ba_changed) {
                    level_grids = new_grids[lev];
                    level_dmap = DistributionMapping(level_grids);
                }
                const auto old_num_setdm = num_setdm;
                RemakeLevel(lev, time, level_grids, level_dmap);
                SetBoxArray(lev, level_grids);
                if (old_num_setdm == num_setdm) {
                    SetDistributionMap(lev, level_dmap);
                }
            }
            coarse_ba_changed = ba_changed;;
        }
        else  // a new level
        {
            DistributionMapping new_dmap(new_grids[lev]);
            const auto old_num_setdm = num_setdm;
            MakeNewLevelFromCoarse(lev, time, new_grids[lev], new_dmap);
            SetBoxArray(lev, new_grids[lev]);
            if (old_num_setdm == num_setdm) {
                SetDistributionMap(lev, new_dmap);
            }
        }
    }

    for (int lev = new_finest+1; lev <= finest_level; ++lev) {
        ClearLevel(lev);
        ClearBoxArray(lev);
        ClearDistributionMap(lev);
    }

    finest_level = new_finest;
}

// Gets the multifab corresponding to the given level and variable name
const MultiFab* 
AmrCoreProblem::getMultifab (std::string varname, int level) const
{
    switch (mapVariableValues[varname]) {
        		case simflowny_vars::pheI_plt:
			return &unp1state_var[level];
		case simflowny_vars::trK_plt:
			return &unp1state_var[level];
		case simflowny_vars::phiI_plt:
			return &unp1state_var[level];
		case simflowny_vars::theta_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_yy_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_yz_plt:
			return &unp1state_var[level];
		case simflowny_vars::phi2_plt:
			return &unSyncAuxFieldGroup_var[level];
		case simflowny_vars::Betau_y_plt:
			return &unp1state_var[level];
		case simflowny_vars::Betau_x_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_yy_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_yz_plt:
			return &unp1state_var[level];
		case simflowny_vars::Gamh_z_plt:
			return &unp1state_var[level];
		case simflowny_vars::Betau_z_plt:
			return &unp1state_var[level];
		case simflowny_vars::chi_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_xz_plt:
			return &unp1state_var[level];
		case simflowny_vars::pheR_plt:
			return &unp1state_var[level];
		case simflowny_vars::piI_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_xx_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_zz_plt:
			return &unp1state_var[level];
		case simflowny_vars::Atd_xy_plt:
			return &unp1state_var[level];
		case simflowny_vars::phiR_plt:
			return &unp1state_var[level];
		case simflowny_vars::peI_plt:
			return &unp1state_var[level];
		case simflowny_vars::Alpha_plt:
			return &unp1state_var[level];
		case simflowny_vars::piR_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_xx_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_zz_plt:
			return &unp1state_var[level];
		case simflowny_vars::Gamh_x_plt:
			return &unp1state_var[level];
		case simflowny_vars::peR_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_xy_plt:
			return &unp1state_var[level];
		case simflowny_vars::Gamh_y_plt:
			return &unp1state_var[level];
		case simflowny_vars::gtd_xz_plt:
			return &unp1state_var[level];
		case simflowny_vars::Rscalar_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::HamCon_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::MomCon_x_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::MomCon_y_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::MomCon_z_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::trA_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::detgtm1_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::M_ADM_surf_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::Jz_ADM_surf_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::N_Noetheri_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::N_Noethere_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::M_Komar_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::Jz_Komar_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::psi4R_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::psi4I_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::Z_x_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::Z_y_plt:
			return &analysisGroup_var[level];
		case simflowny_vars::Z_z_plt:
			return &analysisGroup_var[level];
        
        default:
            amrex::Error("Regridding error. Variable " + varname + " does not exist.");
    }
    return NULL;
}

// Gets the index corresponding to the given variable name
int
AmrCoreProblem::getVarIdx (std::string varname) {
    switch (mapVariableValues[varname]) {
        		case simflowny_vars::pheI_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::trK_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::phiI_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::theta_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_yy_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_yz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::phi2_plt:
			return mapVarUnSyncAuxFieldIdx[varname];
		case simflowny_vars::Betau_y_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Betau_x_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_yy_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_yz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Gamh_z_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Betau_z_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::chi_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_xz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::pheR_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::piI_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_xx_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_zz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Atd_xy_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::phiR_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::peI_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Alpha_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::piR_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_xx_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_zz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Gamh_x_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::peR_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_xy_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Gamh_y_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::gtd_xz_plt:
			return mapVarFieldIdx[varname];
		case simflowny_vars::Rscalar_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::HamCon_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::MomCon_x_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::MomCon_y_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::MomCon_z_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::trA_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::detgtm1_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::M_ADM_surf_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::Jz_ADM_surf_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::N_Noetheri_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::N_Noethere_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::M_Komar_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::Jz_Komar_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::psi4R_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::psi4I_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::Z_x_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::Z_y_plt:
			return mapVarAnalysisIdx[varname];
		case simflowny_vars::Z_z_plt:
			return mapVarAnalysisIdx[varname];
        
        default:
            amrex::Error("Regridding error. Variable " + varname + " does not exist.");
    }
    return NULL;
}

// tag all cells for refinement
// overrides the pure virtual function in AmrCore
void
AmrCoreProblem::ErrorEst (int lev, TagBoxArray& tags, Real /*time*/, int /*ngrow*/)
{
    static bool first = true;
    static Vector<Real> tagerr;
    int varIdx;
    static int regriddingType;
    static std::string shadowTime;

    // read in an array of "tagerr", which is the tagging threshold
    // in this example, we tag values of "phi" which are greater than tagerr
    // for that particular level
    // in subroutine state_error, you could use more elaborate tagging, such
    // as more advanced logical expressions, or gradients, etc.
    ParmParse pp("amr");
    int n = pp.countval("tag_err");
    if (n > 0) {
        pp.getarr("tag_err", tagerr, 0, n);
    }
    std::string tag_field; 
    pp.get("tag_field", tag_field);
    varIdx = getVarIdx(tag_field);

    pp.get("regridding_type", regriddingType);
    if (regriddingType == 1) {
        pp.get("tag_shadow_time", shadowTime);
    }

    const int   tagval = TagBox::SET;

    //FUNCTION regridding
    if (regriddingType == 0) {
        const MultiFab* state = getMultifab(tag_field, lev);
        MultiFab& regrid_tag_mf = regrid_tag_var[lev];

    #ifdef AMREX_USE_OMP
    #pragma omp parallel if(Gpu::notInLaunchRegion())
    #endif
        {

            for (MFIter mfi(regrid_tag_mf,TilingIfNotGPU()); mfi.isValid(); ++mfi)
            {
                const Box& bx  = mfi.tilebox();
                const auto statefab = state->array(mfi);
                const auto tagfab  = tags.array(mfi);
                auto regrid_tag  = regrid_tag_mf.array(mfi);
                Real error = tagerr[MIN(tagerr.size() - 1, lev - use_fixed_upto_level)];

                amrex::ParallelFor(bx,
                [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                {
                    state_error(i, j, k, tagfab, statefab, error, tagval, varIdx);
                    regrid_tag(i,j,k) = tagfab(i,j,k);
                });
            }
        }
    }
    //SHADOW regridding
    if (regriddingType == 1) {
        const MultiFab* state = getMultifab(tag_field, lev);
        MultiFab* shadowState;
		if (shadowTime == "un") {
			shadowState = &unstate_var[lev];		}
		else if (shadowTime == "rk1") {
			shadowState = &rk1state_var[lev];		}
		else if (shadowTime == "rk2") {
			shadowState = &rk2state_var[lev];		}
		else if (shadowTime == "rk3") {
			shadowState = &rk3state_var[lev];		}
		else if (shadowTime == "unp1") {
			shadowState = &unp1state_var[lev];		}
		else {
			amrex::Error("Regridding configuration error. Shadow time " + shadowTime + " does not exist. Use one of [\"un\", \"rk1\", \"rk2\", \"rk3\", \"unp1\"]");
		}


        MultiFab& regrid_tag_mf = regrid_tag_var[lev];

    #ifdef AMREX_USE_OMP
    #pragma omp parallel if(Gpu::notInLaunchRegion())
    #endif
        {

            for (MFIter mfi(regrid_tag_mf,TilingIfNotGPU()); mfi.isValid(); ++mfi)
            {
                const Box& bx  = mfi.tilebox();
                const auto statefab = state->array(mfi);
                const auto shadowStatefab = shadowState->array(mfi);
                const auto tagfab  = tags.array(mfi);
                auto regrid_tag  = regrid_tag_mf.array(mfi);
                Real error = tagerr[MIN(tagerr.size() - 1, lev - use_fixed_upto_level)];

                amrex::ParallelFor(bx,
                [=] AMREX_GPU_DEVICE (int i, int j, int k) noexcept
                {
                    state_error(i, j, k, tagfab, statefab, shadowStatefab, error, tagval, varIdx);
                    regrid_tag(i,j,k) = tagfab(i,j,k);
                });
            }
        }
    }

}

// Fills vectors with alternative interpolation variables
void 
AmrCoreProblem::FillInterpolatorAlternativeSets()
{
    int lastIndex;
	//Add all alternative variables to list
	std::set<int> t_state_interpolator_change;
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "pheI")) {
		t_state_interpolator_change.insert(simflowny_vars::pheI);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "trK")) {
		t_state_interpolator_change.insert(simflowny_vars::trK);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "phiI")) {
		t_state_interpolator_change.insert(simflowny_vars::phiI);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "theta")) {
		t_state_interpolator_change.insert(simflowny_vars::theta);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_yy")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_yy);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_yz")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_yz);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Betau_y")) {
		t_state_interpolator_change.insert(simflowny_vars::Betau_y);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Betau_x")) {
		t_state_interpolator_change.insert(simflowny_vars::Betau_x);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_yy")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_yy);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_yz")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_yz);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Gamh_z")) {
		t_state_interpolator_change.insert(simflowny_vars::Gamh_z);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Betau_z")) {
		t_state_interpolator_change.insert(simflowny_vars::Betau_z);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "chi")) {
		t_state_interpolator_change.insert(simflowny_vars::chi);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_xz")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_xz);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "pheR")) {
		t_state_interpolator_change.insert(simflowny_vars::pheR);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "piI")) {
		t_state_interpolator_change.insert(simflowny_vars::piI);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_xx")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_xx);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_zz")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_zz);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Atd_xy")) {
		t_state_interpolator_change.insert(simflowny_vars::Atd_xy);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "phiR")) {
		t_state_interpolator_change.insert(simflowny_vars::phiR);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "peI")) {
		t_state_interpolator_change.insert(simflowny_vars::peI);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Alpha")) {
		t_state_interpolator_change.insert(simflowny_vars::Alpha);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "piR")) {
		t_state_interpolator_change.insert(simflowny_vars::piR);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_xx")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_xx);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_zz")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_zz);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Gamh_x")) {
		t_state_interpolator_change.insert(simflowny_vars::Gamh_x);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "peR")) {
		t_state_interpolator_change.insert(simflowny_vars::peR);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_xy")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_xy);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "Gamh_y")) {
		t_state_interpolator_change.insert(simflowny_vars::Gamh_y);
	}
	if (std::count(alternative_interpolator_variables.begin(), alternative_interpolator_variables.end(), "gtd_xz")) {
		t_state_interpolator_change.insert(simflowny_vars::gtd_xz);
	}
	//Group consecutive variables
	lastIndex = -3;
	for (std::set<int>::iterator it=t_state_interpolator_change.begin(); it!=t_state_interpolator_change.end(); ++it) {
		int value = *it;
		if (lastIndex + 1 < value) {
			state_interpolator_change.insert(value);
		}
		lastIndex = value;
	}
	//Add last interpolation change if necessary
	if (lastIndex >= 0 && lastIndex + 1 < NumFields) {
		state_interpolator_change.insert(lastIndex + 1);
	}

}

// read in some parameters from inputs file
void
AmrCoreProblem::ReadParameters ()
{
    {
        ParmParse pp("amr"); // Traditionally, these have prefix, amr.

        pp.query("regrid_int", regrid_int);
        pp.query("interpolator", spatialInterpolator);

        pp.query("alternative_interpolator", alternativeSpatialInterpolator);
        pp.queryarr("alternative_interpolator_variables", alternative_interpolator_variables);

        //FMR parameters
        pp.get("fmr_levels", fmr_levels);
        fmr_lo.resize(fmr_levels);
        fmr_hi.resize(fmr_levels);
        int current_ratio = 1;
        for (int i = 0; i < fmr_levels; i++) {
            std::vector<Real> lo;
            std::vector<Real> hi;
            pp.getarr(("fmr_level_" + std::to_string(i) + "_lo").c_str(), lo);
            pp.getarr(("fmr_level_" + std::to_string(i) + "_hi").c_str(), hi);
            //Convert domain units to indices
            current_ratio = current_ratio * refRatio(i)[0];
            const auto dx = Geom(i).CellSize();
            const auto problo = Geom(i).ProbLoArray();
            const auto probhi = Geom(i).ProbHiArray();

            if (
                problo[0] > lo[0] || probhi[0] < hi[0]
                || problo[1] > lo[1] || probhi[1] < hi[1]
                #if AMREX_SPACEDIM == 3
                || problo[2] > lo[2] || probhi[2] < hi[2]
                #endif    
                ) {
                amrex::Abort("FMR level must be within domain.");
            }

            fmr_lo[i].resize(AMREX_SPACEDIM);
            fmr_hi[i].resize(AMREX_SPACEDIM);

            fmr_lo[i][0] = floor((lo[0] - problo[0]) / dx[0]);
            fmr_lo[i][1] = floor((lo[1] - problo[1]) / dx[1]);
            #if AMREX_SPACEDIM == 3
                fmr_lo[i][2] = floor((lo[2] - problo[2]) / dx[2]);
            #endif

            fmr_hi[i][0] = ceil((hi[0] - problo[0]) / dx[0]) - 1;
            fmr_hi[i][1] = ceil((hi[1] - problo[1]) / dx[1]) - 1;
            #if AMREX_SPACEDIM == 3
                fmr_hi[i][2] = ceil((hi[2] - problo[2]) / dx[2]) - 1;
            #endif
        }
    }
    /* 
        Plot parameters 
    */
    {   
        ParmParse pp;
 
        //Common parameters
        ParmParse pp_plt("plt");
        pp_plt.get("plot_overwrite", plot_overwrite);

        //Full plot params
        ParmParse pp_full("plt_full");
        pp_full.get("dir", plot_dir);
        pp_full.get("int", plot_int);
        pp_full.getarr("varnames", plot_varnames);

        //Integral plot parameters
        int dbcounter = 1;
        while (pp.getEntries("plt_integral_" + std::to_string(dbcounter)).size() > 0) {
            ParmParse pp_int("plt_integral_" + std::to_string(dbcounter));
            std::string t_plot_integral_dir;
            int t_plot_integral_int;
            Vector<std::string> t_plot_integral_varnames;
            Vector<std::string> t_plot_integral_calculations;
            pp_int.get("dir", t_plot_integral_dir);
            pp_int.get("int", t_plot_integral_int);
            pp_int.getarr("varnames", t_plot_integral_varnames);
            pp_int.getarr("calculations", t_plot_integral_calculations);
            plot_integral_dir.push_back(t_plot_integral_dir);
            plot_integral_int.push_back(t_plot_integral_int);
            plot_integral_varnames.push_back(t_plot_integral_varnames);
            plot_integral_calculations.push_back(t_plot_integral_calculations);
            first_plot_integral.push_back(true);
            dbcounter++;
        }
        
        //Point plot parameters
        dbcounter = 1;
        while (pp.getEntries("plt_point_" + std::to_string(dbcounter)).size() > 0) {
            ParmParse pp_int("plt_point_" + std::to_string(dbcounter));
            std::string t_plot_point_dir;
            int t_plot_point_int;
            Vector<std::string> t_plot_point_varnames;
            Vector<Real> t_plot_point_coordinates;
            pp_int.get("dir", t_plot_point_dir);
            pp_int.get("int", t_plot_point_int);
            pp_int.getarr("varnames", t_plot_point_varnames);
            pp_int.getarr("coordinates", t_plot_point_coordinates);
            plot_point_dir.push_back(t_plot_point_dir);
            plot_point_int.push_back(t_plot_point_int);
            plot_point_varnames.push_back(t_plot_point_varnames);
            plot_point_coordinates.push_back(t_plot_point_coordinates);
            first_plot_point.push_back(true);
            dbcounter++;  
        }

		//Slice plot parameters
		dbcounter = 1;
		while (pp.getEntries("plt_slice_" + std::to_string(dbcounter)).size() > 0) {
			ParmParse pp_int("plt_slice_" + std::to_string(dbcounter));
			std::string t_plot_slice_dir;
			int t_plot_slice_int;
			Vector<std::string> t_plot_slice_varnames;
			int t_plot_slice_plane_normal_axis;
			Real t_plot_slice_distance_to_origin;
			pp_int.get("dir", t_plot_slice_dir);
			pp_int.get("int", t_plot_slice_int);
			pp_int.getarr("varnames", t_plot_slice_varnames);
			pp_int.get("plane_normal_axis", t_plot_slice_plane_normal_axis);
			pp_int.get("distance_to_origin", t_plot_slice_distance_to_origin);
			plot_slice_dir.push_back(t_plot_slice_dir);
			plot_slice_int.push_back(t_plot_slice_int);
			plot_slice_varnames.push_back(t_plot_slice_varnames);
			plot_slice_plane_normal_axis.push_back(t_plot_slice_plane_normal_axis);
			plot_slice_distance_to_origin.push_back(t_plot_slice_distance_to_origin);
			first_plot_slice.push_back(true);
			dbcounter++;
		}

		//Sphere plot parameteres
		dbcounter = 1;
		while (pp.getEntries("plt_sphere_" + std::to_string(dbcounter)).size() > 0) {
			ParmParse pp_int("plt_sphere_" + std::to_string(dbcounter));
			std::string t_plot_sphere_dir;
			int t_plot_sphere_int;
			Vector<std::string> t_plot_sphere_varnames;
			Vector<Real> t_plot_sphere_center;
			Real t_plot_sphere_radius;
			Vector<int> t_plot_sphere_resolution;
			pp_int.get("dir", t_plot_sphere_dir);
			pp_int.get("int", t_plot_sphere_int);
			pp_int.getarr("varnames", t_plot_sphere_varnames);
			pp_int.getarr("center", t_plot_sphere_center);
			pp_int.get("radius", t_plot_sphere_radius);
			pp_int.getarr("resolution", t_plot_sphere_resolution);
			plot_sphere_dir.push_back(t_plot_sphere_dir);
			plot_sphere_int.push_back(t_plot_sphere_int);
			plot_sphere_varnames.push_back(t_plot_sphere_varnames);
			plot_sphere_center.push_back(t_plot_sphere_center);
			plot_sphere_radius.push_back(t_plot_sphere_radius);
			plot_sphere_resolution.push_back(t_plot_sphere_resolution);
			first_plot_sphere.push_back(true);
			dbcounter++;
		}

    }
    {
        ParmParse pp("chk");
        pp.query("file", chk_file);
        pp.query("int", chk_int);
        pp.query("restart",restart_chkfile);
    }

    {
        ParmParse pp("problem");

        pp.query("cfl", cfl);
        pp.get("do_subcycle", do_subcycle);
        pp.query("random_seed", random_seed_param);

		pp.query("restart_field_reset", restart_field_reset);
		pp.get("dissipation_factor_Atd_xx", parameters.dissipation_factor_Atd_xx);
		pp.get("dissipation_factor_Atd_xy", parameters.dissipation_factor_Atd_xy);
		pp.get("dissipation_factor_gtd_zz", parameters.dissipation_factor_gtd_zz);
		pp.get("dissipation_factor_Atd_xz", parameters.dissipation_factor_Atd_xz);
		pp.get("dissipation_factor_Gamh_y", parameters.dissipation_factor_Gamh_y);
		pp.get("dissipation_factor_Gamh_x", parameters.dissipation_factor_Gamh_x);
		pp.get("Atd_xx_falloff", parameters.Atd_xx_falloff);
		pp.get("dissipation_factor_Gamh_z", parameters.dissipation_factor_Gamh_z);
		pp.get("p_kappa_z2", parameters.p_kappa_z2);
		pp.get("p_kappa_z1", parameters.p_kappa_z1);
		pp.get("Gamh_z_asymptotic", parameters.Gamh_z_asymptotic);
		pp.get("gtd_yz_asymptotic", parameters.gtd_yz_asymptotic);
		pp.get("gtd_xz_asymptotic", parameters.gtd_xz_asymptotic);
		pp.get("Betau_y_falloff", parameters.Betau_y_falloff);
		pp.get("lambda_f1", parameters.lambda_f1);
		pp.get("gtd_xz_falloff", parameters.gtd_xz_falloff);
		pp.get("Betau_z_0", parameters.Betau_z_0);
		pp.get("lambda_f0", parameters.lambda_f0);
		pp.get("lambda_f3", parameters.lambda_f3);
		pp.get("lambda_f2", parameters.lambda_f2);
		pp.get("pheR_falloff", parameters.pheR_falloff);
		pp.get("Atd_zz_falloff", parameters.Atd_zz_falloff);
		pp.get("Gamh_z_falloff", parameters.Gamh_z_falloff);
		pp.get("piI_asymptotic", parameters.piI_asymptotic);
		pp.get("zcenter1", parameters.zcenter1);
		pp.get("zcenter2", parameters.zcenter2);
		pp.get("trK_falloff", parameters.trK_falloff);
		pp.get("xcenter2", parameters.xcenter2);
		pp.get("xcenter1", parameters.xcenter1);
		pp.get("chi_falloff", parameters.chi_falloff);
		pp.get("dissipation_factor_theta", parameters.dissipation_factor_theta);
		pp.get("gtd_yz_falloff", parameters.gtd_yz_falloff);
		pp.get("peR_falloff", parameters.peR_falloff);
		pp.get("gtd_xx_asymptotic", parameters.gtd_xx_asymptotic);
		pp.get("piR_falloff", parameters.piR_falloff);
		pp.get("Betau_y_asymptotic", parameters.Betau_y_asymptotic);
		pp.get("dissipation_factor_Atd_zz", parameters.dissipation_factor_Atd_zz);
		pp.get("pheI_falloff", parameters.pheI_falloff);
		pp.get("peI_falloff", parameters.peI_falloff);
		pp.get("dissipation_factor_Betau_z", parameters.dissipation_factor_Betau_z);
		pp.get("dissipation_factor_Betau_y", parameters.dissipation_factor_Betau_y);
		pp.get("dissipation_factor_Betau_x", parameters.dissipation_factor_Betau_x);
		pp.get("dissipation_factor_Alpha", parameters.dissipation_factor_Alpha);
		pp.get("gtd_zz_falloff", parameters.gtd_zz_falloff);
		pp.get("theta_asymptotic", parameters.theta_asymptotic);
		pp.get("pheR_asymptotic", parameters.pheR_asymptotic);
		pp.get("p_feta", parameters.p_feta);
		pp.get("Gamh_x_falloff", parameters.Gamh_x_falloff);
		pp.get("dissipation_factor_peR", parameters.dissipation_factor_peR);
		pp.get("eta_damping_exp", parameters.eta_damping_exp);
		pp.get("phiR_asymptotic", parameters.phiR_asymptotic);
		pp.get("Gamh_y_asymptotic", parameters.Gamh_y_asymptotic);
		pp.get("sflambda", parameters.sflambda);
		pp.get("Betau_x_falloff", parameters.Betau_x_falloff);
		pp.get("dissipation_factor_Atd_yy", parameters.dissipation_factor_Atd_yy);
		pp.get("dissipation_factor_Atd_yz", parameters.dissipation_factor_Atd_yz);
		pp.get("dissipation_factor_peI", parameters.dissipation_factor_peI);
		pp.get("gtd_xy_falloff", parameters.gtd_xy_falloff);
		pp.get("omega2", parameters.omega2);
		pp.get("omega1", parameters.omega1);
		pp.get("gtd_zz_asymptotic", parameters.gtd_zz_asymptotic);
		pp.get("Alpha_asymptotic", parameters.Alpha_asymptotic);
		pp.get("dissipation_factor_pheR", parameters.dissipation_factor_pheR);
		pp.get("Betau_x_asymptotic", parameters.Betau_x_asymptotic);
		pp.get("Atd_yy_falloff", parameters.Atd_yy_falloff);
		pp.get("trK0", parameters.trK0);
		pp.get("lambda_2", parameters.lambda_2);
		pp.get("lambda_1", parameters.lambda_1);
		pp.get("p_sfmass", parameters.p_sfmass);
		pp.get("lambda_0", parameters.lambda_0);
		pp.get("phiR_falloff", parameters.phiR_falloff);
		pp.get("dissipation_factor_pheI", parameters.dissipation_factor_pheI);
		pp.get("Gamh_x_asymptotic", parameters.Gamh_x_asymptotic);
		pp.get("R_0", parameters.R_0);
		pp.get("Atd_xy_asymptotic", parameters.Atd_xy_asymptotic);
		pp.get("trK_asymptotic", parameters.trK_asymptotic);
		pp.get("Atd_yy_asymptotic", parameters.Atd_yy_asymptotic);
		pp.get("nstars", parameters.nstars);
		pp.get("p_kappa_cc", parameters.p_kappa_cc);
		pp.get("gtd_yy_asymptotic", parameters.gtd_yy_asymptotic);
		pp.get("Betau_z_asymptotic", parameters.Betau_z_asymptotic);
		pp.get("Atd_xz_falloff", parameters.Atd_xz_falloff);
		pp.get("theta_falloff", parameters.theta_falloff);
		pp.get("pheI_asymptotic", parameters.pheI_asymptotic);
		pp.get("gtd_xx_falloff", parameters.gtd_xx_falloff);
		pp.get("phiI_asymptotic", parameters.phiI_asymptotic);
		pp.get("vx1", parameters.vx1);
		pp.get("gtd_xy_asymptotic", parameters.gtd_xy_asymptotic);
		pp.get("peR_asymptotic", parameters.peR_asymptotic);
		pp.get("vx2", parameters.vx2);
		pp.get("sfsigma", parameters.sfsigma);
		pp.get("dissipation_factor_chi", parameters.dissipation_factor_chi);
		pp.get("Betau_x_0", parameters.Betau_x_0);
		pp.get("Atd_yz_falloff", parameters.Atd_yz_falloff);
		pp.get("dissipation_factor_gtd_xz", parameters.dissipation_factor_gtd_xz);
		pp.get("dissipation_factor_gtd_xx", parameters.dissipation_factor_gtd_xx);
		pp.get("dissipation_factor_gtd_xy", parameters.dissipation_factor_gtd_xy);
		pp.get("piR_asymptotic", parameters.piR_asymptotic);
		pp.get("peI_asymptotic", parameters.peI_asymptotic);
		pp.get("Atd_xy_falloff", parameters.Atd_xy_falloff);
		pp.get("dissipation_factor_trK", parameters.dissipation_factor_trK);
		pp.get("tend", parameters.tend);
		pp.get("Atd_xx_asymptotic", parameters.Atd_xx_asymptotic);
		pp.get("dissipation_factor_piR", parameters.dissipation_factor_piR);
		pp.get("Betau_y_0", parameters.Betau_y_0);
		pp.get("piI_falloff", parameters.piI_falloff);
		pp.get("amp_phi", parameters.amp_phi);
		pp.get("dissipation_factor_gtd_yy", parameters.dissipation_factor_gtd_yy);
		pp.get("Gamh_y_falloff", parameters.Gamh_y_falloff);
		pp.get("dissipation_factor_gtd_yz", parameters.dissipation_factor_gtd_yz);
		pp.get("Atd_zz_asymptotic", parameters.Atd_zz_asymptotic);
		pp.get("dissipation_factor_piI", parameters.dissipation_factor_piI);
		pp.get("Alpha_falloff", parameters.Alpha_falloff);
		pp.get("Betau_z_falloff", parameters.Betau_z_falloff);
		pp.get("ycenter2", parameters.ycenter2);
		pp.get("chi_asymptotic", parameters.chi_asymptotic);
		pp.get("Atd_xz_asymptotic", parameters.Atd_xz_asymptotic);
		pp.get("Atd_yz_asymptotic", parameters.Atd_yz_asymptotic);
		pp.get("dissipation_factor_phiR", parameters.dissipation_factor_phiR);
		pp.get("ycenter1", parameters.ycenter1);
		pp.get("phiI_falloff", parameters.phiI_falloff);
		pp.get("chi_floor", parameters.chi_floor);
		pp.get("gtd_yy_falloff", parameters.gtd_yy_falloff);
		pp.get("dissipation_factor_phiI", parameters.dissipation_factor_phiI);

    }
    //External eos parameters
#ifdef EXTERNAL_EOS
    {
        ParmParse pp("external_EOS");
        pp.query("eos_type", Commons::ExternalEos::reprimand_eos_type);
        pp.query("atmo_Ye", Commons::ExternalEos::reprimand_atmo_Ye);
        pp.query("max_z",Commons::ExternalEos::reprimand_max_z);
        pp.query("max_b",Commons::ExternalEos::reprimand_max_b);
        pp.query("c2p_acc",Commons::ExternalEos::reprimand_c2p_acc);
        pp.query("atmo_rho",Commons::ExternalEos::reprimand_atmo_rho);
        pp.query("rho_strict",Commons::ExternalEos::reprimand_rho_strict);
        pp.query("max_rho",Commons::ExternalEos::reprimand_max_rho);
        pp.query("max_eps",Commons::ExternalEos::reprimand_max_eps);
        pp.query("gamma_th",Commons::ExternalEos::reprimand_gamma_th);
    }
#endif
}

// set covered coarse cells to be the average of overlying fine cells
void
AmrCoreProblem::AverageDown ()
{
    for (int lev = finest_level-1; lev >= 0; --lev)
    {
		amrex::average_down_nodal(unp1state_var[lev+1], unp1state_var[lev], refRatio(lev), 0);
		amrex::average_down_nodal(unSyncAuxFieldGroup_var[lev+1], unSyncAuxFieldGroup_var[lev], refRatio(lev), 0);

    }
}

// more flexible version of AverageDown() that lets you average down across multiple levels
void
AmrCoreProblem::AverageDownTo (int crse_lev)
{
    BL_PROFILE("AmrCoreProblem::AverageDownTo(crse_lev)");
	amrex::average_down_nodal(unp1state_var[crse_lev+1], unp1state_var[crse_lev], refRatio(crse_lev), 1);
	amrex::average_down_nodal(unSyncAuxFieldGroup_var[crse_lev+1], unSyncAuxFieldGroup_var[crse_lev], refRatio(crse_lev), 1);

}

// compute a new multifab by coping in phi from valid region and filling ghost cells
// works for single level and 2-level cases (fill fine grid ghost by interpolating from coarse)
void
AmrCoreProblem::FillPatch(int lev, Real time, MultiFab& mf, Vector<MultiFab>& origin_data, int tcomp, std::set<int> interpolator_mapper_changes, bool syncSingleLevel)
{
    int icomp = 0;
    int ncomp;
    if (lev == 0 || syncSingleLevel)
    {
        ncomp = tcomp - icomp;
        Vector<MultiFab*> smf;
        Vector<Real> stime;
        GetData(lev, time, smf, stime, origin_data);

        if(Gpu::inLaunchRegion())
        {
            GpuBndryFuncFab<AmrCoreFill> gpu_bndry_func(AmrCoreFill{});
            PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > physbc(geom[lev],bcs,gpu_bndry_func);
            simflowny::FillPatchSingleLevel(mf, time, smf, stime, icomp, icomp, ncomp,
                                        geom[lev], IntVect(AMREX_D_DECL(1 ,1 ,1 )), physbc, 0, 0, 0);
        }
        else
        {
            CpuBndryFuncFab bndry_func(nullptr);  // Without EXT_DIR, we can pass a nullptr.
            PhysBCFunct<CpuBndryFuncFab> physbc(geom[lev],bcs,bndry_func);
            simflowny::FillPatchSingleLevel(mf, time, smf, stime, icomp, icomp, ncomp,
                                        geom[lev], IntVect(AMREX_D_DECL(1 ,1 ,1 )), physbc, 0, 0, 0);
        }
    }
    else
    {
        Vector<MultiFab*> cmf, fmf;
        Vector<Real> ctime, ftime;
        GetData(lev-1, time, cmf, ctime, origin_data);
        GetData(lev  , time, fmf, ftime, origin_data);

        if(Gpu::inLaunchRegion())
        {
            GpuBndryFuncFab<AmrCoreFill> gpu_bndry_func(AmrCoreFill{});
            PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > cphysbc(geom[lev-1],bcs,gpu_bndry_func);
            PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > fphysbc(geom[lev],bcs,gpu_bndry_func);

            icomp = 0;
            ncomp = 0;
            bool alt_mapper = false;
            //Launch bocks of interpolation changes
            for (auto it = interpolator_mapper_changes.begin(); it != interpolator_mapper_changes.end(); it++) {
                ncomp = *it - icomp;
                //Skip empty synchronization if necessary
                if (ncomp > 0) {
                    if (alt_mapper) {
                        simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, 0, 0, bcs, 0);
                    }
                    else {
                        simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, 0, 0, bcs, 0);
                    }
                }
                icomp = *it;
                //Change mapper
                alt_mapper = !alt_mapper;
            }

            //Last mapping
            ncomp = tcomp - icomp;
            if (alt_mapper) {
                simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, 0, 0, bcs, 0);
            }
            else {
                simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, 0, 0, bcs, 0);
            }
        }
        else
        {
            CpuBndryFuncFab bndry_func(nullptr);  // Without EXT_DIR, we can pass a nullptr.
            PhysBCFunct<CpuBndryFuncFab> cphysbc(geom[lev-1],bcs,bndry_func);
            PhysBCFunct<CpuBndryFuncFab> fphysbc(geom[lev],bcs,bndry_func);

            icomp = 0;
            ncomp = 0;
            bool alt_mapper = false;
            //Launch bocks of interpolation changes
            for (auto it = interpolator_mapper_changes.begin(); it != interpolator_mapper_changes.end(); it++) {
                ncomp = *it - icomp;
                //Skip empty synchronization if necessary
                if (ncomp > 0) {
                    if (alt_mapper) {
                        simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, 0, 0, bcs, 0);
                    }
                    else {
                        simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, 0, 0, bcs, 0);
                    }
                }
                icomp = *it;
                //Change mapper
                alt_mapper = !alt_mapper;
            }

            //Last mapping
            ncomp = tcomp - icomp;
            if (alt_mapper) {
                simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, 0, 0, bcs, 0);
            }
            else {
                simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, 0, 0, bcs, 0);
            }
        }
    }
}
void AmrCoreProblem::FillPatch (int lev, Real time, MultiFab& mf, Vector<MultiFab>& origin_data_un, Vector<MultiFab>& origin_data_rk1, Vector<MultiFab>& origin_data_rk2, Vector<MultiFab>& origin_data_rk3, Vector<MultiFab>& origin_data_unp1, int tcomp, int interp_step, int time_substep_number, std::set<int> interpolator_mapper_changes, bool syncSingleLevel) {
	int icomp = 0;
	int ncomp;
	if (lev == 0 || syncSingleLevel) {
		ncomp = tcomp - icomp;
		Vector<MultiFab*> smf;
		Vector<Real> stime;
		GetData(lev, time, smf, stime, origin_data_un, origin_data_rk1, origin_data_rk2, origin_data_rk3, origin_data_unp1);

		if(Gpu::inLaunchRegion()) {
			GpuBndryFuncFab<AmrCoreFill> gpu_bndry_func(AmrCoreFill{});
			PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > physbc(geom[lev],bcs,gpu_bndry_func);
			simflowny::FillPatchSingleLevel(mf, time, smf, stime, icomp, icomp, ncomp, geom[lev], IntVect(AMREX_D_DECL(1 ,1 ,1 )), physbc, 0, interp_step, time_substep_number);
		}
		else {
			CpuBndryFuncFab bndry_func(nullptr);  // Without EXT_DIR, we can pass a nullptr.
			PhysBCFunct<CpuBndryFuncFab> physbc(geom[lev],bcs,bndry_func);
			simflowny::FillPatchSingleLevel(mf, time, smf, stime, icomp, icomp, ncomp, geom[lev], IntVect(AMREX_D_DECL(1 ,1 ,1 )), physbc, 0, interp_step, time_substep_number);
		}
	}
	else {
		Vector<MultiFab*> cmf, fmf;
		Vector<Real> ctime, ftime;
		GetData(lev-1, time, cmf, ctime, origin_data_un, origin_data_rk1, origin_data_rk2, origin_data_rk3, origin_data_unp1);
		GetData(lev  , time, fmf, ftime, origin_data_un, origin_data_rk1, origin_data_rk2, origin_data_rk3, origin_data_unp1);

		if(Gpu::inLaunchRegion()) {
			GpuBndryFuncFab<AmrCoreFill> gpu_bndry_func(AmrCoreFill{});
			PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > cphysbc(geom[lev-1],bcs,gpu_bndry_func);
			PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > fphysbc(geom[lev],bcs,gpu_bndry_func);

			icomp = 0;
			ncomp = 0;
			bool alt_mapper = false;
			//Launch bocks of interpolation changes
			for (auto it = interpolator_mapper_changes.begin(); it != interpolator_mapper_changes.end(); it++) {
				ncomp = *it - icomp;
				//Skip empty synchronization if necessary
				if (ncomp > 0) {
					if (alt_mapper) {
						simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime,icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, interp_step, time_substep_number, bcs, 0);
					} else {
						simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, interp_step, time_substep_number, bcs, 0);
					}
				}
				icomp = *it;
				//Change mapper
				alt_mapper = !alt_mapper;
			}

			//Last mapping
			ncomp = tcomp - icomp;
			if (alt_mapper) {
				simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, interp_step, time_substep_number, bcs, 0);
			} else {
				simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, interp_step, time_substep_number, bcs, 0);
			}
		}
		else {
			CpuBndryFuncFab bndry_func(nullptr);  // Without EXT_DIR, we can pass a nullptr.
			PhysBCFunct<CpuBndryFuncFab> cphysbc(geom[lev-1],bcs,bndry_func);
			PhysBCFunct<CpuBndryFuncFab> fphysbc(geom[lev],bcs,bndry_func);

			icomp = 0;
			ncomp = 0;
			bool alt_mapper = false;
			//Launch bocks of interpolation changes
			for (auto it = interpolator_mapper_changes.begin(); it != interpolator_mapper_changes.end(); it++) {
				ncomp = *it - icomp;
				//Skip empty synchronization if necessary
				if (ncomp > 0) {
					if (alt_mapper) {
						simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime,icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, interp_step, time_substep_number, bcs, 0);
					} else {
						simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, interp_step, time_substep_number, bcs, 0);
					}
				}
				icomp = *it;
				//Change mapper
				alt_mapper = !alt_mapper;
			}

			//Last mapping
			ncomp = tcomp - icomp;
			if (alt_mapper) {
				simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), alternative_mapper, interp_step, time_substep_number, bcs, 0);
			} else {
				simflowny::FillPatchTwoLevels(mf, time, cmf, ctime, fmf, ftime, icomp, icomp, ncomp, geom[lev-1], geom[lev], cphysbc, 0, fphysbc, 0, refRatio(lev-1), mapper, interp_step, time_substep_number, bcs, 0);
			}
		}
	}
}


// fill an entire multifab by interpolating from the coarser level
// this comes into play when a new level of refinement appears
void
AmrCoreProblem::FillCoarsePatch (int lev, Real time, MultiFab& mf, Vector<MultiFab>& origin_data, int icomp, int ncomp)
{
    BL_ASSERT(lev > 0);

    Vector<MultiFab*> cmf;
    Vector<Real> ctime;
    GetData(lev-1, time, cmf, ctime, origin_data);
    Interpolater* mapper = &node_bilinear_interp;

    if (cmf.size() != 1) {
        amrex::Abort("FillCoarsePatch: how did this happen?");
    }

    if(Gpu::inLaunchRegion())
    {
        GpuBndryFuncFab<AmrCoreFill> gpu_bndry_func(AmrCoreFill{});
        PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > cphysbc(geom[lev-1],bcs,gpu_bndry_func);
        PhysBCFunct<GpuBndryFuncFab<AmrCoreFill> > fphysbc(geom[lev],bcs,gpu_bndry_func);

        simflowny::InterpFromCoarseLevel(mf, time, *cmf[0], 0, icomp, ncomp, geom[lev-1], geom[lev],
                                     cphysbc, 0, fphysbc, 0, refRatio(lev-1),
                                     mapper, bcs, 0);
    }
    else
    {
        CpuBndryFuncFab bndry_func(nullptr);  // Without EXT_DIR, we can pass a nullptr.
        PhysBCFunct<CpuBndryFuncFab> cphysbc(geom[lev-1],bcs,bndry_func);
        PhysBCFunct<CpuBndryFuncFab> fphysbc(geom[lev],bcs,bndry_func);

        simflowny::InterpFromCoarseLevel(mf, time, *cmf[0], 0, icomp, ncomp, geom[lev-1], geom[lev],
                                     cphysbc, 0, fphysbc, 0, refRatio(lev-1),
                                     mapper, bcs, 0);
    }
}

// utility to copy in data from phi_old and/or phi_new into another multifab
void
AmrCoreProblem::GetData (int lev, Real time, Vector<MultiFab*>& data, Vector<Real>& datatime, Vector<MultiFab>& origin_data)
{
    data.clear();
    datatime.clear();

    data.push_back(&origin_data[lev]);
    datatime.push_back(t_new[lev]);
}
void AmrCoreProblem::GetData (int lev, Real time, Vector<MultiFab*>& data, Vector<Real>& datatime, Vector<MultiFab>& origin_data_un, Vector<MultiFab>& origin_data_rk1, Vector<MultiFab>& origin_data_rk2, Vector<MultiFab>& origin_data_rk3, Vector<MultiFab>& origin_data_unp1) {
	data.clear();
	datatime.clear();

	data.push_back(&origin_data_un[lev]);
	data.push_back(&origin_data_rk1[lev]);
	data.push_back(&origin_data_rk2[lev]);
	data.push_back(&origin_data_rk3[lev]);
	data.push_back(&origin_data_unp1[lev]);
	datatime.push_back(t_old[lev]);
	datatime.push_back(t_new[lev]);
}



// Advance a level by dt
// (includes a recursive call for finer levels)
void
AmrCoreProblem::timeStepWithSubcycling (int lev, Real time, int iteration)
{
    if (regrid_int > 0)  // We may need to regrid
    {

        // help keep track of whether a level was already regridded
        // from a coarser level call to regrid
        static Vector<int> last_regrid_step(max_level+1, 0);

        // regrid changes level "lev+1" so we don't regrid on max_level
        // also make sure we don't regrid fine levels again if
        // it was taken care of during a coarser regrid
        if (lev < max_level && istep[lev] > last_regrid_step[lev])
        {
            if (istep[lev] % regrid_int == 0)
            {
                // regrid could add newly refine levels (if finest_level < max_level)
                // so we save the previous finest level index
                int old_finest = finest_level;
                regrid(lev, time);

                // mark that we have regridded this level already
                for (int k = lev; k <= finest_level; ++k) {
                    last_regrid_step[k] = istep[k];
                }

                // if there are newly created levels, set the time step
                for (int k = old_finest+1; k <= finest_level; ++k) {
                    dt[k] = dt[k-1] / MaxRefRatio(k-1);
                }
            }
        }
    }

    if (Verbose()) {
        amrex::Print() << "[Level " << lev << " step " << istep[lev]+1 << "] ";
        amrex::Print() << "ADVANCE with time = " << t_new[lev]
                       << " dt = " << dt[lev] << std::endl;
    }

    // Advance a single level for a single time step, and update flux registers

    t_old[lev] = t_new[lev];
    t_new[lev] += dt[lev];

    AdvanceLevel(lev, time, dt[lev], iteration, nsubsteps[lev], true, parameters);

    ++istep[lev];

    if (Verbose())
    {
        amrex::Print() << "[Level " << lev << " step " << istep[lev] << "] ";
        amrex::Print() << "Advanced " << CountCells(lev) << " cells" << std::endl;
    }

    if (lev < finest_level)
    {
        // recursive call for next-finer level
        for (int i = 0; i < nsubsteps[lev+1]; ++i)
        {
            timeStepWithSubcycling(lev+1, time+(i)*dt[lev+1], i);
        }

        AverageDownTo(lev); // average lev+1 down to lev
    }

}

// Advance all the levels with the same dt
void
AmrCoreProblem::timeStepNoSubcycling (Real time, int iteration)
{
    if (max_level > 0 && regrid_int > 0)  // We may need to regrid
    {
        if (istep[0] % regrid_int == 0)
        {
            regrid(0, time);
        }
    }

    if (Verbose()) {
        for (int lev = 0; lev <= finest_level; lev++)
        {
           amrex::Print() << "[Level " << lev << " step " << istep[lev]+1 << "] ";
           amrex::Print() << "ADVANCE with time = " << t_new[lev]
                          << " dt = " << dt[0] << std::endl;
        }
    }

    for (int lev = 0; lev <= finest_level; lev++)
    {
        AdvanceLevel(lev, time, dt[lev], iteration, 0, false, parameters);
    }

    // Make sure the coarser levels are consistent with the finer levels
    AverageDown ();

    for (int lev = 0; lev <= finest_level; lev++)
        ++istep[lev];

    if (Verbose())
    {
        for (int lev = 0; lev <= finest_level; lev++)
        {
            amrex::Print() << "[Level " << lev << " step " << istep[lev] << "] ";
            amrex::Print() << "Advanced " << CountCells(lev) << " cells" << std::endl;
        }
    }

}

// a wrapper for EstTimeStep
void
AmrCoreProblem::ComputeDt ()
{
    Vector<Real> dt_tmp(finest_level+1);

    for (int lev = 0; lev <= finest_level; ++lev)
    {
        dt_tmp[lev] = EstTimeStep(lev, t_new[lev]);
    }
    ParallelDescriptor::ReduceRealMin(&dt_tmp[0], dt_tmp.size());

    constexpr Real change_max = 1.1;
    Real dt_0 = dt_tmp[0];
    int n_factor = 1;

    for (int lev = 0; lev <= finest_level; ++lev) {
        dt_tmp[lev] = std::min(dt_tmp[lev], change_max*dt[lev]);
        n_factor *= nsubsteps[lev];
        dt_0 = std::min(dt_0, n_factor*dt_tmp[lev]);
    }

    dt[0] = dt_0;

    for (int lev = 1; lev <= finest_level; ++lev) {
        dt[lev] = dt[lev-1] / nsubsteps[lev];
    }
}

// compute dt from CFL considerations
Real
AmrCoreProblem::EstTimeStep (int lev, Real time)
{
    BL_PROFILE("AmrCoreProblem::EstTimeStep()");

    Real dt_est = std::numeric_limits<Real>::max();

    const Real* dx  =  geom[lev].CellSize();

    for (int idim = 0; idim < AMREX_SPACEDIM; ++idim)
    {
        dt_est = amrex::min(dt_est, dx[idim]*cfl);
    }

    return dt_est;
}

// get plotfile name
std::string
AmrCoreProblem::PlotFileName (int lev) const
{
    return amrex::Concatenate(plot_file, lev, 5);
}

// put together an array of multifabs for writing
Vector<std::unique_ptr<MultiFab> >
AmrCoreProblem::PlotFileMF (Vector<std::string> varnames, bool withGhost) const
{
    Vector<std::unique_ptr<MultiFab> > r(finest_level + 1);
    for (int i = 0; i <= finest_level; ++i) {
		r[i].reset(new MultiFab(unp1state_var[i].boxArray(), unp1state_var[i].DistributionMap(),varnames.size(),withGhost?NUM_GHOST_CELLS:0));


        int varCounter = 0;
        for (int j = 0; j < varnames.size(); j++) {
            switch (mapVariableValues[varnames[j]]) {
				case simflowny_vars::pheI_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::pheI,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::trK_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::trK,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::phiI_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::phiI,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::theta_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::theta,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_yy_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_yy,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_yz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_yz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::phi2_plt:
					MultiFab::Copy(*r[i],unSyncAuxFieldGroup_var[i],simflowny_vars::phi2,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Betau_y_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Betau_y,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Betau_x_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Betau_x,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_yy_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_yy,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_yz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_yz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Gamh_z_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Gamh_z,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Betau_z_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Betau_z,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::chi_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::chi,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_xz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_xz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::pheR_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::pheR,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::piI_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::piI,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_xx_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_xx,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_zz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_zz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Atd_xy_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Atd_xy,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::phiR_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::phiR,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::peI_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::peI,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Alpha_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Alpha,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::piR_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::piR,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_xx_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_xx,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_zz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_zz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Gamh_x_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Gamh_x,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::peR_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::peR,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_xy_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_xy,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Gamh_y_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::Gamh_y,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::gtd_xz_plt:
					MultiFab::Copy(*r[i],unp1state_var[i],simflowny_vars::gtd_xz,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Rscalar_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Rscalar,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::HamCon_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::HamCon,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::MomCon_x_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::MomCon_x,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::MomCon_y_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::MomCon_y,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::MomCon_z_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::MomCon_z,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::trA_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::trA,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::detgtm1_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::detgtm1,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::M_ADM_surf_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::M_ADM_surf,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Jz_ADM_surf_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Jz_ADM_surf,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::N_Noetheri_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::N_Noetheri,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::N_Noethere_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::N_Noethere,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::M_Komar_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::M_Komar,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Jz_Komar_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Jz_Komar,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::psi4R_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::psi4R,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::psi4I_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::psi4I,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Z_x_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Z_x,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Z_y_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Z_y,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;
				case simflowny_vars::Z_z_plt:
					MultiFab::Copy(*r[i],analysisGroup_var[i],simflowny_vars::Z_z,varCounter,1,withGhost?NUM_GHOST_CELLS:0);
					varCounter++;
					break;

                default:
                    amrex::Error("Output configuration error. Variable " + varnames[j] + " does not exist.");
            }
        }
    }

    return r;
}


// write plotfile to disk
void
AmrCoreProblem::WritePlotFile (int outputCycle, Real time)
{
    BL_PROFILE("AmrCoreProblem::WritePlotFile()");
    const std::string& plotfilename = PlotFileName(outputCycle);
    std::string fullpathname = plot_dir + "/" + plotfilename;
    if (first_plot){
        first_plot = false;
        if (plot_overwrite) {
            if (ParallelContext::IOProcessorSub()) {
                FileSystem::RemoveAll(plot_dir);
            }
            ParallelDescriptor::Barrier();
        }
        PreBuildDirectorHierarchy(plot_dir, "", 0, false);
        ParallelDescriptor::Barrier();
    }
    const auto& mf = PlotFileMF(plot_varnames);

    amrex::Print() << "Writing plotfile " << fullpathname << "\n";

    simflowny::WriteMultiLevelPlotfileHDF5 (fullpathname, finest_level+1, GetVecOfConstPtrs(mf), plot_varnames,
                                        Geom(), time, istep, refRatio());
}

// write plotfile to disk
void
AmrCoreProblem::WritePlotIntegralFile (int index, Real time)
{
    BL_PROFILE("AmrCoreProblem::WritePlotIntegralFile()");
    if (first_plot_integral[index]) {
        first_plot_integral[index] = false;
        if (plot_overwrite) {
            if (ParallelContext::IOProcessorSub()) {
                FileSystem::RemoveAll(plot_integral_dir[index]);
            }
            ParallelDescriptor::Barrier();
        }
        PreBuildDirectorHierarchy(plot_integral_dir[index], "", 0, false);
        ParallelDescriptor::Barrier();
    }

    const auto& mf = PlotFileMF(plot_integral_varnames[index]);

    amrex::Print() << "Writing integral plotfile " << plot_integral_dir[index] << "\n";

    simflowny::WriteMultiLevelIntegralPlot(plot_integral_dir[index], finest_level+1, GetVecOfConstPtrs(mf), plot_integral_varnames[index],
                                   Geom(), time, refRatio(), plot_integral_calculations[index]);

}

// write plotfile to disk
void
AmrCoreProblem::WritePlotPointFile (int index, Real time)
{
    BL_PROFILE("AmrCoreProblem::WritePlotPointFile()");
    if (first_plot_point[index]) {
        first_plot_point[index] = false;
        if (plot_overwrite) {
            if (ParallelContext::IOProcessorSub()) {
                FileSystem::RemoveAll(plot_point_dir[index]);
            }
            ParallelDescriptor::Barrier();
        }
        PreBuildDirectorHierarchy(plot_point_dir[index], "", 0, false);
        ParallelDescriptor::Barrier();
    }

    const auto& mf = PlotFileMF(plot_point_varnames[index]);

    amrex::Print() << "Writing point plotfile " << plot_point_dir[index] << "\n";

    simflowny::WriteMultiLevelPointPlot(plot_point_dir[index], finest_level+1, GetVecOfConstPtrs(mf), plot_point_varnames[index],
                                   Geom(), time, plot_point_coordinates[index]);

}

// write slice plotfile to disk
void AmrCoreProblem::WritePlotSliceFile (int outputCycle, int index, Real time) {
	BL_PROFILE("AmrCoreProblem::WritePlotSliceFile()");
	const std::string& plotfilename = PlotFileName(outputCycle);
	std::string fullpathname = plot_slice_dir[index] + "/" + plotfilename;
	if (first_plot_slice[index]) {
		first_plot_slice[index] = false;
		if (plot_overwrite) {
			if (ParallelContext::IOProcessorSub()) {
				FileSystem::RemoveAll(plot_slice_dir[index]);
			}
			ParallelDescriptor::Barrier();
		}
		PreBuildDirectorHierarchy(plot_slice_dir[index], "", 0, false);
		ParallelDescriptor::Barrier();
	}
	const auto& mf = PlotFileMF(plot_slice_varnames[index]);

	amrex::Print() << "Writing slice plotfile " << fullpathname << "\n";

	simflowny::WriteMultiLevelSlicePlot(fullpathname, finest_level+1, GetVecOfConstPtrs(mf), plot_slice_varnames[index], Geom(), time, istep, refRatio(), plot_slice_plane_normal_axis[index], plot_slice_distance_to_origin[index]);
}

// write sphere to disk
void AmrCoreProblem::WritePlotSphereFile (int outputCycle, int index, Real time) {
	BL_PROFILE("AmrCoreProblem::WritePlotSphereFile()");
	const std::string& plotfilename = PlotFileName(outputCycle);
	std::string fullpathname = plot_sphere_dir[index] + "/" + plotfilename;
	if (first_plot_sphere[index]) {
		first_plot_sphere[index] = false;
		if (plot_overwrite) {
			if (ParallelContext::IOProcessorSub()) {
				FileSystem::RemoveAll(plot_sphere_dir[index]);
			}
			ParallelDescriptor::Barrier();
		}
		PreBuildDirectorHierarchy(plot_sphere_dir[index], "", 0, false);
		ParallelDescriptor::Barrier();
	}
	const auto& mf = PlotFileMF(plot_sphere_varnames[index], true);

	amrex::Print() << "Writing sphere plotfile " << fullpathname << "\n";

	simflowny::WriteMultiLevelSpherePlot(fullpathname, finest_level+1, GetVecOfConstPtrs(mf), plot_sphere_varnames[index], Geom(), time, istep, refRatio(), plot_sphere_radius[index], plot_sphere_center[index], plot_sphere_resolution[index]);
}

/*void 
AmrCoreProblem::calculateMask(int fineLevel, int coarseLevel)
{
    const MultiFab* mask_f_mf = &hierarchy_mask[fineLevel];
    MultiFab* mask_c_mf = &hierarchy_mask[coarseLevel];

    BoxArray fine_ba = mask_f_mf->boxArray();
    BoxArray coarse_ba = mask_c_mf->boxArray();
    const IntVect& rr = refRatio(coarseLevel);

    //Iterate over all coarsen boxes
    for(int b = 0; b < coarse_ba.size(); b++) {
        Box box = coarse_ba[b];
        //Convert fine box to coarser indices
        Box fbx = amrex::refine(box,rr);

        //Intersect fine box with fine box array
        std::vector< std::pair< int, Box > > intersections = fine_ba.intersections(fbx);
        //For all intersections set mask to 0
        for(long unsigned i=0; i < intersections.size(); i++) { 
            Box intersectBox = intersections[i].second;
            //Convert intersection to coarse box indices and set values of mask
            mask_c_mf->setVal(0, amrex::coarsen(intersectBox, rr), 0);
        }
    }
}*/

void
AmrCoreProblem::WriteCheckpointFile () const
{

    // chk00010            write a checkpoint file with this root directory
    // chk00010/Header     this contains information you need to save (e.g., finest_level, t_new, etc.) and also
    //                     the BoxArrays at each level
    // chk00010/Level_0/
    // chk00010/Level_1/
    // etc.                these subdirectories will hold the MultiFab data at each level of refinement

    // checkpoint file name, e.g., chk00010
    const std::string& checkpointname = amrex::Concatenate(chk_file,istep[0]);

    amrex::Print() << "Writing checkpoint " << checkpointname << "\n";

    const int nlevels = finest_level+1;

    // ---- prebuild a hierarchy of directories
    // ---- dirName is built first.  if dirName exists, it is renamed.  then build
    // ---- dirName/subDirPrefix_0 .. dirName/subDirPrefix_nlevels-1
    // ---- if callBarrier is true, call ParallelDescriptor::Barrier()
    // ---- after all directories are built
    // ---- ParallelDescriptor::IOProcessor() creates the directories
    amrex::PreBuildDirectorHierarchy(checkpointname, "Level_", nlevels, true);

    // write Header file
   if (ParallelDescriptor::IOProcessor()) {

       std::string HeaderFileName(checkpointname + "/Header");
       VisMF::IO_Buffer io_buffer(VisMF::IO_Buffer_Size);
       std::ofstream HeaderFile;
       HeaderFile.rdbuf()->pubsetbuf(io_buffer.dataPtr(), io_buffer.size());
       HeaderFile.open(HeaderFileName.c_str(), std::ofstream::out   |
                                               std::ofstream::trunc |
                                               std::ofstream::binary);
       if( ! HeaderFile.good()) {
           amrex::FileOpenFailed(HeaderFileName);
       }

       HeaderFile.precision(17);

       // write out title line
       HeaderFile << "Checkpoint file for AmrCoreProblem\n";

       // write out finest_level
       HeaderFile << finest_level << "\n";

       // write out array of istep
       for (int i = 0; i < istep.size(); ++i) {
           HeaderFile << istep[i] << " ";
       }
       HeaderFile << "\n";

       // write out array of dt
       for (int i = 0; i < dt.size(); ++i) {
           HeaderFile << dt[i] << " ";
       }
       HeaderFile << "\n";

       // write out array of t_new
       for (int i = 0; i < t_new.size(); ++i) {
           HeaderFile << t_new[i] << " ";
       }
       HeaderFile << "\n";

        //Write output variables
        HeaderFile << next_mesh_dump_iteration << "\n";
        for (int i = 0; i < next_integration_dump_iteration.size(); ++i) {
            HeaderFile << next_integration_dump_iteration[i] << " ";
        }
        HeaderFile << "\n";
        for (int i = 0; i < next_point_dump_iteration.size(); ++i) {
            HeaderFile << next_point_dump_iteration[i] << " ";
        }
        HeaderFile << "\n";
		for (int i = 0; i < next_slice_dump_iteration.size(); ++i) {
			HeaderFile << next_slice_dump_iteration[i] << " ";
		}
		HeaderFile << "\n";
		for (int i = 0; i < next_sphere_dump_iteration.size(); ++i) {
			HeaderFile << next_sphere_dump_iteration[i] << " ";
		}
		HeaderFile << "\n";

        for (int i = 0; i < current_iteration.size(); ++i) {
            HeaderFile << current_iteration[i] << " ";
        }
        HeaderFile << "\n";

       // write the BoxArray at each level
       for (int lev = 0; lev <= finest_level; ++lev) {
           boxArray(lev).writeOn(HeaderFile);
           HeaderFile << '\n';
       }
   }

   // write the MultiFab data to, e.g., chk00010/Level_0/
   for (int lev = 0; lev <= finest_level; ++lev) {
        VisMF::Write(regions_var[lev], amrex::MultiFabFileFullPrefix(lev, checkpointname, "Level_", "regions"));
		VisMF::Write(unp1state_var[lev], amrex::MultiFabFileFullPrefix(lev, checkpointname, "Level_", "state"));
		VisMF::Write(unSyncAuxFieldGroup_var[lev], amrex::MultiFabFileFullPrefix(lev, checkpointname, "Level_", "unSyncAuxState"));

   }

}

namespace {
// utility to skip to next line in Header
void GotoNextLine (std::istream& is)
{
    constexpr std::streamsize bl_ignore_max { 100000 };
    is.ignore(bl_ignore_max, '\n');
}
}

void
AmrCoreProblem::ReadCheckpointFile ()
{

    amrex::Print() << "Restart from checkpoint " << restart_chkfile << "\n";

    // Header
    std::string File(restart_chkfile + "/Header");

    VisMF::IO_Buffer io_buffer(VisMF::GetIOBufferSize());

    Vector<char> fileCharPtr;
    ParallelDescriptor::ReadAndBcastFile(File, fileCharPtr);
    std::string fileCharPtrString(fileCharPtr.dataPtr());
    std::istringstream is(fileCharPtrString, std::istringstream::in);

    std::string line, word;

    // read in title line
    std::getline(is, line);

    // read in finest_level
    is >> finest_level;
    GotoNextLine(is);

    // read in array of istep
    std::getline(is, line);
    {
        std::istringstream lis(line);
        int i = 0;
        while (lis >> word) {
            istep[i++] = std::stoi(word);
        }
    }

    // read in array of dt
    std::getline(is, line);
    {
        std::istringstream lis(line);
        int i = 0;
        while (lis >> word) {
            dt[i++] = std::stod(word);
        }
    }

    // read in array of t_new
    std::getline(is, line);
    {
        std::istringstream lis(line);
        int i = 0;
        while (lis >> word) {
            t_new[i++] = std::stod(word);
        }
    }

    //Read output variables
    is >> next_mesh_dump_iteration;
    GotoNextLine(is);
    std::getline(is, line);
    {
        std::istringstream lis(line);
        while (lis >> word) {
            next_integration_dump_iteration.push_back(std::stod(word));
        }
    }
    std::getline(is, line);
    {
        std::istringstream lis(line);
        while (lis >> word) {
            next_point_dump_iteration.push_back(std::stod(word));
        }
    }
	std::getline(is, line);
	{
		std::istringstream lis(line);
		while (lis >> word) {
			next_slice_dump_iteration.push_back(std::stod(word));
		}
	}
	std::getline(is, line);
	{
		std::istringstream lis(line);
		while (lis >> word) {
			next_sphere_dump_iteration.push_back(std::stod(word));
		}
	}
    
    std::getline(is, line);
    {
        std::istringstream lis(line);
        while (lis >> word) {
            current_iteration.push_back(std::stod(word));
        }
    }

    for (int lev = 0; lev <= finest_level; ++lev) {

        // read in level 'lev' BoxArray from Header
        BoxArray ba;
        ba.readFrom(is);
        GotoNextLine(is);

        // create a distribution mapping
        DistributionMapping dm { ba, ParallelDescriptor::NProcs() };

        // set BoxArray grids and DistributionMapping dmap in AMReX_AmrMesh.H class
        SetBoxArray(lev, ba);
        SetDistributionMap(lev, dm);

        // build MultiFab
        regions_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumRegions, NUM_GHOST_CELLS);
	unstate_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk2state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	rk3state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unp1state_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumFields, NUM_GHOST_CELLS);
	unSyncAuxFieldGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumUnSyncAuxFields, NUM_GHOST_CELLS);
	analysisGroup_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, NumAnalysisVars, NUM_GHOST_CELLS);


        regrid_tag_var[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, NUM_GHOST_CELLS);

        /*hierarchy_mask[lev].define(amrex::convert(ba,IntVect::TheNodeVector()), dm, 1, 0);
        hierarchy_mask[lev].setVal(1);

        //unmask coarser level cells overlaping this new box array.
        if (lev > 0) {
            calculateMask(lev, lev-1);
        }*/
    }

    // read in the MultiFab data
    for (int lev = 0; lev <= finest_level; ++lev) {
        VisMF::Read(regions_var[lev], amrex::MultiFabFileFullPrefix(lev, restart_chkfile, "Level_", "regions"));
		VisMF::Read(unp1state_var[lev], amrex::MultiFabFileFullPrefix(lev, restart_chkfile, "Level_", "state"));
		VisMF::Read(unSyncAuxFieldGroup_var[lev], amrex::MultiFabFileFullPrefix(lev, restart_chkfile, "Level_", "unSyncAuxState"));

    }

}

void AmrCoreProblem::ResetFields()
{
	// first loop over all points (valid + ghosts) on all levels
    for (int lev = 0; lev <= finest_level; ++lev) {
        amrex::MultiFab& regions_mf = regions_var[lev];
        amrex::MultiFab& unp1state_mf = unp1state_var[lev];
        amrex::MultiFab& unSyncAuxFieldGroup_mf = unSyncAuxFieldGroup_var[lev];

        const auto& regions_arrays = regions_mf.arrays();
        const auto& unp1state_arrays = unp1state_mf.arrays();
        const auto& unSyncAuxFieldGroup_arrays = unSyncAuxFieldGroup_mf.arrays();

        amrex::ParallelFor(unp1state_mf, amrex::IntVect(NUM_GHOST_CELLS),
            [=] AMREX_GPU_DEVICE(int box_no, int i, int j, int k)
            {
                // only need to add fields below which will be modified

                const auto& regions_array = regions_arrays[box_no];
                // regions_array(i, j, k, region_index) = regions_array(i, j, k, region_index);
                regions_array(i, j, k, FOV_1) = regions_array(i, j, k, FOV_1);

                const auto& unp1state_array = unp1state_arrays[box_no];
                // unp1state_array(i, j, k, field_index) = unp1state_array(i, j, k, field_index);
                // unp1state_array(i, j, k, tauf) = unp1state_array(i, j, k, tauf);

                const auto& unSyncAuxFieldGroup_array = unSyncAuxFieldGroup_arrays[box_no];
                // unSyncAuxFieldGroup_array(i, j, k, unsync_aux_field_index) = unSyncAuxFieldGroup_array(i, j, k, unsync_aux_field_index);
                // unSyncAuxFieldGroup_array(i, j, k, vfd_y) = unSyncAuxFieldGroup_array(i, j, k, vfd_y);
            });

		}
    amrex::Gpu::streamSynchronize();

    // second loop over valid points on all levels (derivatives can be calculated)
    for (int lev = 0; lev <= finest_level; ++lev) {
        amrex::MultiFab& regions_mf = regions_var[lev];
        amrex::MultiFab& unp1state_mf = unp1state_var[lev];
        amrex::MultiFab& unSyncAuxFieldGroup_mf = unSyncAuxFieldGroup_var[lev];

        // Fill intralevel and interlevel ghosts first for calculating derivatives
        FillPatch(lev, t_new[lev], regions_mf, regions_var, NumRegions, std::set<int>());
        FillPatch(lev, t_new[lev], unp1state_mf, unp1state_var, NumFields, state_interpolator_change);
        FillPatch(lev, t_new[lev], unSyncAuxFieldGroup_mf, unSyncAuxFieldGroup_var, NumUnSyncAuxFields, auxFieldGroup_interpolator_change);

        const auto& regions_arrays = regions_mf.arrays();
        const auto& unp1state_arrays = unp1state_mf.arrays();
        const auto& unSyncAuxFieldGroup_arrays = unSyncAuxFieldGroup_mf.arrays();


        amrex::ParallelFor(unp1state_mf,
            [=] AMREX_GPU_DEVICE(int box_no, int i, int j, int k)
            {
                // calculate derivatives here as required

                // only need to add fields below which will be modified

                const auto& unp1state_array = unp1state_arrays[box_no];
                // unp1state_array(i, j, k, field_index) = unp1state_array(i, j, k, field_index);
                // unp1state_array(i, j, k, tauf) = unp1state_array(i, j, k, tauf);

                const auto& unSyncAuxFieldGroup_array = unSyncAuxFieldGroup_arrays[box_no];
                // unSyncAuxFieldGroup_array(i, j, k, unsync_aux_field_index) = unSyncAuxFieldGroup_array(i, j, k, unsync_aux_field_index);
                // unSyncAuxFieldGroup_array(i, j, k, vfd_y) = unSyncAuxFieldGroup_array(i, j, k, vfd_y);
            });
        amrex::Gpu::streamSynchronize();
        
        // Fill intralevel ghosts again after modifying valid points
        bool sync_single_level = true;
        FillPatch(lev, t_new[lev], regions_mf, regions_var, NumRegions, std::set<int>(), sync_single_level);
        FillPatch(lev, t_new[lev], unp1state_mf, unp1state_var, NumFields, state_interpolator_change, sync_single_level);
        FillPatch(lev, t_new[lev], unSyncAuxFieldGroup_mf, unSyncAuxFieldGroup_var, NumUnSyncAuxFields, auxFieldGroup_interpolator_change, sync_single_level);

        postInitialization(lev, t_new[lev], parameters);
    }
}
